#include "QWsSocket.h"

int QWsSocket::maxBytesPerFrame = 1400;

QWsSocket::QWsSocket( QTcpSocket * socket, QObject * parent, quint8 protVers ) :
	QAbstractSocket( QAbstractSocket::UnknownSocketType, parent )
{
	tcpSocket = socket;
	protocolVersion = protVers;

	//setSocketState( QAbstractSocket::UnconnectedState );
	setSocketState( socket->state() );

	connect( tcpSocket, SIGNAL(readyRead()), this, SLOT(dataReceived()) );
	connect( tcpSocket, SIGNAL(disconnected()), this, SLOT(tcpSocketDisconnected()) );
	connect( tcpSocket, SIGNAL(aboutToClose()), this, SLOT(tcpSocketAboutToClose()) );
}

QWsSocket::~QWsSocket()
{
}

void QWsSocket::dataReceived()
{
	if ( protocolVersion == 0 )
	{
		dataReceivedV0();
		return;
	}

	QByteArray BA; // ReadBuffer
	quint8 byte; // currentByteBuffer

	// FIN, RSV1-3, Opcode
	BA = tcpSocket->read(1);
	byte = BA[0];
	quint8 FIN = (byte >> 7);
	quint8 RSV1 = ((byte & 0x7F) >> 6);
	quint8 RSV2 = ((byte & 0x3F) >> 5);
	quint8 RSV3 = ((byte & 0x1F) >> 4);
	EOpcode Opcode = (EOpcode)(byte & 0x0F);

	// Mask, PayloadLength
	BA = tcpSocket->read(1);
	byte = BA[0];
	quint8 Mask = (byte >> 7);
	quint64 PayloadLength = (byte & 0x7F);
	// Extended PayloadLength
	if ( PayloadLength == 126 )
	{
		BA = tcpSocket->read(2);
		PayloadLength = ((quint8)BA[0] << 8) + (quint8)BA[1];
	}
	else if ( PayloadLength == 127 )
	{
		BA = tcpSocket->read(8);
		PayloadLength = 0;
		quint64 plbyte;
		for ( int i=0 ; i<8 ; i++ )
		{
			plbyte = (quint8)BA[i];
			PayloadLength += ( plbyte << (7-i)*8 );
		}
	}

	// MaskingKey
	QByteArray MaskingKey;
	if ( Mask )
	{
		MaskingKey = tcpSocket->read(4);
	}

	// Extension // UNSUPPORTED FOR NOW

	// ApplicationData
	if ( PayloadLength )
	{
		QByteArray ApplicationData = tcpSocket->read( PayloadLength );
		if ( Mask )
			ApplicationData = QWsSocket::mask( ApplicationData, MaskingKey );
		currentFrame.append( ApplicationData );
	}

	if ( FIN )
	{
		if ( Opcode == OpBinary )
		{
			emit frameReceived( currentFrame );
		}
		else if ( Opcode == OpText )
		{
			QString byteString;
			byteString.reserve(currentFrame.size());
			for (int i=0 ; i<currentFrame.size() ; i++)
				byteString[i] = currentFrame[i];
			emit frameReceived( byteString );
		}
		else if ( Opcode == OpPing )
		{
			QByteArray pongRequest = QWsSocket::composeHeader( true, OpPong, 0 );
			write( pongRequest );
		}
		else if ( Opcode == OpPong )
		{
			quint64 ms = pingTimer.elapsed();
			emit pong(ms);
		}
		else if ( Opcode == OpClose )
		{
			tcpSocket->close();
		}
		currentFrame.clear();
	}

	if ( tcpSocket->bytesAvailable() )
		dataReceived();
}

void QWsSocket::dataReceivedV0()
{
	QByteArray BA, rawData;
	quint8 type, b = 0x00;

	BA = tcpSocket->read(1);
	type = BA[0];

	if ( ( type & 0x80 ) == 0x00 ) // MSB of type not set
	{
		if ( type != 0x00 )
		{
			// ABORT CONNEXION
			tcpSocket->readAll();
			return;
		}
		
		// read data
		do
		{
			BA = tcpSocket->read(1);
			b = BA[0];
			if ( b != 0xFF )
				rawData.append( b );
		} while ( b != 0xFF );

		currentFrame.append( rawData );
	}
	else // MSB of type set
	{
		if ( type != 0xFF )
		{
			// ABORT CONNEXION
			tcpSocket->readAll();
			return;
		}

		quint8 length = 0x00;
		
		bool bIsNotZero = true;
		do
		{
			BA = tcpSocket->read(1);
			b = BA[0];
			bIsNotZero = ( b != 0x00 ? true : false );
			if ( bIsNotZero ) // b must be != 0
			{
				quint8 b_v = b & 0x7F;
				length *= 128;
				length += b_v;
			}
		} while ( ( ( b & 0x80 ) == 0x80 ) && bIsNotZero );

		BA = tcpSocket->read(length); // discard this bytes
	}

	if ( currentFrame.size() > 0 )
	{
		QString byteString;
		byteString.reserve( currentFrame.size() );
		for (int i=0 ; i<currentFrame.size() ; i++)
			byteString[i] = currentFrame[i];
		emit frameReceived( byteString );
		currentFrame.clear();
	}

	if ( tcpSocket->bytesAvailable() )
		dataReceived();
}

qint64 QWsSocket::write ( const QString & string, int maxFrameBytes )
{
	if ( protocolVersion == 0 )
	{
		return QWsSocket::write( string.toAscii(), maxFrameBytes );
	}

	if ( maxFrameBytes == 0 )
		maxFrameBytes = maxBytesPerFrame;

	QList<QByteArray> framesList = QWsSocket::composeFrames( string.toUtf8(), false, maxFrameBytes );
	return writeFrames( framesList );
}

qint64 QWsSocket::write ( const QByteArray & byteArray, int maxFrameBytes )
{
	if ( protocolVersion == 0 )
	{
		QByteArray BA;
		BA.append( (char)0x00 );
		BA.append( byteArray );
		BA.append( (char)0xFF );
		return writeFrame( BA );
	}

	if ( maxFrameBytes == 0 )
		maxFrameBytes = maxBytesPerFrame;

	QList<QByteArray> framesList = QWsSocket::composeFrames( byteArray, true, maxFrameBytes );
	return writeFrames( framesList );
}

qint64 QWsSocket::writeFrame ( const QByteArray & byteArray )
{
	return tcpSocket->write( byteArray );
}

qint64 QWsSocket::writeFrames ( QList<QByteArray> framesList )
{
	qint64 nbBytesWritten = 0;
	for ( int i=0 ; i<framesList.size() ; i++ )
	{
		nbBytesWritten += writeFrame( framesList[i] );
	}
	return nbBytesWritten;
}

void QWsSocket::close( QString reason )
{
	if ( protocolVersion == 0 )
	{
		QByteArray BA;
		BA.append( (char)0xFF );
		BA.append( (char)0x00 );
		tcpSocket->write(BA);
		tcpSocket->close();
		return;
	}

	// Compose and send close frame
	quint64 messageSize = reason.size();
	QByteArray maskingKey = generateMaskingKey();
	QByteArray BA;
	quint8 byte;

	QByteArray header = QWsSocket::composeHeader( true, OpClose, 0 );
	BA.append( header );

	// Reason // UNSUPPORTED FOR NOW
	
	tcpSocket->write( BA );

	tcpSocket->close();
}

void QWsSocket::tcpSocketAboutToClose()
{
	emit aboutToClose();
}

void QWsSocket::tcpSocketDisconnected()
{
	emit disconnected();
}

QByteArray QWsSocket::generateMaskingKey()
{
	QByteArray key;
	for ( int i=0 ; i<4 ; i++ )
	{
		key.append( qrand() % 0x100 );
	}

	return key;
}

QByteArray QWsSocket::mask( QByteArray data, QByteArray maskingKey )
{
	for ( int i=0 ; i<data.size() ; i++ )
	{
		data[i] = ( data[i] ^ maskingKey[ i % 4 ] );
	}

	return data;
}

QList<QByteArray> QWsSocket::composeFrames( QByteArray byteArray, bool asBinary, int maxFrameBytes )
{
	if ( maxFrameBytes == 0 )
		maxFrameBytes = maxBytesPerFrame;

	QList<QByteArray> framesList;

    QByteArray maskingKey;// = generateMaskingKey();

	int nbFrames = byteArray.size() / maxFrameBytes + 1;

	for ( int i=0 ; i<nbFrames ; i++ )
	{
		QByteArray BA;

		// fin, size
		bool fin = false;
		quint64 size = maxFrameBytes;
		EOpcode opcode = OpContinue;
		if ( i == nbFrames-1 ) // for multi-frames
		{
			fin = true;
			size = byteArray.size();
		}
		if ( i == 0 )
		{
			if ( asBinary )
				opcode = OpBinary;
			else
				opcode = OpText;
		}
		
		// Header
		QByteArray header = QWsSocket::composeHeader( fin, opcode, size, maskingKey );
		BA.append( header );
		
		// Application Data
		QByteArray dataForThisFrame = byteArray.left( size );
		byteArray.remove( 0, size );
		
		dataForThisFrame = QWsSocket::mask( dataForThisFrame, maskingKey );
		BA.append( dataForThisFrame );
		
		framesList << BA;
	}

	return framesList;
}

QByteArray QWsSocket::composeHeader( bool fin, EOpcode opcode, quint64 payloadLength, QByteArray maskingKey )
{
	QByteArray BA;
	quint8 byte;

	// FIN, RSV1-3, Opcode
	byte = 0x00;
	// FIN
	if ( fin )
		byte = (byte | 0x80);
	// Opcode
	byte = (byte | opcode);
	BA.append( byte );

	// Mask, PayloadLength
	byte = 0x00;
	QByteArray BAsize;
	// Mask
	if ( maskingKey.size() == 4 )
		byte = (byte | 0x80);
	// PayloadLength
	if ( payloadLength <= 125 )
	{
		byte = (byte | payloadLength);
	}
	// Extended payloadLength
	else
	{
		// 2 bytes
		if ( payloadLength <= 0xFFFF )
		{
			byte = ( byte | 126 );
			BAsize.append( ( payloadLength >> 1*8 ) & 0xFF );
			BAsize.append( ( payloadLength >> 0*8 ) & 0xFF );
		}
		// 8 bytes
		else if ( payloadLength <= 0x7FFFFFFF )
		{
			byte = ( byte | 127 );
			BAsize.append( ( payloadLength >> 7*8 ) & 0xFF );
			BAsize.append( ( payloadLength >> 6*8 ) & 0xFF );
			BAsize.append( ( payloadLength >> 5*8 ) & 0xFF );
			BAsize.append( ( payloadLength >> 4*8 ) & 0xFF );
			BAsize.append( ( payloadLength >> 3*8 ) & 0xFF );
			BAsize.append( ( payloadLength >> 2*8 ) & 0xFF );
			BAsize.append( ( payloadLength >> 1*8 ) & 0xFF );
			BAsize.append( ( payloadLength >> 0*8 ) & 0xFF );
		}
	}
	BA.append( byte );
	BA.append( BAsize );

	// Masking
	if ( maskingKey.size() == 4 )
		BA.append( maskingKey );

	return BA;
}

void QWsSocket::ping()
{
	pingTimer.restart();
	QByteArray pingFrame = QWsSocket::composeHeader( true, OpPing, 0 );
	writeFrame( pingFrame );
}

QString QWsSocket::composeOpeningHandShake( QString ressourceName, QString host, QString origin, QString extensions, QString key )
{
	QString hs;
	hs.append("GET /ws HTTP/1.1\r\n");
	hs.append("Host: pmx\r\n");
	hs.append("Upgrade: websocket\r\n");
	hs.append("Connection: Upgrade\r\n");
	hs.append("Sec-WebSocket-Version: 6\r\n");
	hs.append("Sec-WebSocket-Origin: http://pmx\r\n");
	hs.append("Sec-WebSocket-Extensions: deflate-stream\r\n");
	hs.append("Sec-WebSocket-Key: x3JJHMbDL1EzLkh9GBhXDw==\r\n");
	hs.append("\r\n");
	return hs;
}
