﻿using System;
using System.Threading;
using System.Windows;
using System.Windows.Media;
using System.Collections.Generic;

namespace Wally
{
	public class Wally
	{
		IGame game;
		Action<GameState> action;

		public Wally(IGame game)
		{
			this.game = game;
			action = attackFromCenter;
		}

		public void Run()
		{
			game.SetName("Wally Windbreaker");

			while (game.Connected())
			{
				try
				{
					GameState state = game.GetState();
					if (state == null) break;
					action(state);
				}
				catch { }
			}
		}

		private void attackFromCenter(GameState state)
		{
			Cloud me = state.Me;

			var clouds = new List<Cloud>(state.AllClouds);
			clouds.Sort((c1, c2) => (c1.Position - me.Position).Length.CompareTo((c2.Position - me.Position).Length));

			var powers = new List<double>();
			for (double p = .3; p <= .7; p += .02) powers.Add(p);
			var attack = getAttackWind(me, powers.ToArray(), clouds, null, false);
			if (attack.wind.Length > 0)
			{
				game.Wind(attack.wind);
				action = inFlight;
			}
			Thread.Sleep(100);
		}

		private void inFlight(GameState state)
		{
			Cloud me = state.Me;
			var clouds = new List<Cloud>(state.AllClouds);
			clouds.Sort((c1, c2) => (c1.Position - me.Position).Length.CompareTo((c2.Position - me.Position).Length));

			if (clouds.Count == 0)
			{
				Thread.Sleep(100);
				action = celebrate;
				return;
			}

			if (evaluateTrajectory(me, clouds, .3) > 1.2 * me.Vapor)
			{
				Thread.Sleep(100);
				return;
			}

			double collisionTime = double.MaxValue;
			Collision collisionInfo = null;
			foreach (Cloud cloud in state.AllClouds)
			{
				if (cloud.Vapor >= me.Vapor)
				{
					var info = Collision.getCollision(me, cloud);
					if (info != null && info.time1 > 0 && info.time1 < collisionTime)
					{
						collisionTime = info.time1;
						collisionInfo = info;
					}
				}
			}
			if (collisionInfo != null && collisionInfo.time1 < .5)
			{
				adjustVelocity(me, me.Velocity + collisionInfo.escape / collisionInfo.time1 / 10);
				Thread.Sleep(100);
				return;

			}


			double dLeft = me.Position.X - me.Radius;
			double dRight = 1280 - me.Position.X - me.Radius;
			double dHorizontal;
			if (me.Velocity.X > 1)
				dHorizontal = dRight;
			else if (me.Velocity.X < -1)
				dHorizontal = dLeft;
			else
				dHorizontal = Math.Min(dLeft, dRight);

			double dTop = me.Position.Y - me.Radius;
			double dBottom = 720 - me.Position.Y - me.Radius;
			double dVertical;
			if (me.Velocity.Y > 1)
				dVertical = dBottom;
			else if (me.Velocity.Y < -1)
				dVertical = dTop;
			else
				dVertical = Math.Min(dTop, dBottom);

			double wallAttackMargin = 4;
			double wallJumpStrength = .8;
			var powers = new List<double>();
			for (double p = .5; p <= .86; p += .02) powers.Add(p);
			var attackPowers = powers.ToArray();
			if (dHorizontal < wallAttackMargin)
			{
				if (dVertical < wallAttackMargin)
				{
					var attack = getAttackWind(me, attackPowers, clouds);
					if (attack.wind.Length > 0)
					{
						game.Wind(attack.wind);
						Thread.Sleep(100);
						return;
					}
				}
				else if (dVertical > 64 && dVertical / Math.Abs(10 * me.Velocity.Y) > .72)
				{
					var attack = getAttackWind(me, attackPowers, clouds, w => Math.Acos(Math.Abs(w.X) / w.Length) > Math.PI / 12);
					if (attack.wind.Length > 0)
					{
						game.Wind(attack.wind);
						Thread.Sleep(100);
						return;
					}
					else
					{
						int xSign = dHorizontal == dLeft ? 1 : -1;
						int ySign = Math.Sign(me.Velocity.Y);
						var wind = new Vector(xSign * Math.Cos(Math.PI / 12), ySign * Math.Sin(Math.PI / 12)) * wallJumpStrength * me.Vapor / 2;
						if (evaluateWind(wind, me, clouds) > 0)
						{
							game.Wind(wind);
							Thread.Sleep(100);
							return;
						}
						else
						{
							wind = new Vector(xSign * Math.Cos(Math.PI / 12), -ySign * Math.Sin(Math.PI / 12)) * wallJumpStrength * me.Vapor / 2;
							if (evaluateWind(wind, me, clouds) > 0)
							{
								game.Wind(wind);
								Thread.Sleep(100);
								return;
							}
						}
					}
				}
			}
			else
			{
				if (dVertical < wallAttackMargin && dHorizontal > 64 && dHorizontal / Math.Abs(10 * me.Velocity.X) > 1.28)
				{
					var attack = getAttackWind(me, attackPowers, clouds, w => Math.Acos(Math.Abs(w.Y) / w.Length) > Math.PI / 12);
					if (attack.wind.Length > 0)
					{
						game.Wind(attack.wind);
						Thread.Sleep(100);
						return;
					}
					else
					{
						int ySign = dVertical == dTop ? 1 : -1;
						int xSign = Math.Sign(me.Velocity.X);
						var wind = new Vector(xSign * Math.Sin(Math.PI / 12), ySign * Math.Cos(Math.PI / 12)) * wallJumpStrength * me.Vapor / 2;
						if (evaluateWind(wind, me, clouds) > 0)
						{
							game.Wind(wind);
							Thread.Sleep(100);
							return;
						}
						else
						{
							wind = new Vector(-xSign * Math.Sin(Math.PI / 12), ySign * Math.Cos(Math.PI / 12)) * wallJumpStrength * me.Vapor / 2;
							if (evaluateWind(wind, me, clouds) > 0)
							{
								game.Wind(wind);
								Thread.Sleep(100);
								return;
							}
						}
					}
				}
			}

			const double slowMargin = 160;
			double stopMargin = wallAttackMargin / 2;
			if (dHorizontal >= stopMargin && dHorizontal < slowMargin)
				reduceHorizontalSpeed(me, dHorizontal / 4);
			else if (dHorizontal < stopMargin)
				reduceHorizontalSpeed(me, 0);
			if (dVertical >= stopMargin && dVertical < slowMargin)
				reduceVerticalSpeed(me, dVertical / 4);
			else if (dVertical < stopMargin)
				reduceVerticalSpeed(me, 0);

			Thread.Sleep(100);
		}

		private void celebrate(GameState state)
		{
			Cloud me = state.Me;
			for (double angle = 0; me.Vapor > 500; angle += .07)
			{
				double radians = 2 * Math.PI * angle;
				game.Wind(17 * new Vector(Math.Cos(angle), Math.Sin(angle)));
				game.Wind(17 * new Vector(Math.Cos(angle + 2 * Math.PI / 3), Math.Sin(angle + 2 * Math.PI / 3)));
				game.Wind(17 * new Vector(Math.Cos(angle + 4 * Math.PI / 3), Math.Sin(angle + 4 * Math.PI / 3)));
				me.Vapor -= 51;
			}
			action = attackFromCenter;
			Thread.Sleep(100);
			return;
		}

		class AttackWind
		{
			public Vector wind;
			public double value;
		}

		private AttackWind getAttackWind(Cloud me, double[] powers, List<Cloud> clouds, Predicate<Vector> filterWind = null, bool conserveVapor = true)
		{
			AttackWind best = new AttackWind();
			foreach (double power in powers)
			{
				var w = getAttackWind(me, power, clouds, filterWind, conserveVapor);
				if (w.value > best.value)
					best = w;
			}
			return best;
		}

		private AttackWind getAttackWind(Cloud me, double power, List<Cloud> clouds, Predicate<Vector> filterWind = null, bool conserveVapor = true)
		{
			double bestValue = me.Vapor - (conserveVapor ? 0 : power * me.Vapor / 2);
			Vector bestWind = new Vector(0, 0);
			foreach (Cloud cloud in clouds)
			{
				var intercept = new Intercept(cloud, me, me.Vapor / 2 * power);
				if (intercept.time <= 0) continue;
				if (filterWind != null && filterWind(intercept.wind)) continue;
				double value = evaluateWind(intercept.wind, me, clouds, conserveVapor);
				if (value > bestValue)
				{
					bestValue = value;
					bestWind = intercept.wind;
				}
			}
			AttackWind attackWind = new AttackWind();
			attackWind.wind = bestWind;
			attackWind.value = bestValue;
			return attackWind;
		}

		private double evaluateWind(Vector wind, Cloud me, List<Cloud> clouds, bool conserveVapor = true)
		{
			Vector velocity = getVelocityForWind(me, wind);
			Cloud xMe = new Cloud(me.Position.X, me.Position.Y, velocity.X, velocity.Y, me.Vapor);
			if (!conserveVapor) xMe.Vapor -= wind.Length;
			return evaluateTrajectory(xMe, clouds, 1.28);
		}

		private static double evaluateTrajectory(Cloud me, List<Cloud> clouds, double time)
		{
			Cloud xMe = new Cloud(me.Position.X, me.Position.Y, me.Velocity.X, me.Velocity.Y, me.Vapor);
			foreach (Cloud cloud in clouds)
			{
				var collision = Collision.getCollision(xMe, cloud);
				if (collision != null && collision.time1 > 0 && collision.time1 < time)
				{
					xMe.Vapor += collision.vaporChange;
					if (xMe.Vapor < 1)
						return 0;
				}
			}
			return xMe.Vapor;
		}

		class Intercept
		{
			public Cloud target;
			public double time;
			public Vector wind;

			public Intercept(Cloud target, Cloud me, double windStrength)
			{
				this.target = target;
				double a = me.Position.X - target.Position.X;
				double b = me.Position.Y - target.Position.Y;
				double c = me.Velocity.X - target.Velocity.X;
				double d = me.Velocity.Y - target.Velocity.Y;
				double s = windStrength / Math.Sqrt(me.Vapor - windStrength) * 5;
				double angle = 2 * Math.Atan((a * s + Math.Sqrt(a * a * (s * s - d * d) + 2 * a * b * c * d + b * b * (s * s - c * c))) / (-a * d + b * c - b * s));
				time = -a / (s * Math.Cos(angle) - c);
				wind = windStrength * new Vector(Math.Cos(angle), Math.Sin(angle));
			}
		}


		private void reduceSpeed(Cloud me, double speed)
		{
			if (me.Velocity.Length > speed)
				adjustSpeed(me, speed / me.Velocity.Length);
		}
		private void reduceHorizontalSpeed(Cloud me, double speed)
		{
			if (Math.Abs(me.Velocity.X) > speed)
				adjustVelocity(me, new Vector(speed * Math.Sign(me.Velocity.X), me.Velocity.Y));
		}
		private void reduceVerticalSpeed(Cloud me, double speed)
		{
			if (Math.Abs(me.Velocity.Y) > speed)
				adjustVelocity(me, new Vector(me.Velocity.X, speed * Math.Sign(me.Velocity.Y)));
		}

		private void adjustSpeed(Cloud me, double fraction)
		{
			Vector targetVelocity = fraction * me.Velocity;
			targetVelocity = adjustVelocity(me, targetVelocity);
		}

		private Vector adjustVelocity(Cloud me, Vector targetVelocity)
		{
			Vector targetVelocityChange = targetVelocity - me.Velocity;
			Vector wind = new Vector();
			for (int i = 0; i < 10; i++)
			{
				wind = targetVelocityChange * Math.Sqrt(me.Vapor - wind.Length) / 5;
				if (wind.Length > me.Vapor / 2)
				{
					wind *= me.Vapor / 2 / wind.Length;
					break;
				}
			}
			game.Wind(wind);
			return targetVelocity;
		}

		private Vector getVelocityForWind(Cloud me, Vector wind)
		{
			return me.Velocity + 5 * wind / Math.Sqrt(me.Vapor - wind.Length);
		}
	}
}