import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

public class GenOptionsPanel extends ParameterGroup implements ParameterListener {
	private ParameterListener listener;
	private JComponent gui;

	private double color_blending;
	private double panx, pany, zoom;
	private double[] rot_matrix_left;
	private double[] rot_matrix_right;
	private boolean stereo_pair;
	private boolean normalize_coordinates;
	private double sph_offset, sph_radius;
	private int grid_effect;
	private long last_reset;
	private boolean depth_shade_enable;
	private double depth_shade_amount;

	private double[] rotated = new double[3];
	private int[] draw_color = new int[3];

	public GenOptionsPanel(ParameterListener listener) {
		super("GEN_OPTIONS", "Generation Options", null);

		this.listener = listener;
		
		addParameter(new SliderParameter("COLOR_BLEND", "Color Blending", 0, 1, .3));
		addParameter(new SliderParameter("ROT_X", "Rotation X", -180, 180, 0));
		addParameter(new SliderParameter("ROT_Y", "Rotation Y", -180, 180, 0));
		addParameter(new SliderParameter("ROT_Z", "Rotation Z", -180, 180, 0));
		addParameter(new SliderParameter("SCALE", "Scale", .1, 5, 1));
		addParameter(new SliderParameter("PAN_X", "Pan X", -5, 5, 0));
		addParameter(new SliderParameter("PAN_Y", "Pan Y", -5, 5, 0));
		addParameter(new SliderParameter("GRID_EFFECT", "Grid Effect", 0, 10, 0));
		addParameter(new CheckboxParameter("STEREO_PAIR", "Stereo Pair", false));
		addParameter(new SliderParameter("STEREO_ROT", "Stereo Rotation", 0, 5, 2));
		addParameter(new CheckboxParameter("PROJ_SPH", "Project to Sphere", false));
		addParameter(new SliderParameter("SPH_RAD", "Sphere Radius", 0, 2, 1));
		addParameter(new SliderParameter("SPH_OFF", "Sphere Offset", 0, 2, 0));
		addParameter(new CheckboxParameter("DEPTH_SHADE_ENABLE", "Depth Shading Enable", false));
		addParameter(new SliderParameter("DEPTH_SHADE_AMOUNT", "Depth Shade Amount", 0, 5, 0));

		super.addParameterListener(this);

		parameterChanged(this);
	}

	public JComponent createGUIComponent() {
		if(gui != null) return gui;

		JPanel inner_panel = new JPanel();
		inner_panel.setLayout(new BoxLayout(inner_panel, BoxLayout.Y_AXIS));

		inner_panel.add(super.createGUIComponent());

		JScrollPane sp = new JScrollPane(inner_panel);
		JPanel panel = new JPanel();
		panel.setLayout(new BorderLayout());
		panel.add(sp, BorderLayout.CENTER);

		this.gui = panel;
		return gui;
	}

	public void parameterChanged(Object src) {
		color_blending = Math.pow(getDouble("COLOR_BLEND"), 2);
		double rotx = getDouble("ROT_X");
		double roty = getDouble("ROT_Y");
		double rotz = getDouble("ROT_Z");
		zoom = getDouble("SCALE");
		panx = getDouble("PAN_X");
		pany = getDouble("PAN_Y");
		grid_effect = (int)getDouble("GRID_EFFECT");

		stereo_pair = getBoolean("STEREO_PAIR");
		double stereo_sep;
		if(stereo_pair) {
			stereo_sep = getDouble("STEREO_ROT");
		} else {
			stereo_sep = 0;
		}
		normalize_coordinates = getBoolean("PROJ_SPH");
		sph_radius = getDouble("SPH_RAD");
		sph_offset = getDouble("SPH_OFF") * sph_radius;

		double[] rot_matrix_center =
			Matrix3D.matrixMult(Matrix3D.getRotMat(1, 0, 0, rotx*Math.PI/180.0),
			Matrix3D.matrixMult(Matrix3D.getRotMat(0, 1, 0, roty*Math.PI/180.0),
			Matrix3D.getRotMat(0, 0, 1, rotz*Math.PI/180.0)));

		rot_matrix_left = Matrix3D.matrixMult(rot_matrix_center,
			Matrix3D.getRotMat(0, 1, 0, -stereo_sep/2.0*Math.PI/180.0));
		rot_matrix_right = Matrix3D.matrixMult(rot_matrix_center,
			Matrix3D.getRotMat(0, 1, 0,  stereo_sep/2.0*Math.PI/180.0));

		depth_shade_enable = getBoolean("DEPTH_SHADE_ENABLE");
		depth_shade_amount = getDouble("DEPTH_SHADE_AMOUNT");

		if(listener != null) listener.parameterChanged(this);
	}

	public void iterate(
		long iter, boolean[] ret_flags,
		Xform xform, int xfid,
		double[] coord, int[] color,
		int[] buffer32, long[] buffer64, int width, int height,
		boolean enable_stereo, double scale_single, double scale_stereo
	) {
		boolean reset = false;
		boolean enable_plot = true;
		if(grid_effect>0) {
			int mod = ((int)iter)%grid_effect;
			reset = mod==0;
			enable_plot = mod==(grid_effect-1);
		} else {
			for(int i=0; i<coord.length; i++)
				if(coord[i] > 20.0 || coord[i] < -20.0) {
					reset = true;
					ret_flags[1] = true;
				}
			if(reset) last_reset = iter;
			if(last_reset > iter) last_reset = iter;
			// skip first points
			enable_plot = iter - last_reset > 10;
		}

		if(reset) {
			int axis = -1;
			if(grid_effect > 0) axis = 
				(int)Math.floor((double)coord.length*Math.random());
			for(int i=0; i<coord.length; i++) {
				coord[i] = i==axis 
					? Math.random()*2.0 - 1.0
					: Math.floor(2.0*Math.random() + .5) - 1.0;
			}
		}

		xform.iterate(coord);

		for(int i=0; i<3; i++) color[i] *= color_blending;
		switch(xfid) {
			case 0: color[0] += 1000; break;
			case 1: color[1] += 1000; break;
			case 2: color[2] += 1000; break;
			default: 
				color[0] += 1000;
				color[1] += 1000;
				color[2] += 1000;
				break;
		}
		for(int i=0; i<3; i++) color[i] *= .9;

		if(enable_plot) {
			Matrix3D.vecMult(
				rotated,
				rot_matrix_left,
				coord);
			double xf = (rotated[0] + panx) * zoom;
			double yf = (rotated[1] + pany) * zoom;

			if(depth_shade_enable) {
				double exp = 
					rotated[2] > 1 ? 1 :
	1.0/Math.pow(2.0 - rotated[2], depth_shade_amount);
				for(int i=0; i<3; i++) draw_color[i] = 
					(int)(Math.pow(color[i], exp));
			} else {
				for(int i=0; i<3; i++) draw_color[i] = color[i];
			}
	
			if(stereo_pair && enable_stereo) {
				int xi =  (int)Math.floor(xf * scale_stereo) + width/4;
				int yi = -(int)Math.floor(yf * scale_stereo) + height/2;
				if(xi>=0 && xi<width/2 && yi>=0 && yi<height)
					ret_flags[0] |= plotPoint(xi, yi, buffer32, buffer64,
						width, height, draw_color);
	
				Matrix3D.vecMult(
					rotated,
					rot_matrix_right,
					coord);
				xf = (rotated[0] + panx) * zoom;
				yf = (rotated[1] + pany) * zoom;
	
				xi =  (int)Math.floor(xf * scale_stereo) + 3*width/4;
				yi = -(int)Math.floor(yf * scale_stereo) + height/2;
				if(xi>=width/2 && xi<width && yi>=0 && yi<height)
					ret_flags[0] |= plotPoint(xi, yi, buffer32, buffer64,
						width, height, draw_color);
			} else {
				int xi =  (int)Math.floor(xf * scale_single) + width/2;
				int yi = -(int)Math.floor(yf * scale_single) + height/2;
				if(xi>=0 && xi<width && yi>=0 && yi<height)
					ret_flags[0] |= plotPoint(xi, yi, buffer32, buffer64,
						width, height, draw_color);
			}
		}

/*
		if(xfid==0) {
			int axis = (int)Math.floor((double)coord.length*Math.random());
			for(int i=0; i<coord.length; i++) {
				coord[i] = i==axis 
					? Math.random()*2.0 - 1.0
					: Math.floor(2.0*Math.random() + .5) - 1.0;
			}
		}
*/

		if(normalize_coordinates) {
			double len = sph_offset*sph_offset;
			for(int i=0; i<coord.length; i++)
				len += coord[i]*coord[i];
			len = Math.sqrt(len);
			if(len>0.0001)
				for(int i=0; i<coord.length; i++)
					coord[i] *= sph_radius / len;
		}
	}

	private boolean plotPoint(
		int x, int y,
		int[] buffer32, long[] buffer64, int width, int height, 
		int[] color
	) {
		if(buffer64 != null) {
			int off = (x + y*width)*3;
			for(int i=0; i<3; i++)
				buffer64[off + i] += color[i];
			return false;
		} else {
			int off = (x + y*width)*3;
			for(int i=0; i<3; i++) {
				// check for overflow
				if(buffer32[off + i] >= Integer.MAX_VALUE - color[i]) {
					return true;
				}
			}
			for(int i=0; i<3; i++)
				buffer32[off + i] += color[i];
			return false;
		}
	}
}
