module Util where

import Data.Vec 
import Graphics.GPipe
import Graphics.GPipe.Texture.Load
import Codec.Image.STB
import Data.Bitmap.IO (withBitmap)
import Data.Either
import TextGen
import Foreign
import Foreign.C
import Codec.Binary.UTF8.String (encodeString)

data TimePoint = TP { coord::Vec2 Float, millis::Float }
data TextPath = Path { begin::TimePoint, stable::TimePoint, fadeout::TimePoint, end::TimePoint }

data RawScene = LoadMarquee {text'::String, scale'::Float, path'::TextPath}
              | LoadPicture {picTexturefile::String, scale'::Float, picbegins::Float, picends::Float, pos'::(Vec2 Float)}
              | LoadSinBrickwall {sinTexturefile::String, rawbegins::Float, rawends::Float}
              | LoadNick {nick'::String, scale'::Float, pos'::(Vec2 Float), time'::Float}

data Scene = Marquee      {text::String, texture::Maybe (Texture2D RGBAFormat), scale::Float, size::Maybe (Vec2 Int), path::TextPath}
           | Picture      {texture::Maybe (Texture2D RGBAFormat), scale::Float, size::Maybe (Vec2 Int), pos::(Vec2 Float), begins::Float, ends::Float}
           | SinBrickwall {texture::Maybe (Texture2D RGBAFormat), size::Maybe (Vec2 Int), begins::Float, ends::Float}
           | Nick         {nick::String, texture::Maybe (Texture2D RGBAFormat), scale::Float, size::Maybe (Vec2 Int), pos::(Vec2 Float), blinkTime::Float}

pnt :: Float -> Float -> Vec2 Float
pnt x y = x:.y:.()

-- Type delcared loading for satisfying type function TextureFormat
loadTexture' :: String -> IO (Texture2D RGBAFormat)
loadTexture' path = loadTexture RGBA8 path

load (LoadMarquee text scaling path) = do
  pak <- toTexture text "Monospace 66"
  return $ packScene pak --texture dim
    where
      packScene (texture, dim) = Marquee text (Just texture) scaling (Just dim) path
load (LoadSinBrickwall tfile begins ends) = do
  texture <- loadTexture' tfile
  dim <- getTexture2DSize tfile
  return $ packScene texture dim
    where
      packScene texture dim = SinBrickwall (Just texture) (Just dim) begins ends
load (LoadPicture tfile scale begins ends pos) = do
  texture <- loadTexture' tfile
  dim <- getTexture2DSize tfile
  return $ packScene texture dim
    where
      packScene texture dim = Picture (Just texture) scale (Just dim) pos begins ends
load (LoadNick text scaling pos time) = do
  pakitsi <- toTexture text "Monospace 66"
  return $ packScene pakitsi --texture dim
    where
      packScene (texture, dim) = Nick text (Just texture) scaling (Just dim) pos time

stationaryPath :: Float -> Float -> Float -> Float -> TextPath
stationaryPath x y begin end = Path (TP (pnt x y) begin) (TP (pnt x y) begin) (TP (pnt x y) end) (TP (pnt x y) end)

-- Utilities to get image dimensions from a file

texture2DSizeFromImage path (w,h) comp 0 ptr = return (w:.h:.())
texture2DSizeFromImage path _ _ _ _ = ioError $ userError ("loadTexture: Row padding is not supported, in " ++ show path)

getTexture2DSize' io path = do image <- loadImage path
                               either
                                  (ioError . userError)
                                  (flip withBitmap io)
                                  image

-- Returns dimensions of a image file
getTexture2DSize :: FilePath -> IO (Vec2 Int)
getTexture2DSize path = getTexture2DSize' (texture2DSizeFromImage path) path

defMarqueeTime = 300 -- ms

calcBegin :: (Float, Float) -> Float
calcBegin (begin, end) = begin - defMarqueeTime

calcStable :: (Float, Float) -> Float
calcStable (begin, end) = begin

calcFadeout :: (Float, Float) -> Float
calcFadeout (begin, end) = end - defMarqueeTime

calcEnd :: (Float, Float) -> Float
calcEnd (begin, end) = end

generateText rawTxt font =
    withCString txt $ \txt' ->
        withCString font $ \font' -> do
          img <- createTextImage txt' font'
          return img
    where txt = encodeString rawTxt -- Unicode to UTF-8

toTexture :: String -> String -> IO (Texture2D RGBAFormat, Vec2 Int)
toTexture txt font = do
    imagePtr <- generateText txt font
    image <- peek imagePtr 
    let size = ((mkInt (width image)):.(mkInt (height image):.()))
    tex <- newTexture (PerComp4 UnsignedByteFormat) RGBA8 size [idata image]
    return (tex, size)

mkInt :: CInt -> Int
mkInt n = fromIntegral n

mag (x:.y:.()) = x * y
