module Display where

import Graphics.GPipe
import qualified Data.Vec as Vec
import Data.Vec.Nat
import Data.Vec.LinAlg.Transform3D
import Data.Monoid
import Env
import Util

-- Quad is useful for displaying textures

quad :: Float -> Float -> PrimitiveStream Triangle (Vec3 (Vertex Float), Vec2 (Vertex Float)) 
quad width height = toGPUStream TriangleStrip $ zip vertices uvCoords
    where vertices = [(-w):.(-h):.0:.(), (-w):.h:.0:.(), w:.(-h):.0:.(), w:.h:.0:.()]
          uvCoords = [0:.1:.(), 0:.0:.(), 1:.1:.(), 1:.0:.()]
          w = width / 2
          h = height / 2

-- Render pixelshader fullscreen for postprocessing etc.
-- Argument is a FragmentStream taking Vec2 (UV) and returning Vec3 (color)

renderFullscreen :: Vec2 Float
                 -> Vec2 Float
                 -> (FragmentStream (Vec2 (Fragment Float)) 
                 -> FragmentStream (Vec3 (Fragment Float), Fragment Float))
                 -> FragmentStream (Color RGBAFormat (Fragment Float))
renderFullscreen pos (w:.h:.()) effect = outputColor $ effect $ inputUv $ rasterizeFrontAndBack $ putOnScreen pos $ quad w h

-- Orthogonal projection suitable for displaying quads fullscreen
putOnScreen :: (Vec2 Float)
            -> PrimitiveStream Triangle (Vec3 (Vertex Float), Vec2 (Vertex Float))
            -> PrimitiveStream Triangle (Vec4 (Vertex Float), Vec2 (Vertex Float))
putOnScreen (x:.y:.()) = fmap project 
    where project (v, uv) = (orthoMat `multmv` (homPos v), uv)
          homPos v = homPoint v :: Vec4 (Vertex Float)
          orthoMat = toGPU $ orthogonal (-10) 10 (2:.2:.()) `multmm` translateMat
          translateMat = translation (x:.y:.0:.())

-- Exists for future purposes, now only strips front/backfacing polygon tag
inputUv :: FragmentStream (Fragment Bool, Vec2 (Fragment Float))
        -> FragmentStream (Vec2 (Fragment Float))
inputUv coords = fmap (\(_, uv) -> (uv)) coords

-- Converts floating point color values to format suitable for framebuffer
outputColor :: FragmentStream (Vec3 (Fragment Float), Fragment Float)
            -> FragmentStream (Color RGBAFormat (Fragment Float))
outputColor input = fmap colorToRGBA input
    where colorToRGBA (color, alpha) = RGBA color alpha

-- Actual drawing functions
drawTexture :: Texture2D RGBAFormat 
              -> FragmentStream (Vec2 (Fragment Float))
              -> FragmentStream (Vec3 (Fragment Float), Fragment Float)
drawTexture tex input = fmap color input
    where color (uv) = stripRGB $ sample (Sampler Linear Wrap) tex uv

-- sample returns color format, don't know how to handle it so let's make it disappear
stripRGB :: Color RGBAFormat a -> (Vec3 a, a)
stripRGB (RGBA color alpha) = (color, alpha)

-- Initialize framebuffer with color

emptyFB :: Float -> Float -> Float -> Float -> FrameBuffer RGBAFormat () ()
emptyFB r g b a = newFrameBufferColor (RGBA (r:.b:.g:.()) a)

-- Paint transparent surfaces
paintTransparent :: FragmentStream (Color RGBAFormat (Fragment Float))
                   -> FrameBuffer RGBAFormat d s
                   -> FrameBuffer RGBAFormat d s
paintTransparent = paintColor blend (RGBA (True:.True:.True:.()) True)
    where blend = Blend (FuncAdd, FuncAdd) (factor, factor) (RGBA (2:.2:.2:.()) 1)
          factor = (SrcAlpha, OneMinusSrcAlpha)
-- Return our finished framebuffer

-- FIXME: rename: nothing marquee spesific here anymore
calcMarqueeSize :: Vec2 Int -> Float -> Vec2 Float
calcMarqueeSize (w:.h:.()) scaling = (width:.height:.())
  where
    width  = scaling
    height = ((fromIntegral h)/(fromIntegral w)) * scaling

-- Calculates image size, fitting image by its width to given scale
fitToScale :: Vec2 Int -> Float -> Vec2 Float
fitToScale (w:.h:.()) scaling = (width:.height:.())
  where
    ratio  = scaling/(fromIntegral w)
    width  = ratio * (fromIntegral w)
    height = ratio * (fromIntegral h) * 1.77 -- sleeve constant
      -- FIXME: i'm tired. basically, the non square images e.g. 1280x720
      --        should be scaled to square coordinates ((2,2), which represents full screen)


-- FIXME: some copy/paste here
-- Handles currently only Marquee scenes, but could also handle others as well.
drawScene :: FrameBuffer RGBAFormat () () -> Scene -> Time -> FrameBuffer RGBAFormat () ()
drawScene bg (Marquee text (Just tex) scaling (Just dim) path) time = paint (render pos size (drawTexture tex)) bg
  where
    pos = marqueeScenePos time path
    size = calcMarqueeSize dim scaling
    paint = paintTransparent
    render = renderFullscreen
drawScene bg (SinBrickwall (Just tex) (Just dim) begins ends) time
    | time >= begins && time < ends = paint (render pos size (drawTextureWithSin tex time)) bg
    | otherwise                     = bg
  where
    pos = (sin (time/1000)/3):.(cos (time/1000)/3):.()
    size = 5:.3:.() -- fullscreen
    paint = paintTransparent
    render = renderFullscreen
drawScene bg (Picture (Just tex) scaling (Just dim) pos begins ends) time
    | time >= begins && time < ends = paint (render pos size (drawTexture tex)) bg
    | otherwise                     = bg
  where
    size = fitToScale dim scaling -- fullscreen
    paint = paintTransparent
    render = renderFullscreen
drawScene bg (Nick text (Just tex) scaling (Just dim) pos t) time
    | time >= 68000 && time < 76000 = paint (render pos size ((transparentize a) . (drawTexture tex))) bg
    | otherwise                                   = bg
  where
    size = calcMarqueeSize dim scaling -- fullscreen
    paint = paintTransparent
    render = renderFullscreen
    a = flash ((time - 68000) / 1000) (t / 10) 0.5 
drawScene bg _ _ = bg

-- Draws all scenes onto background.
renderFrame :: Vec2 Int -> Time -> [Scene] -> FrameBuffer RGBAFormat () ()
renderFrame size time scenes = statics $ foldl (\acc scene -> drawScene acc scene time) clear scenes
  where
    statics bg = drawStatics bg time
    clear      = emptyFB 0.0 0.0 0.0 0.0

-- Old scene displaying functionality
marqueeScenePos :: Float -> TextPath -> Vec2 Float
marqueeScenePos time path
    | time >= (millis p1) && time < (millis p2) = calcPos time (millis p1) (millis p2) (coord p1) (coord p2)
    | time >= (millis p2) && time < (millis p3) = calcPos time (millis p2) (millis p3) (coord p2) (coord p3)
    | time >= (millis p3) && time < (millis p4) = calcPos time (millis p3) (millis p4) (coord p3) (coord p4)
    | otherwise               = 100:.100:.() -- Somewhat arbitrary, fix to use Maybe
    where p1 = begin path
          p2 = stable path
          p3 = fadeout path
          p4 = end path


drawStatics :: FrameBuffer RGBAFormat () () -> Time -> FrameBuffer RGBAFormat () ()
drawStatics bg time
	| time >= pacmanTime && time < pacmanTimeEnds = paintTransparent (renderPacman time (0:.0:.())) bg
        | otherwise                                   = bg
    where
	pacmanTime = 39500
        pacmanTimeEnds = 48000
	t = (time - 39500) / 1000
	x = t / 10
	y = sin(t)

calcRatio :: Float -> Float -> Float -> Float
calcRatio x a b = (x - a) / (b - a)

calcPos :: Float -> Float -> Float -> Vec2 Float -> Vec2 Float -> Vec2 Float
calcPos pos low high v1 v2 = v1 + (v2 - v1) * (ratio:.ratio:.()) -- Todo:: figure if there is overloading for lerp
    where ratio = calcRatio pos low high


-- Actual drawing functions
drawTextureWithSin :: Texture2D RGBAFormat
              -> Float
              -> FragmentStream (Vec2 (Fragment Float))
              -> FragmentStream (Vec3 (Fragment Float), Fragment Float)
drawTextureWithSin tex time input = fmap (sinEffect tex (toGPU time)) input 

sinEffect :: Texture2D RGBAFormat -> Fragment Float -> Vec2 (Fragment Float) -> (Vec3 (Fragment Float), Fragment Float)
sinEffect tex time (u:.v:.()) = ((color' color_orig) * (d:.d:.d:.()), alpha color_orig)
    where color_orig = stripRGB $ sample (Sampler Linear Wrap) tex (u:.v:.())
          d = 1.0 - (step w distance)
          distance = abs (v - (smoothsin * 0.5 + 0.5))
          smoothsin = sin(u * 10 + time / 1000)
          alpha (c, a) = a
          color' (c, a) = c
          w = 0.2

colorize :: Vec3 Float -> FragmentStream (Vec3 (Fragment Float), Fragment Float) -> FragmentStream (Vec3 (Fragment Float), Fragment Float)
colorize rgb input = fmap (colorFunc $ toGPU rgb) input
	where colorFunc cNew (c, a) = (cNew, a)

flash :: Float -> Float -> Float -> Float
flash time target width = max 0 $ min 1 $ 1.2 - distance 
	where distance = abs $ time - target 

transparentize :: Float -> FragmentStream (Vec3 (Fragment Float), Fragment Float) -> FragmentStream (Vec3 (Fragment Float), Fragment Float)
transparentize newA input = fmap (alphaFunc $ toGPU newA) input
	where alphaFunc a' (c, a) = (c, a * a')


renderPacman :: Float -> Vec2 Float -> FragmentStream (Color RGBAFormat (Fragment Float))
renderPacman time pos = outputColor $ stripUV $ alphaCircle time $ addColor (0.6:.0.1:.0.1:.(), 0.8) $ inputUv $ rasterizeFrontAndBack $ putOnScreen pos $ quad 2 2

addColor :: (Vec3 Float, Float) -> FragmentStream (Vec2 (Fragment Float)) -> FragmentStream (Vec3 (Fragment Float), Fragment Float, Vec2 (Fragment Float))
addColor rgb input = fmap (func $ toGPU rgb) input 
	where func (c, a) uv = (c, a, uv)

alphaCircle :: Float -> FragmentStream (Vec3 (Fragment Float), Fragment Float, Vec2 (Fragment Float)) -> FragmentStream (Vec3 (Fragment Float), Fragment Float, Vec2 (Fragment Float))
alphaCircle radius input = fmap (alphaFunc $ toGPU radius) input

alphaFunc :: Fragment Float -> (Vec3 (Fragment Float), Fragment Float, Vec2 (Fragment Float)) -> (Vec3 (Fragment Float), Fragment Float, Vec2 (Fragment Float))
alphaFunc time (rgb, a, (u:.v:.())) = (rgb, (step (sin (u * 20 + v * 20 + t) + sin (v * 10 * (sin (t + u)))) 0.2) * 0.2, (u:.v:.()))
	      where distance u v = sqrt $ (u-0.5) ** 2 + (v-0.5) ** 2
	            t = time / 1000

stripUV :: FragmentStream (Vec3 (Fragment Float), Fragment Float, Vec2 (Fragment Float)) -> FragmentStream (Vec3 (Fragment Float), Fragment Float)
stripUV input = fmap (\(rgb, a, uv) -> (rgb, a)) input


