/*****************************************************************
|
|      MPEG audio decoder. Sample Player
|
|      (c) 1996-1998 MpegTV, LLC
|      Author: Gilles Boccon-Gibod (gilles@mpegtv.com)
|
 ****************************************************************/

/*
** This sample application shows how to use the Xaudio SYNC API to
** play a bistream in a program that does not need the decoder to run
** in its own thread or process. This is typically the case of a command
** line program, even though a command like program could very well use
** the ASYNC API also.
*/

/*----------------------------------------------------------------------
|       includes
+---------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

/* xaudio general includes */
#include "xaudio.h"
#include "decoder.h"

/* xaudio plug-in modules */
#include "file_input.h"
#include "file_output.h"
#include "audio_output.h"
#include "mpeg_codec.h"

/*----------------------------------------------------------------------
|       global options
+---------------------------------------------------------------------*/
struct {
    struct {
        int port;
        int volume;
        char *name;
    } output;
} Options;

/*----------------------------------------------------------------------
|       constants
+---------------------------------------------------------------------*/
#define PROGTITLE       "XAUDIO. MPEG Audio Player"
#define VERSION         "1.0a"
#define PROGNAME        "xaudio"

char *ModeStrings[4] = {
    "stereo", "joint-stereo", "dual-channel", "single-channel"
};

char *LayerStrings[4] = {
    "invalid", "I", "II", "III"
};

/*----------------------------------------------------------------------
|       cmdline_print_version
+---------------------------------------------------------------------*/
static void
cmdline_print_version(void)
{
    fprintf(stderr, PROGNAME " version " VERSION " [decoder library %s]\n",
            decoder_version(NULL, NULL, NULL));
}

/*----------------------------------------------------------------------
|       cmdline_error
+---------------------------------------------------------------------*/
static void
cmdline_error(void)
{
    cmdline_print_version();
    fprintf(stderr, "use '" PROGNAME " -h' for a list of options\n");
    exit(1);
}

/*----------------------------------------------------------------------
|       cmdline_usage
+---------------------------------------------------------------------*/
static void
cmdline_usage(void)
{
    fprintf(stderr, 
            PROGTITLE "\n"
            "Version " VERSION " [decoder library %s]\n\n"
            "USAGE: " PROGNAME " [options] [<bitstream url>]\n"
            " <bitstream url> specifies input bitstream from.\n"
            " it can be one of:\n"
            "    <filename> or file:<filename>\n"
            "    http://[[user]:[pass]@]<hostname>[:port]/<path> (standard http URL)\n" 
            "    ftp://[[user]:[pass]@]<hostname>[:port]/<path> (standard ftp URL)\n"
            "    udp://<port> (listens for UDP broadcast on port <port>\n\n"
            " GENERAL OPTIONS:\n"
            "  -h, -help            prints this usage information\n"
            " OUTPUT OPTIONS:\n"
            "  -mono                output mono audio (left channel only)\n"
            "  -mono=left           same as -mono\n"
            "  -mono=right          output mono audio (right channel only)\n"
            "  -mono=mix            output mono audio (average left and right channels)\n\n"
            "  -output=<name>       specifies the name of the output.\n"
            "                       the output module is chosen based on the\n"
            "                       name (use 'xaudio -m' for a list of modules)\n"
            "                       with the builtin output modules, you can use:\n"
            "                       raw:<filename>, or just <filename> for raw PCM output\n"
            "                         or\n"
            "                       wav:<filename> for WAV output\n"
            "                         or\n"
            "                       - (just the characher '-') for standard output\n" 
            "  -port=line           enable audio on the sound card's line out\n"
            "  -port=speaker        enable audio on the speaker\n"
            "  -port=headphone      enable audio on the headphone\n"
            "                       (note: more than one -port=... option can be selected)\n"
            "  -volume=<n>          sets the volume of the audio output\n"
            "                       <n> is between 0 and 100\n",
            decoder_version(NULL, NULL, NULL)
            );
    exit(0);
}

/*----------------------------------------------------------------------
|       cmdline_parse_argv
+---------------------------------------------------------------------*/
static char **
cmdline_parse_argv(XA_DecoderInfo *decoder, char **argv)
{
    char *option;

    while((option = *argv++)) {
        if (!strcmp(option, "-h") || !strcmp(option, "-help")) {
            cmdline_usage();
        } else if (!strcmp(option, "-mono") || !strcmp(option, "-mono=left")) {
            decoder_codec_set_channels(decoder, XA_OUTPUT_MONO_LEFT);
        } else if (!strcmp(option, "-mono=right")) {
            decoder_codec_set_channels(decoder, XA_OUTPUT_MONO_RIGHT);
        } else if (!strcmp(option, "-mono=mix")) {
            decoder_codec_set_channels(decoder, XA_OUTPUT_MONO_MIX);
        } else if (!strncmp(option, "-output=", sizeof("-output=")-1)) {
            char *output = option+sizeof("-output=")-1;
            if (*output == '\0') {
                fprintf(stderr, PROGNAME ": invalid output filename\n");
                cmdline_error();
            }
            Options.output.name = output;
        } else if (!strncmp(option, "-port=", sizeof("-port=")-1)) {
            char *port = option+sizeof("-port=")-1;
            if (!strcmp(port, "line")) {
                Options.output.port |= XA_DECODER_CONTROL_OUTPUT_LINE;
            } else if (!strcmp(port, "speaker")) {
                Options.output.port |= XA_DECODER_CONTROL_OUTPUT_SPEAKER;
            } else  if (!strcmp(port, "headphone")) {
                Options.output.port |= XA_DECODER_CONTROL_OUTPUT_HEADPHONE;
            } else {
                fprintf(stderr, PROGNAME ": invalid port name [%s]\n",
                        option);
                cmdline_error();
            }
        } else if (!strncmp(option, "-volume=", sizeof("-volume=")-1)) {
            int volume = strtol(option+sizeof("-volume=")-1, NULL, 0);
            if (errno == EINVAL) {
                fprintf(stderr, PROGNAME ": invalid volume value [%s]\n", 
                        option);
                cmdline_error();
            }
            if (volume < 0 || volume > 100) {
                fprintf(stderr, PROGNAME ": volume out of range [%d]\n",
                        volume);
                cmdline_error();
            }
            Options.output.volume = volume;
        } else {
            /* filename */
            if (option[0] == '-') {
                fprintf(stderr, PROGNAME ": invalid otpion %s\n", option);
                cmdline_error();
            }
            return --argv;
        }
    }
    return --argv;
}

/*----------------------------------------------------------------------
|       cmdline_notify_progress
+---------------------------------------------------------------------*/
static void
cmdline_notify_progress(void *client, int source, int code, long value, 
                        const char *message)
{
    fprintf(stderr, "XAUDIO: progress code=%d value=%ld %s\n",
            code, value, message);
}

/*----------------------------------------------------------------------
|       cmdline_notify_debug
+---------------------------------------------------------------------*/
static void
cmdline_notify_debug(void *client, int source, int level, 
                     const char *message, const char *reason)
{
    fprintf(stderr, "XAUDIO: debug, source=%d, level=%d, %s ", 
            source, level, message);
    if (reason && strlen(reason)) {
        fprintf(stderr, "(%s)", reason);
    }
    fprintf(stderr, "\n");
}

/*----------------------------------------------------------------------
|       cmdline_notify_error
+---------------------------------------------------------------------*/
static void
cmdline_notify_error(void *client, int source, int code, 
                     const char *message, const char *reason)
{
    fprintf(stderr, "XAUDIO: error, source=%d, code=%d, %s", 
            source, code, message);
    if (reason && strlen(reason)) {
        fprintf(stderr, "(%s)", reason);
    }
    fprintf(stderr, "\n");
}

/*----------------------------------------------------------------------
|       cmdline_init
+---------------------------------------------------------------------*/
static XA_DecoderInfo *
cmdline_init(void)
{
    XA_DecoderInfo *decoder;
    int             status;

    /* create a decoder object */
    status = decoder_new(&decoder);
    if (status != XA_SUCCESS) {
        fprintf(stderr, 
                "XAUDIO: cannot init decoder (%s)",
                xaudio_error_string(status));
        exit(1);
    }

    /* register mpeg audio codec */
    {
        XA_CodecModule module;

        mpeg_codec_module_register(&module);
        decoder_codec_module_register(decoder, &module);
    }

    /* register the input modules (the registration order is important) */
    {
        XA_InputModule module;

        file_input_module_register(&module);
        decoder_input_module_register(decoder, &module);
    }
    
    /* register the output modules (the registration order is important) */
    {
        XA_OutputModule module;

        file_output_module_register(&module);
        decoder_output_module_register(decoder, &module);
        audio_output_module_register(&module);
        decoder_output_module_register(decoder, &module);
    }

    /* set the notification function */
    decoder->notification_client.client          = decoder;
    decoder->notification_client.notify_progress = cmdline_notify_progress;
    decoder->notification_client.notify_debug    = cmdline_notify_debug;
    decoder->notification_client.notify_error    = cmdline_notify_error;

    /* output parameters */
    Options.output.port                 = 0;
    Options.output.volume               = -1;
    Options.output.name                 = NULL;

    return decoder;
}

/*----------------------------------------------------------------------
|       cmdline_monitor_stream_info
+---------------------------------------------------------------------*/
static void
cmdline_monitor_stream_info(XA_DecoderInfo *decoder)
{
    static unsigned long duration = 0;

    if (decoder->status->info.changed) {
        fprintf(stderr, 
                "XAUDIO: stream type = MPEG %d, layer %s, %d kbps, %d hz, %s\n",
                decoder->status->info.level,
                LayerStrings[decoder->status->info.layer],
                decoder->status->info.bitrate,
                decoder->status->info.frequency,
                ModeStrings[decoder->status->info.mode]);
    }
    if (decoder->status->info.duration != duration) {
        duration = decoder->status->info.duration;
        fprintf(stderr, "XAUDIO: input duration = ");
        if (duration) {
            fprintf(stderr, "%ld seconds\n", duration/1000);
        } else {
            fprintf(stderr, "unknown\n");
        }
    }
}

/*----------------------------------------------------------------------
|       main
+---------------------------------------------------------------------*/
int 
main(int argc, char **argv)
{
    XA_DecoderInfo *decoder;
    char          **urls;
    char           *stdin_input[] = {"-", 0};
    char           *current;
    int             status;

    /* instanciate a decoder */
    decoder = cmdline_init();

    /* read options */
    urls = cmdline_parse_argv(decoder, argv+1);

    /* are we playing from stdin ? */
    if (!urls[0]) {
        urls = stdin_input;
        fprintf(stderr, "XAUDIO: playing from stdin\n");
    }

    if (Options.output.port == 0) {
        Options.output.port = 
            XA_DECODER_CONTROL_OUTPUT_SPEAKER | 
            XA_DECODER_CONTROL_OUTPUT_LINE;
    };

    /* open output */
    decoder_output_new(decoder, 
                       Options.output.name,    
                       XA_DECODER_OUTPUT_AUTOSELECT);
    status = decoder_output_open(decoder);
    if (status != XA_SUCCESS) {
        fprintf(stderr, 
                "XAUDIO: cannot instanciate  output (%s)\n",
                xaudio_error_string(status));
        exit(1);
    }

    {
        XA_OutputControl control;
        unsigned long    to_set = 0;

        control.ports = Options.output.port;
        to_set |= XA_DECODER_CONTROL_PORTS;

        if (Options.output.volume != -1) {
            control.pcm_level = Options.output.volume;
            to_set |= XA_DECODER_CONTROL_PCM_LEVEL;
        }

        decoder_output_set_control(decoder, &control, to_set);
    }

    while((current = *urls++)) {
        decoder_input_new(decoder, current, XA_DECODER_INPUT_AUTOSELECT);
        status = decoder_input_open(decoder);
        if (status == XA_SUCCESS) {
            cmdline_monitor_stream_info(decoder);
            do {
                status = decoder_play(decoder);
                if (status == XA_SUCCESS) cmdline_monitor_stream_info(decoder);
            } while(status == XA_SUCCESS ||
                    status == XA_ERROR_INVALID_FRAME ||
                    status == XA_ERROR_TIMEOUT);

            decoder_output_set_control(decoder, NULL, 
                                       XA_DECODER_CONTROL_DRAIN);
            decoder_input_close(decoder);
        } else {
            fprintf(stderr, 
                    "XAUDIO: cannot open input (%s)\n",
                    xaudio_error_string(status));
        }
    }

    decoder_delete(decoder);
    return 0;
}












































