/*****************************************************************
|
|      XAUDIO SDK. Sample Output Filter
|
|      (c) 1996-1998 MpegTV, LLC
|      Author: Gilles Boccon-Gibod (gilles@mpegtv.com)
|
 ****************************************************************/

/*
** This is a sample input filter.
** This module shows a simple implementation of an output filter
** that removes every other sample in the output buffer (reducing its
** size to 1/2 of the original size, so that the sound will play twice
** as fast.
** It also prints a message or shows a message box for each of the 
** functions called, to show the order in which the Xaudio library makes 
** calls to the module's methods.
*/

/*----------------------------------------------------------------------
|       includes
+---------------------------------------------------------------------*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
 
#include "decoder.h"
#include "xaudio.h"

/*----------------------------------------------------------------------
|       types
+---------------------------------------------------------------------*/
typedef struct {
    void           *output;
    XA_DecoderInfo *decoder;
} OutputFilter;

/*----------------------------------------------------------------------
|       constants and macros
+---------------------------------------------------------------------*/
#ifdef WIN32
#include <windows.h>
#define TRACE(x) MessageBox(NULL, x, "Output Module", MB_OK)
#else
#define TRACE(x) printf("OUTPUT MODULE: %s\n", (x))
#endif
#define TRACE_0(x) TRACE(x)
#define TRACE_1(x, a) {char s[256]; sprintf(s, (x), a); TRACE(s);}
#define TRACE_2(x, a, b) {char s[256]; sprintf(s, (x), a, b); TRACE(s);}
#define TRACE_3(x, a, b, c) {char s[256]; sprintf(s, (x), a, b, c); TRACE(s);}

/*----------------------------------------------------------------------
|       output_filter_get_name
+---------------------------------------------------------------------*/
static const char *
output_filter_get_name(void)
{
    return "sample-output-filter";
}

/*----------------------------------------------------------------------
|       output_filter_new
+---------------------------------------------------------------------*/
static int
output_filter_new(void **ret_filter, int id, 
                  XA_DecoderInfo *decoder)
{
    /* allocate an output filter object */
    OutputFilter *filter = (OutputFilter *)malloc(sizeof(OutputFilter));
    if (filter == NULL) return XA_ERROR_OUT_OF_MEMORY;
    *ret_filter = filter;

    TRACE_1("output_filter_new, id=%d", id);

    /* store the parameters */
    filter->decoder = decoder;

    return XA_SUCCESS;
}

/*----------------------------------------------------------------------
|       output_filter_delete
+---------------------------------------------------------------------*/
static int
output_filter_delete(void *filter) 
{
    TRACE_0("output_filter_delete");

    if (filter) {
        free(filter);
    }

    return XA_SUCCESS;
}

/*----------------------------------------------------------------------
|       output_filter_write
+---------------------------------------------------------------------*/
static int
output_filter_write(XA_OutputFilterNode *node, 
                    void *buffer, unsigned long size, int bytes_per_sample,
                    unsigned int channels, unsigned int sample_rate)
{
    short *samples = (short *)buffer;
    int    i;

    for (i=0; i<size/4; i++) {
        samples[i] = samples[2*(i-(i%2))+(i%2)];
    }
    
    return node->next->vtable.output_write(node->next, 
                                           buffer, size/2,
                                           bytes_per_sample, 
                                           channels, 
                                           sample_rate);
}

/*----------------------------------------------------------------------
|       output_output_filter_register
+---------------------------------------------------------------------*/
int XA_EXPORT
output_filter_register(XA_OutputFilter *filter)
{
    filter->filter_get_name     = output_filter_get_name;
    filter->filter_new          = output_filter_new;
    filter->filter_delete       = output_filter_delete;
    filter->output_open         = NULL;
    filter->output_close        = NULL;
    filter->output_write        = output_filter_write;
    filter->output_get_buffer   = NULL;
    filter->output_set_control  = NULL;
    filter->output_get_control  = NULL;
    filter->output_get_caps     = NULL;
    filter->output_get_status   = NULL;
    filter->output_send_message = NULL;

    return XA_SUCCESS;
} 
