/*----------------------------------------------------------------------
|
|-----------------------------------------------------------------------
|                   Copyright (c) 1997, Alain Jobart (aj@cybersoft.org)
|-----------------------------------------------------------------------
|
|       simple_motif.c
|
|          Motif interface example
|
+---------------------------------------------------------------------*/

/*----------------------------------------------------------------------
|       includes
+---------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

#include <Xm/MainW.h>
#include <Xm/Form.h>
#include <Xm/PushB.h>
#include <Xm/Scale.h>
#include <Xm/FileSB.h>
#include <Xm/Label.h>
#include <Xm/Text.h>

#include "xaudio.h"
#include "decoder.h"
#include "player.h"
#include "control_pipe.h"

/*----------------------------------------------------------------------
|       types
+---------------------------------------------------------------------*/
/* Current state of the player:
 * PLAYER_INITIALIZING means the forked process is initializing
 * PLAYER_IDLE         means stopped
 * PLAYER_PLAYING      means playing something
 * PLAYER_PAUSE        means pause */
typedef enum {
    PLAYER_INITIALIZING,
    PLAYER_IDLE,
    PLAYER_PLAYING,
    PLAYER_PAUSE
} PlayerState;

/*----------------------------------------------------------------------
|       constants
+---------------------------------------------------------------------*/
#define DRAG_TIMEOUT    500

/*----------------------------------------------------------------------
|       global variables
+---------------------------------------------------------------------*/

/* Current state of the player */
static PlayerState State;

/* List of files to play */
static char **file_list = NULL;

/* forked process */
static XA_PipeControl *player;

/* X variables */
Display *display;
Screen *screen;
XtAppContext app_context;

/* Main widgets */
static Widget toplevel;
static Widget main_w;
static Widget play_button; 
static Widget pause_button; 
static Widget stop_button;
static Widget open_button;
static Widget exit_button;
static Widget timecode;
static Widget slide_bar; 
static Widget duration;
static Widget filename;
static Widget params;

/* Flags */
int currently_dragging = 0;

/* Default starting ressources of the application */
static char *app_resources[] =
{
    "*background:      grey",
    "*foreground:      black",
    "*text.background: whitesmoke",
    NULL
};

/* input ID on the communication pipe */
static XtInputId input_id = 0;

/*----------------------------------------------------------------------
|       PlayFile
+---------------------------------------------------------------------*/
static void
PlayFile(char *name)
{
    XtVaSetValues (filename, XmNvalue, name, NULL);
    control_message_send (player, XA_MSG_COMMAND_INPUT_OPEN, name);
    control_message_send (player, XA_MSG_COMMAND_PLAY);
}

/*----------------------------------------------------------------------
|       OpenPressedOk
+---------------------------------------------------------------------*/
static void
OpenPressedOk (Widget w, XtPointer cd, XtPointer call_data)
{
    XmFileSelectionBoxCallbackStruct *cbs =
        (XmFileSelectionBoxCallbackStruct *) call_data;
    char *file = NULL;
    XmStringGetLtoR (cbs->value, XmSTRING_DEFAULT_CHARSET, &file);
    XtUnmapWidget (XtParent (w));

    PlayFile(file);

    XtFree (file);
}

/*----------------------------------------------------------------------
|       OpenPressedCancel
+---------------------------------------------------------------------*/
static void
OpenPressedCancel (Widget w, XtPointer cd, XtPointer call_data)
{
    XtUnmapWidget (XtParent (w));
}

/*----------------------------------------------------------------------
|       OpenPressed
+---------------------------------------------------------------------*/
static void
OpenPressed (Widget w, XtPointer cd, XtPointer call_data)
{
    Widget dialog;

    dialog = XmCreateFileSelectionDialog (toplevel, "Open File",
                                          NULL, 0);

    XtAddCallback (dialog, XmNokCallback, OpenPressedOk, NULL);
    XtAddCallback (dialog, XmNcancelCallback, OpenPressedCancel, NULL);

    XtManageChild (dialog);
    XtPopup (XtParent(dialog), XtGrabExclusive);
}

/*----------------------------------------------------------------------
|       PlayPressed
+---------------------------------------------------------------------*/
static void
PlayPressed (Widget w, XtPointer client_data, XtPointer call_data)
{
    control_message_send (player, XA_MSG_COMMAND_PLAY);
}

/*----------------------------------------------------------------------
|       PausePressed
+---------------------------------------------------------------------*/
static void
PausePressed (Widget w, XtPointer client_data, XtPointer call_data)
{
    control_message_send (player, XA_MSG_COMMAND_PAUSE);
}

/*----------------------------------------------------------------------
|       StopPressed
+---------------------------------------------------------------------*/
static void
StopPressed (Widget w, XtPointer client_data, XtPointer call_data)
{
    control_message_send (player, XA_MSG_COMMAND_STOP);
}

/*----------------------------------------------------------------------
|       ExitPressed
+---------------------------------------------------------------------*/
static void
ExitPressed (Widget w, XtPointer client_data, XtPointer call_data)
{
    control_message_send (player, XA_MSG_COMMAND_EXIT);
}

/*----------------------------------------------------------------------
|       SlideCB
+---------------------------------------------------------------------*/
static void
SlideCB (Widget w, XtPointer client_data, XtPointer call_data)
{
    if (currently_dragging != 2) {
        int value, maximum;
        XtVaGetValues (w,
                       XmNvalue, &value,
                       XmNmaximum, &maximum,
                       NULL);
        control_message_send (player, XA_MSG_COMMAND_SEEK, value, maximum);
    }

    currently_dragging = 0;

}

/*----------------------------------------------------------------------
|       SlideDragCB
+---------------------------------------------------------------------*/
void
SlideDragCB (Widget w, XtPointer client_data, XtPointer call_data)
{
    /* say we're moving the slide bar */
    currently_dragging = 1;
}

/*----------------------------------------------------------------------
|       CreateMainWindow
+---------------------------------------------------------------------*/
static void
CreateMainWindow()
{
    Widget main_form, label;

    main_w = XtVaCreateWidget ("main_w", xmMainWindowWidgetClass, toplevel,
                               NULL);
    
    main_form = XtVaCreateWidget ("form", xmFormWidgetClass, main_w,
                                  NULL);
    
    /* create the Play button */
    play_button = XtVaCreateManagedWidget ("Play", xmPushButtonWidgetClass,
                                           main_form,
                                           XmNleftAttachment, XmATTACH_FORM,
                                           XmNtopAttachment, XmATTACH_FORM,
                                           NULL);
    XtAddCallback (play_button, XmNactivateCallback, PlayPressed, NULL);
    XtSetSensitive (play_button, False);

    /* create the Pause button */
    pause_button = XtVaCreateManagedWidget ("Pause", xmPushButtonWidgetClass,
                                            main_form,
                                            XmNleftAttachment, XmATTACH_WIDGET,
                                            XmNleftWidget, play_button,
                                            XmNtopAttachment, XmATTACH_FORM,
                                            NULL);
    XtAddCallback (pause_button, XmNactivateCallback, PausePressed, NULL);
    XtSetSensitive (pause_button, False);

    /* create the Stop button */
    stop_button = XtVaCreateManagedWidget ("Stop", xmPushButtonWidgetClass,
                                            main_form,
                                            XmNleftAttachment, XmATTACH_WIDGET,
                                            XmNleftWidget, pause_button,
                                            XmNtopAttachment, XmATTACH_FORM,
                                            NULL);
    XtAddCallback (stop_button, XmNactivateCallback, StopPressed, NULL);
    XtSetSensitive (stop_button, False);

    /* create the Exit button */
    exit_button = XtVaCreateManagedWidget ("Exit", xmPushButtonWidgetClass,
                                           main_form,
                                           XmNtopAttachment, XmATTACH_FORM,
                                           XmNrightAttachment, XmATTACH_FORM,
                                           NULL);
    XtAddCallback (exit_button, XmNactivateCallback, ExitPressed, NULL);

    /* create the Open button */
    open_button = XtVaCreateManagedWidget ("Open File...", 
                                           xmPushButtonWidgetClass,
                                           main_form,
                                           XmNtopAttachment, XmATTACH_FORM,
                                           XmNrightAttachment, XmATTACH_WIDGET,
                                           XmNrightWidget, exit_button,
                                           NULL);
    XtAddCallback (open_button, XmNactivateCallback, OpenPressed, NULL);

    /* create the slide bar */
    slide_bar = XtVaCreateManagedWidget ("time_slide_bar", xmScaleWidgetClass,
                                         main_form,
                                         XmNorientation, XmHORIZONTAL,
                                         XmNwidth, 300,
                                         XmNtopAttachment, XmATTACH_WIDGET,
                                         XmNtopWidget, play_button,
                                         XmNleftAttachment, XmATTACH_FORM,
                                         XmNrightAttachment, XmATTACH_FORM,
                                         XmNminimum, 0,
                                         XmNmaximum, 400,
                                         XmNpageIncrement, 15,
                                         XmNincrement, 1,
                                         XmNvalue, 0,
                                         NULL);
    XtAddCallback (slide_bar, XmNvalueChangedCallback, SlideCB, NULL);
    XtAddCallback (slide_bar, XmNdragCallback, SlideDragCB, NULL);

    /* create the timecode label */
    label = XtVaCreateManagedWidget ("Timecode:", xmLabelWidgetClass,
                                      main_form,
                                      XmNtopAttachment, XmATTACH_WIDGET,
                                      XmNtopWidget, slide_bar,
                                      XmNleftAttachment, XmATTACH_FORM,
                                      NULL);

    /* create the timecode display */
    timecode = XtVaCreateManagedWidget ("timecode", xmTextWidgetClass,
                                        main_form,
                                        XmNeditable, False,
                                        XmNvalue, "00:00:00",
                                        XmNcolumns, 8,
                                        XmNtopAttachment, XmATTACH_WIDGET,
                                        XmNtopWidget, slide_bar,
                                        XmNleftAttachment, XmATTACH_WIDGET,
                                        XmNleftWidget, label,
                                        NULL);

    /* create the duration display */
    duration = XtVaCreateManagedWidget ("duration",
                                        xmTextWidgetClass, main_form,
                                        XmNeditable, False,
                                        XmNvalue, "00:00:00",
                                        XmNcolumns, 8,
                                        XmNtopAttachment, XmATTACH_WIDGET,
                                        XmNtopWidget, slide_bar,
                                        XmNrightAttachment, XmATTACH_FORM,  
                                        NULL);

    /* create the duration label */
    label = XtVaCreateManagedWidget ("Duration:", xmLabelWidgetClass,
                                      main_form,
                                      XmNtopAttachment, XmATTACH_WIDGET,
                                      XmNtopWidget, slide_bar,
                                      XmNrightAttachment, XmATTACH_WIDGET,
                                      XmNrightWidget, duration,
                                      NULL);


    /* create the filename display */
    filename = XtVaCreateManagedWidget ("filename", xmTextWidgetClass,
                                        main_form,
                                        XmNeditable, False,
                                        XmNvalue, "",
                                        XmNtopAttachment, 
                                        XmATTACH_WIDGET,
                                        XmNtopWidget, duration,
                                        XmNleftAttachment, XmATTACH_FORM,
                                        XmNrightAttachment, XmATTACH_FORM,
                                        NULL);

    /* create the params display */
    params = XtVaCreateManagedWidget ("params", xmTextWidgetClass,
                                      main_form,
                                      XmNeditable, False,
                                      XmNvalue, "",
                                      XmNtopAttachment, XmATTACH_WIDGET,
                                      XmNtopWidget, filename,
                                      XmNleftAttachment, XmATTACH_FORM,
                                      XmNrightAttachment, XmATTACH_FORM,
                                      NULL);
    XtManageChild (main_form);
}

/*----------------------------------------------------------------------
|       PipeInputCB
+---------------------------------------------------------------------*/
static void
PipeInputCB (XtPointer cd, int *fd, XtInputId *i)
{
    XA_Message message;
    int status;

    status = control_message_wait((void*)player, &message, 
                                  XA_TIMEOUT_INFINITE);
    do {
        if (status < 0) {
            if (status == XA_ERROR_PIPE_CLOSED) {
                /* display the error message */
                fprintf (stderr, "XAPLAYER: decoder process exited. Exiting.\n");

                /* stop waiting of the file descriptor
                 * just for demo purposes, since we exit just after this
                 */
                if (input_id) {
                    XtRemoveInput (input_id);
                    input_id = 0;
                }

                /* and exit */
                exit (EXIT_FAILURE);
            } else {
                fprintf (stderr, "XAPLAYER: error %d in message_wait()\n",
                         status);
            }
        } else {
            switch (message.code) {
              case XA_MSG_NOTIFY_READY:
                State = PLAYER_IDLE;
                /* if we had a filename on the command line, play it */
                if (file_list) {
                    PlayFile(*file_list++);
                }
                break;

              case XA_MSG_NOTIFY_EXITED:
                player_delete((void*)player);
                exit(0);
                break;

              case XA_MSG_NOTIFY_INPUT_CAPS:
                XtVaSetValues (slide_bar, XmNeditable, 
                               message.data.caps & XA_DECODER_INPUT_SEEKABLE,
                               NULL);
                break;

              case XA_MSG_NOTIFY_PLAYER_STATE:
                switch(message.data.state) {
                  case XA_PLAYER_STATE_PLAYING:
                    State = PLAYER_PLAYING;
                    XtSetSensitive (pause_button, True);
                    XtSetSensitive (stop_button, True);
                    break;

                  case XA_PLAYER_STATE_PAUSED:
                    State = PLAYER_PAUSE;
                    break;

                  case XA_PLAYER_STATE_STOPPED:
                    State = PLAYER_IDLE;
                    XtSetSensitive (pause_button, False);
                    XtSetSensitive (stop_button, False);
                    XtVaSetValues (timecode, XmNvalue, "00:00:00", NULL);
                    break;

                  case XA_PLAYER_STATE_EOF:
                    /* here we can go to the next track */
                    if (file_list && *file_list) {
                        PlayFile(*file_list++);    
                    } else {
                        State = PLAYER_IDLE;
                        XtSetSensitive (pause_button, False);
                        XtSetSensitive (stop_button, False);
                    }
                    break;

                  default:
                    break;
                }
                break;

              case XA_MSG_NOTIFY_ACK:
                if (message.data.ack == XA_MSG_COMMAND_INPUT_OPEN) {
                    XtSetSensitive (play_button, True);
                }
                break;

              case XA_MSG_NOTIFY_INPUT_POSITION:
                if (currently_dragging == 0) {
                    int maximum;
                    XtVaGetValues (slide_bar, XmNmaximum, &maximum, NULL);
                    if (maximum != message.data.position.range) {
                        XtVaSetValues (slide_bar,
                                       XmNvalue, message.data.position.offset,
                                       XmNmaximum, message.data.position.range,
                                       NULL);
                    } else {
                        XtVaSetValues (slide_bar,
                                       XmNvalue, message.data.position.offset,
                                       NULL);
                    }
                }
                break;

              case XA_MSG_NOTIFY_INPUT_TIMECODE:
                {
                    char code[20];
                    sprintf (code, "%.2d:%.2d:%.2d", message.data.timecode.h,
                             message.data.timecode.m, message.data.timecode.s);
                    XtVaSetValues (timecode, XmNvalue, code, NULL);
                }
                break;

              case XA_MSG_NOTIFY_INPUT_DURATION:
                {
                    int h, m, s;
                    char code[20];

                    h = message.data.duration / 3600;
                    m = (message.data.duration - (h * 3600)) / 60;
                    s = message.data.duration % 60;
                    sprintf (code, "%.2d:%.2d:%.2d", h, m, s);
                    XtVaSetValues (duration, XmNvalue, code, NULL);
                    break;
                }

              case XA_MSG_NOTIFY_INPUT_STREAM_INFO:
                {
                    char info[256];
                    sprintf(info, "MPEG %d, layer %d, %d kbps, %d Hz",
                            message.data.stream_info.level,
                            message.data.stream_info.layer,
                            message.data.stream_info.bitrate,
                            message.data.stream_info.frequency);
                    XtVaSetValues (params, XmNvalue, info, NULL); 
                    break;
                }

              case XA_MSG_NOTIFY_OUTPUT_CAPS:
                /* here we can use the capabilities of the output device */
                break;

              case XA_MSG_NOTIFY_OUTPUT_BALANCE:
                /* get balance in message.data.volume.balance (0-100) */
                break;

              case XA_MSG_NOTIFY_OUTPUT_PCM_LEVEL:
                /* get volume in message.data.volume.level (0-100) */
                break;

              case XA_MSG_NOTIFY_OUTPUT_MASTER_LEVEL:
                /* get master volume in message.data.volume.master (0-100) */
                break;

              case XA_MSG_NOTIFY_ERROR: {
                  fprintf (stderr, "XAPLAYER: error %d (%s)\n",
                           message.data.error.code,
                           message.data.error.message);
                break;
              }

              case XA_MSG_NOTIFY_DEBUG:
                printf ("XAPLAYER: debug message (source=%d,level=%d): %s\n",
                        message.data.debug.source,
                        message.data.debug.level,
                        message.data.debug.message);
                break;

              default:
                break;
            }
        }
    } while (((status = control_message_get ((void*)player, &message)) != 0)
        && (input_id));
}

/*----------------------------------------------------------------------
|       main
+---------------------------------------------------------------------*/
int
main (int argc, char **argv)
{
    /* Open the X connexion */
    XtSetLanguageProc(NULL, NULL, NULL);
    toplevel = XtVaAppInitialize(&app_context, "XAudio Sample Motif Player",
                                 NULL, 0, &argc, argv,
                                 app_resources, NULL, 0);
    display = XtDisplay (toplevel);
    screen = XtScreen (toplevel);

    /* try to start the player (if this fails, why should we go on ?) */
    if (player_new((void**)&player, NULL) != XA_SUCCESS) {
        fprintf (stderr, "XAPLAYER: cannot not create decoder thread.\n");
        exit (EXIT_FAILURE);
    }

    /* Init the state machine */
    State = PLAYER_INITIALIZING;

    /* add the callback to process the messages */
    input_id = XtAppAddInput(app_context, player->command_read_pipe,
                             (XtPointer) XtInputReadMask, PipeInputCB, NULL);

    /* Creates main window */
    XtVaSetValues(toplevel, XmNtitle, "XAudio Sample Motif Player", NULL);
    CreateMainWindow();

    /* if we have a file list, store it */
    if (argc >= 2) file_list = &argv[1];

    /* and start the motif main loop */
    XtManageChild (main_w);
    XtRealizeWidget(toplevel);
    XtAppMainLoop(app_context);
    
    return 1;
}





