#include <stdlib.h>
#include <stdio.h>
#include <SDL/SDL.h>
//#include <SDL/SDL_image.h>
#include <fmodex/fmod.h>
#include <fmodex/fmod_errors.h>
#include <math.h>
#include "wiimote.h"

#define MIN_DELAY_CHANGE 30

#define TRUE              1
#define FALSE             0

#define WIDTH             640
#define HEIGHT            480
#define SDL_ERROR         -1

#define TRACKS_BEATBASS_FILENAME "./tracks/beatbass.ogg"
#define TRACKS_TUNE_FILENAME "./tracks/tune.ogg"

#define MUSIC_COMPRESSOR_THRESHOLD -0.2
#define MUSIC_COMPRESSOR_ATTACK 50
#define MUSIC_COMPRESSOR_RELEASE 500
#define MUSIC_COMPRESSOR_GAIN 0

#define STEP_ADVANCE  150
#define STEP_PAN      0.1

#define EFFECT_COUNT_STEP       100

#define CHORUS_RATE_REAL_MAX    20.0
#define CHORUS_DELAY_REAL_MIN   0.1
#define CHORUS_DELAY_REAL_MAX   100

#define ECHO_DELAY_REAL_MIN     10.0
#define ECHO_DELAY_REAL_MAX     5000.0

#define FLANGE_RATE_REAL_MAX    20.0

#define CUTOFF_REAL_MIN      1.0
#define CUTOFF_REAL_MAX      2200.0
#define RESONANCE_REAL_MIN   1.0
#define RESONANCE_REAL_MAX   10.0

#define TUNE_FLANGE_DRY_MIN       0.3
#define TUNE_FLANGE_DRY_MAX       1.0
#define TUNE_FLANGE_DRY_STEP      (TUNE_FLANGE_DRY_MAX - TUNE_FLANGE_DRY_MIN) / EFFECT_COUNT_STEP
#define TUNE_FLANGE_WET_MIN       0.0
#define TUNE_FLANGE_WET_MAX       0.7
#define TUNE_FLANGE_WET_STEP      (TUNE_FLANGE_WET_MAX - TUNE_FLANGE_WET_MIN) / EFFECT_COUNT_STEP
#define TUNE_FLANGE_DEPTH_MIN     0.0
#define TUNE_FLANGE_DEPTH_MAX     0.3
#define TUNE_FLANGE_DEPTH_STEP    (TUNE_FLANGE_DEPTH_MAX - TUNE_FLANGE_DEPTH_MIN) / EFFECT_COUNT_STEP
#define TUNE_FLANGE_RATE_MIN      0.0 * FLANGE_RATE_REAL_MAX
#define TUNE_FLANGE_RATE_MAX      0.6 * FLANGE_RATE_REAL_MAX
#define TUNE_FLANGE_RATE_STEP     (TUNE_FLANGE_RATE_MAX - TUNE_FLANGE_RATE_MIN) / EFFECT_COUNT_STEP

#define TUNE_CHORUS_DRY_MIN         0.5
#define TUNE_CHORUS_DRY_MAX         1.0
#define TUNE_CHORUS_DRY_STEP        (TUNE_CHORUS_DRY_MAX - TUNE_CHORUS_DRY_MIN) / EFFECT_COUNT_STEP
#define TUNE_CHORUS_WET1_MIN        0.0
#define TUNE_CHORUS_WET1_MAX        0.7
#define TUNE_CHORUS_WET1_STEP       (TUNE_CHORUS_WET1_MAX - TUNE_CHORUS_WET1_MIN) / EFFECT_COUNT_STEP
#define TUNE_CHORUS_WET2_MIN        0.0
#define TUNE_CHORUS_WET2_MAX        0.7
#define TUNE_CHORUS_WET2_STEP       (TUNE_CHORUS_WET2_MAX - TUNE_CHORUS_WET2_MIN) / EFFECT_COUNT_STEP
#define TUNE_CHORUS_WET3            0.1
#define TUNE_CHORUS_DELAY           (0.1 * (CHORUS_DELAY_REAL_MAX - CHORUS_DELAY_REAL_MIN)) + CHORUS_DELAY_REAL_MIN
#define TUNE_CHORUS_RATE            0.1 * CHORUS_RATE_REAL_MAX
#define TUNE_CHORUS_DEPTH           0.15
#define TUNE_CHORUS_FEEDBACK        0.05

#define TUNE_LOWPASS_CUTOFF_MIN       (0.4 * (CUTOFF_REAL_MAX - CUTOFF_REAL_MIN)) + CUTOFF_REAL_MIN
#define TUNE_LOWPASS_CUTOFF_MAX       (1.0 * (CUTOFF_REAL_MAX - CUTOFF_REAL_MIN)) + CUTOFF_REAL_MIN
#define TUNE_LOWPASS_CUTOFF_STEP      (TUNE_LOWPASS_CUTOFF_MAX - TUNE_LOWPASS_CUTOFF_MIN) / EFFECT_COUNT_STEP
#define TUNE_LOWPASS_RESONANCE_MIN    (0.0 * (RESONANCE_REAL_MAX - RESONANCE_REAL_MIN)) + RESONANCE_REAL_MIN
#define TUNE_LOWPASS_RESONANCE_MAX    (0.14 * (RESONANCE_REAL_MAX - RESONANCE_REAL_MIN)) + RESONANCE_REAL_MIN
#define TUNE_LOWPASS_RESONANCE_STEP   (TUNE_LOWPASS_RESONANCE_MAX - TUNE_LOWPASS_RESONANCE_MIN) / EFFECT_COUNT_STEP

#define TUNE_HIGHPASS_CUTOFF_MIN       (0.0 * (CUTOFF_REAL_MAX - CUTOFF_REAL_MIN)) + CUTOFF_REAL_MIN
#define TUNE_HIGHPASS_CUTOFF_MAX       (0.8 * (CUTOFF_REAL_MAX - CUTOFF_REAL_MIN)) + CUTOFF_REAL_MIN
#define TUNE_HIGHPASS_CUTOFF_STEP      (TUNE_HIGHPASS_CUTOFF_MAX - TUNE_HIGHPASS_CUTOFF_MIN) / EFFECT_COUNT_STEP
#define TUNE_HIGHPASS_RESONANCE_MIN    (0.0 * (RESONANCE_REAL_MAX - RESONANCE_REAL_MIN)) + RESONANCE_REAL_MIN
#define TUNE_HIGHPASS_RESONANCE_MAX    (0.14 * (RESONANCE_REAL_MAX - RESONANCE_REAL_MIN)) + RESONANCE_REAL_MIN
#define TUNE_HIGHPASS_RESONANCE_STEP  (TUNE_HIGHPASS_RESONANCE_MAX - TUNE_HIGHPASS_RESONANCE_MIN) / EFFECT_COUNT_STEP

#define TUNE_ECHO_DELAY       (0.14 * (ECHO_DELAY_REAL_MAX - ECHO_DELAY_REAL_MIN)) + ECHO_DELAY_REAL_MIN 
#define TUNE_ECHO_DECAY_MIN   0.0
#define TUNE_ECHO_DECAY_MAX   0.5
#define TUNE_ECHO_DECAY_STEP  (TUNE_ECHO_DECAY_MAX - TUNE_ECHO_DECAY_MIN) / EFFECT_COUNT_STEP
#define TUNE_ECHO_WET_MIN   0.0
#define TUNE_ECHO_WET_MAX   0.5
#define TUNE_ECHO_WET_STEP  (TUNE_ECHO_WET_MAX - TUNE_ECHO_WET_MIN) / EFFECT_COUNT_STEP
#define TUNE_ECHO_DRY_MIN   0.5
#define TUNE_ECHO_DRY_MAX   0.9
#define TUNE_ECHO_DRY_STEP  (TUNE_ECHO_DRY_MAX - TUNE_ECHO_DRY_MIN) / EFFECT_COUNT_STEP

#define BEATBASS_FLANGE_DRY_MIN     0.3
#define BEATBASS_FLANGE_DRY_MAX     1.0
#define BEATBASS_FLANGE_DRY_STEP    (BEATBASS_FLANGE_DRY_MAX - BEATBASS_FLANGE_DRY_MIN) / EFFECT_COUNT_STEP
#define BEATBASS_FLANGE_WET_MIN     0.0
#define BEATBASS_FLANGE_WET_MAX     0.7
#define BEATBASS_FLANGE_WET_STEP    (BEATBASS_FLANGE_WET_MAX - BEATBASS_FLANGE_WET_MIN) / EFFECT_COUNT_STEP
#define BEATBASS_FLANGE_DEPTH_MIN   0.0
#define BEATBASS_FLANGE_DEPTH_MAX   0.5
#define BEATBASS_FLANGE_DEPTH_STEP  (BEATBASS_FLANGE_DEPTH_MAX - BEATBASS_FLANGE_DEPTH_MIN) / EFFECT_COUNT_STEP
#define BEATBASS_FLANGE_RATE_MIN    0.0 * FLANGE_RATE_REAL_MAX
#define BEATBASS_FLANGE_RATE_MAX    0.5 * FLANGE_RATE_REAL_MAX
#define BEATBASS_FLANGE_RATE_STEP   (BEATBASS_FLANGE_RATE_MAX - BEATBASS_FLANGE_RATE_MIN) / EFFECT_COUNT_STEP

#define BEATBASS_CHORUS_DRY_MIN     0.5
#define BEATBASS_CHORUS_DRY_MAX     1.0
#define BEATBASS_CHORUS_DRY_STEP   (BEATBASS_CHORUS_DRY_MAX - BEATBASS_CHORUS_DRY_MIN) / EFFECT_COUNT_STEP
#define BEATBASS_CHORUS_WET1_MIN    0.0
#define BEATBASS_CHORUS_WET1_MAX    0.7
#define BEATBASS_CHORUS_WET1_STEP   (BEATBASS_CHORUS_WET1_MAX - BEATBASS_CHORUS_WET1_MIN) / EFFECT_COUNT_STEP
#define BEATBASS_CHORUS_WET2_MIN    0.0
#define BEATBASS_CHORUS_WET2_MAX    0.7
#define BEATBASS_CHORUS_WET2_STEP   (BEATBASS_CHORUS_WET2_MAX - BEATBASS_CHORUS_WET2_MIN) / EFFECT_COUNT_STEP
#define BEATBASS_CHORUS_WET3        0.0
#define BEATBASS_CHORUS_DELAY_MIN   (0.2 * (CHORUS_DELAY_REAL_MAX - CHORUS_DELAY_REAL_MIN)) + CHORUS_DELAY_REAL_MIN
#define BEATBASS_CHORUS_DELAY_MAX   (0.5 * (CHORUS_DELAY_REAL_MAX - CHORUS_DELAY_REAL_MIN)) + CHORUS_DELAY_REAL_MIN
#define BEATBASS_CHORUS_DELAY_STEP   (BEATBASS_CHORUS_DELAY_MAX - BEATBASS_CHORUS_DELAY_MIN) / EFFECT_COUNT_STEP
#define BEATBASS_CHORUS_RATE_MIN    0.0 * CHORUS_RATE_REAL_MAX
#define BEATBASS_CHORUS_RATE_MAX    0.2 * CHORUS_RATE_REAL_MAX
#define BEATBASS_CHORUS_RATE_STEP   (BEATBASS_CHORUS_RATE_MAX - BEATBASS_CHORUS_RATE_MIN) / EFFECT_COUNT_STEP
#define BEATBASS_CHORUS_DEPTH       0.2
#define BEATBASS_CHORUS_FEEDBACK    0.0

#define BEATBASS_LOWPASS_CUTOFF_MIN       0.4
#define BEATBASS_LOWPASS_CUTOFF_MAX       1.0
#define BEATBASS_LOWPASS_CUTOFF_STEP      (BEATBASS_LOWPASS_CUTOFF_MAX - BEATBASS_LOWPASS_CUTOFF_MIN) / EFFECT_COUNT_STEP
#define BEATBASS_LOWPASS_RESONANCE_MIN   0.0
#define BEATBASS_LOWPASS_RESONANCE_MAX   0.14
#define BEATBASS_LOWPASS_RESONANCE_STEP  (BEATBASS_LOWPASS_RESONANCE_MAX - BEATBASS_LOWPASS_RESONANCE_MIN) / EFFECT_COUNT_STEP

#define BEATBASS_HIGHPASS_CUTOFF_MIN       (0.0 * (CUTOFF_REAL_MAX - CUTOFF_REAL_MIN)) + CUTOFF_REAL_MIN
#define BEATBASS_HIGHPASS_CUTOFF_MAX       (0.8 * (CUTOFF_REAL_MAX - CUTOFF_REAL_MIN)) + CUTOFF_REAL_MIN
#define BEATBASS_HIGHPASS_CUTOFF_STEP      (BEATBASS_HIGHPASS_CUTOFF_MAX - BEATBASS_HIGHPASS_CUTOFF_MIN) / EFFECT_COUNT_STEP
#define BEATBASS_HIGHPASS_RESONANCE_MIN    (0.0 * (RESONANCE_REAL_MAX - RESONANCE_REAL_MIN)) + RESONANCE_REAL_MIN
#define BEATBASS_HIGHPASS_RESONANCE_MAX    (0.14 * (RESONANCE_REAL_MAX - RESONANCE_REAL_MIN)) + RESONANCE_REAL_MIN
#define BEATBASS_HIGHPASS_RESONANCE_STEP  (BEATBASS_HIGHPASS_RESONANCE_MAX - BEATBASS_HIGHPASS_RESONANCE_MIN) / EFFECT_COUNT_STEP

typedef struct EffectSet EffectSet;
struct EffectSet {
  FMOD_DSP        *dspchorus;
  FMOD_DSP        *dspflange;
  FMOD_DSP        *dsphighpass;
  FMOD_DSP        *dspecho;
  FMOD_DSP        *dsplowpass;
};

typedef struct Track Track;
struct Track {
  FMOD_SOUND      *stream;
  float            volume;
  FMOD_CHANNEL    *channel;
  EffectSet        effects;
  FMOD_DSP_TYPE    activeEffect[6];
};

typedef struct Music Music;
struct Music {
  Track            tune;
  Track            beatbass;
  FMOD_DSP        *dspcompressor;
  int              marks[6];
  int              indexMark;
  int              downingEffects; // boolean active for the famous safe transition
  Uint32           downingCount;
  Uint32           downingStartTime;
  int              downingStep;
};

int cursors[5];

FMOD_SYSTEM *sound_system;
static Music music;


void ERRCHECK(FMOD_RESULT result)
{
  if (result != FMOD_OK)
  {
    printf("FMOD error! (%d) %s\n", result, FMOD_ErrorString(result));
    exit(-1);
  }
}

/*
FMOD_DSP_CHORUS_DRYMIX Volume of original signal to pass to output. 0.0 to 1.0. Default = 0.5.
FMOD_DSP_CHORUS_WETMIX1 Volume of 1st chorus tap. 0.0 to 1.0. Default = 0.5.
FMOD_DSP_CHORUS_WETMIX2 Volume of 2nd chorus tap. This tap is 90 degrees out of phase of the first tap. 0.0 to 1.0. Default = 0.5.
FMOD_DSP_CHORUS_WETMIX3 Volume of 3rd chorus tap. This tap is 90 degrees out of phase of the second tap. 0.0 to 1.0. Default = 0.5.
FMOD_DSP_CHORUS_DELAY Chorus delay in ms. 0.1 to 100.0. Default = 40.0 ms.
FMOD_DSP_CHORUS_RATE Chorus modulation rate in hz. 0.0 to 20.0. Default = 0.8 hz.
FMOD_DSP_CHORUS_DEPTH Chorus modulation depth. 0.0 to 1.0. Default = 0.03.
FMOD_DSP_CHORUS_FEEDBACK Chorus feedback. Controls how much of the wet signal gets fed back into the chorus buffer. 0.0 to 1.0. Default = 0.0. 
*/
void SetChrorus(FMOD_DSP *dsp, float drymix, float wetmix1, float wetmix2, float wetmix3, float delay, float rate, float depth, float feedback)
{
  FMOD_RESULT       result;
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_CHORUS_DRYMIX, drymix);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_CHORUS_WETMIX1, wetmix1);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_CHORUS_WETMIX2, wetmix2);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_CHORUS_WETMIX3, wetmix3);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_CHORUS_DELAY, delay);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_CHORUS_RATE, rate);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_CHORUS_DEPTH, depth);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_CHORUS_FEEDBACK, feedback);
  ERRCHECK(result);
}

/*
FMOD_DSP_COMPRESSOR_THRESHOLD Threshold level (dB) in the range from -60 through 0. The default value is 0.
FMOD_DSP_COMPRESSOR_ATTACK Gain reduction attack time (milliseconds), in the range from 10 through 200. The default value is 50.
FMOD_DSP_COMPRESSOR_RELEASE Gain reduction release time (milliseconds), in the range from 20 through 1000. The default value is 50.
FMOD_DSP_COMPRESSOR_GAINMAKEUP Make-up gain (dB) applied after limiting, in the range from 0 through 30. The default value is 0. 

threshold -0,1
attack 50
release 500
gain_makeup 0
*/
void SetCompressor(FMOD_DSP *dsp, int threshold, int attack, int release, int gain_makeup)
{
  FMOD_RESULT       result;
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_COMPRESSOR_THRESHOLD, threshold);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_COMPRESSOR_ATTACK, attack);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_COMPRESSOR_RELEASE, release);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_COMPRESSOR_GAINMAKEUP, gain_makeup);
  ERRCHECK(result);
}

/*
FMOD_DSP_ECHO_DELAY Echo delay in ms. 10 to 5000. Default = 500.
FMOD_DSP_ECHO_DECAYRATIO Echo decay per delay. 0 to 1. 1.0 = No decay, 0.0 = total decay (ie simple 1 line delay). Default = 0.5.
FMOD_DSP_ECHO_MAXCHANNELS Maximum channels supported. 0 to 16. 0 = same as fmod's default output polyphony, 1 = mono, 2 = stereo etc. See remarks for more. Default = 0. It is suggested to leave at 0!
FMOD_DSP_ECHO_DRYMIX Volume of original signal to pass to output. 0.0 to 1.0. Default = 1.0.
FMOD_DSP_ECHO_WETMIX Volume of echo signal to pass to output. 0.0 to 1.0. Default = 1.0. 
*/
void SetEcho(FMOD_DSP *dsp, int delay, float decay_ratio, float drymix, float wetmix)
{
  FMOD_RESULT       result;
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_ECHO_DELAY, delay);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_ECHO_DECAYRATIO, decay_ratio);
  ERRCHECK(result);
    
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_ECHO_DRYMIX, drymix);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_ECHO_WETMIX, wetmix);
  ERRCHECK(result);
}

/*
FMOD_DSP_FLANGE_DRYMIX Volume of original signal to pass to output. 0.0 to 1.0. Default = 0.45.
FMOD_DSP_FLANGE_WETMIX Volume of flange signal to pass to output. 0.0 to 1.0. Default = 0.55.
FMOD_DSP_FLANGE_DEPTH Flange depth. 0.01 to 1.0. Default = 1.0.
FMOD_DSP_FLANGE_RATE Flange speed in hz. 0.0 to 20.0. Default = 0.1. 

recommendation du SD : 1.0 dry 0.3
recommendation du SD : 0.0 wet 0.7
*/
void SetFlange(FMOD_DSP *dsp, float drymix, float wetmix, float depth, float rate)
{
  FMOD_RESULT       result;
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_FLANGE_DRYMIX, drymix);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_FLANGE_WETMIX, wetmix);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_FLANGE_DEPTH, depth);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_FLANGE_RATE, rate);
  ERRCHECK(result);
}

/*
FMOD_DSP_HIGHPASS_CUTOFF Highpass cutoff frequency in hz. 1.0 to output 22000.0. Default = 5000.0.
FMOD_DSP_HIGHPASS_RESONANCE Highpass resonance Q value. 1.0 to 10.0. Default = 1.0. 
*/
void SetHighpass(FMOD_DSP *dsp, float cutoff, float resonance)
{
  FMOD_RESULT       result;
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_HIGHPASS_CUTOFF, cutoff);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_HIGHPASS_RESONANCE, resonance);
  ERRCHECK(result);
}

/*
FMOD_DSP_LOWPASS_CUTOFF Lowpass cutoff frequency in hz. 10.0 to 22000.0. Default = 5000.0.
FMOD_DSP_LOWPASS_RESONANCE Lowpass resonance Q value. 1.0 to 10.0. Default = 1.0. 
*/
void SetLowpass(FMOD_DSP *dsp, float cutoff, float resonance)
{
  FMOD_RESULT       result;
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_LOWPASS_CUTOFF, cutoff);
  ERRCHECK(result);
  
  result = FMOD_DSP_SetParameter(dsp, FMOD_DSP_LOWPASS_RESONANCE, resonance);
  ERRCHECK(result);
}

void UpDspTrack(Track *track, FMOD_DSP *dsp)
{
  int active;
  FMOD_RESULT result;

  result = FMOD_DSP_GetActive(dsp, &active);
  ERRCHECK(result);

  if (!active)
  {
    result = FMOD_Channel_AddDSP(track->channel, dsp, 0);
    ERRCHECK(result);
  }
}

void DownDsp(FMOD_DSP *dsp)
{
  int active;
  FMOD_RESULT result;

  result = FMOD_DSP_GetActive(dsp, &active);
  ERRCHECK(result);

  if (active)
  {
    result = FMOD_DSP_Remove(dsp);
    ERRCHECK(result);
  }
}

void UpVolumeTrack(Track *track)
{
  if (track->volume < 1)
  {
    track->volume  ++;
    FMOD_Channel_SetVolume(track->channel, track->volume);
  }
}

void DownTrack(Track *track)
{
  if (track->volume > 0)
  {
    track->volume  --;
    FMOD_Channel_SetVolume(track->channel, track->volume);
  }
}

void SetPanTrack(Track track, float pan)
{
  FMOD_RESULT result;

  result = FMOD_Channel_SetPan(track.channel, pan);
  ERRCHECK(result);
}

void ToLeftTrack(Track track)
{
  FMOD_RESULT result;
  float pan;
  
  result = FMOD_Channel_GetPan(track.channel, &pan);
  ERRCHECK(result);
  
  float newPan = pan - STEP_PAN;
  
  if (newPan >= -1.0)
  { 
    result = FMOD_Channel_SetPan(track.channel, newPan);
    ERRCHECK(result);
  }
}

void ToRightTrack(Track track)
{
  FMOD_RESULT result;
  float pan;
  
  result = FMOD_Channel_GetPan(track.channel, &pan);
  ERRCHECK(result);
  
  float newPan = pan + STEP_PAN;
  
  if (newPan <= 1.0)
  { 
    result = FMOD_Channel_SetPan(track.channel, newPan);
    ERRCHECK(result);
  }
}

/// UP & DOWN EFFECT ///
void SetParameter(FMOD_DSP *dsp, FMOD_DSP_TYPE type, float value, float min, float max)
{
  FMOD_RESULT       result;
  char valuestr[16];
  int valuestrlen = 16;
  
  float newValue = (value * (max - min)) + min;
        
  result = FMOD_DSP_SetParameter(dsp, type, newValue);
  ERRCHECK(result);
  
  result = FMOD_DSP_GetParameter(dsp, type, &value, valuestr, valuestrlen);
  ERRCHECK(result);
  //printf(">>> set get param: %f %s %f \n", value, valuestr, min);
}

void TryDownParameter(FMOD_DSP *dsp, FMOD_DSP_TYPE type, float step, float min)
{
  FMOD_RESULT       result;
  char valuestr[16];
  float value;
  int valuestrlen = 16;
  
  result = FMOD_DSP_GetParameter(dsp, type, &value, valuestr, valuestrlen);
  ERRCHECK(result);
  //printf("down get param: %f %s %f %f \n", value, valuestr, step, min);
  
  float newValue = value - step;
        
  if (newValue >= min)
  {
    result = FMOD_DSP_SetParameter(dsp, type, newValue);
    ERRCHECK(result);
  }
  
  result = FMOD_DSP_GetParameter(dsp, type, &value, valuestr, valuestrlen);
  ERRCHECK(result);
  //printf(">>> down get param: %f %s %f %f \n", value, valuestr, step, min);
}

void TryUpParameter(FMOD_DSP *dsp, FMOD_DSP_TYPE type, float step, float max)
{
  FMOD_RESULT       result;
  char valuestr[16];
  float value;
  int valuestrlen = 16;
  
  result = FMOD_DSP_GetParameter(dsp, type, &value, valuestr, valuestrlen);
  ERRCHECK(result);
  
  //printf("up get param: %f %s %f %f \n", value, valuestr, step, max);
  
  float newValue = value + step;
  if (newValue <= max)
  {
    result = FMOD_DSP_SetParameter(dsp, type, newValue);
    ERRCHECK(result);
  }
  
  result = FMOD_DSP_GetParameter(dsp, type, &value, valuestr, valuestrlen);
  ERRCHECK(result);
  
  //printf(">>> up get param: %f %s %f %f \n", value, valuestr, step, max);
}

/// BEATBASS ///
void DownTirdthParamBeatBass(Track track, int indexMark)
{
  FMOD_DSP_TYPE type = track.activeEffect[indexMark];
  switch(type)
  {
    case FMOD_DSP_TYPE_CHORUS:
      TryDownParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_RATE, BEATBASS_CHORUS_RATE_STEP, BEATBASS_CHORUS_RATE_MIN);
      break;
    case FMOD_DSP_TYPE_FLANGE:
      TryDownParameter(track.effects.dspflange, FMOD_DSP_FLANGE_RATE, BEATBASS_FLANGE_RATE_STEP, BEATBASS_FLANGE_RATE_MIN);
      break;
    default:
      //printf("effect '%d' not supported\n", type);
      break;
  }
  if (cursors[2] - 1 >= 0)
  {
    cursors[2]--;
    on_cursor_effect_event(2, cursors[2]);
  }
}

void UpTirdthParamBeatBass(Track track, int indexMark)
{
  FMOD_DSP_TYPE type = track.activeEffect[indexMark];
  switch(type)
  {
    case FMOD_DSP_TYPE_CHORUS:
      TryUpParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_RATE, BEATBASS_CHORUS_RATE_STEP, BEATBASS_CHORUS_RATE_MAX);
      break;
    case FMOD_DSP_TYPE_FLANGE:
      TryUpParameter(track.effects.dspflange, FMOD_DSP_FLANGE_RATE, BEATBASS_FLANGE_RATE_STEP, BEATBASS_FLANGE_RATE_MAX);
      break;
    default:
      //printf("effect '%d' not supported\n", type);
      break;
  }
  if (cursors[2] + 1 <= 10)
  {
    cursors[2]++;
    on_cursor_effect_event(2, cursors[2]);
  }
}

void DownSecondParamBeatBass(Track track, int indexMark)
{
  FMOD_DSP_TYPE type = track.activeEffect[indexMark];
  switch(type)
  {
    case FMOD_DSP_TYPE_CHORUS:
      TryUpParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_DELAY, BEATBASS_CHORUS_DELAY_STEP, BEATBASS_CHORUS_DELAY_MAX);
      break;
    case FMOD_DSP_TYPE_FLANGE:
      TryDownParameter(track.effects.dspflange, FMOD_DSP_FLANGE_DEPTH, BEATBASS_FLANGE_DEPTH_STEP, BEATBASS_FLANGE_DEPTH_MIN);
      break;
    case FMOD_DSP_TYPE_HIGHPASS:
      TryDownParameter(track.effects.dsphighpass, FMOD_DSP_HIGHPASS_RESONANCE, BEATBASS_HIGHPASS_RESONANCE_STEP, BEATBASS_HIGHPASS_RESONANCE_MIN);
      break;
    case FMOD_DSP_TYPE_LOWPASS:
      TryDownParameter(track.effects.dsphighpass, FMOD_DSP_LOWPASS_RESONANCE, BEATBASS_LOWPASS_RESONANCE_STEP, BEATBASS_LOWPASS_RESONANCE_MIN);
      break;
    default:
      //printf("effect '%d' not supported\n", type);
      break;
  }
  
  if (cursors[1] - 1 >= 0)
  {
    cursors[1]--;
    on_cursor_effect_event(1, cursors[1]);
  }
}

void UpSecondParamBeatBass(Track track, int indexMark)
{
  FMOD_DSP_TYPE type = track.activeEffect[indexMark];
  switch(type)
  {
    case FMOD_DSP_TYPE_CHORUS:
      TryDownParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_DELAY, BEATBASS_CHORUS_DELAY_STEP, BEATBASS_CHORUS_DELAY_MIN);
      break;
    case FMOD_DSP_TYPE_FLANGE:
      TryUpParameter(track.effects.dspflange, FMOD_DSP_FLANGE_DEPTH, BEATBASS_FLANGE_DEPTH_STEP, BEATBASS_FLANGE_DEPTH_MAX);
      break;
    case FMOD_DSP_TYPE_HIGHPASS:
      TryUpParameter(track.effects.dsphighpass, FMOD_DSP_HIGHPASS_RESONANCE, BEATBASS_HIGHPASS_RESONANCE_STEP, BEATBASS_HIGHPASS_RESONANCE_MAX);
      break;
    case FMOD_DSP_TYPE_LOWPASS:
      TryUpParameter(track.effects.dsphighpass, FMOD_DSP_LOWPASS_RESONANCE, BEATBASS_LOWPASS_RESONANCE_STEP, BEATBASS_LOWPASS_RESONANCE_MAX);
      break;
    default:
      //printf("effect '%d' not supported\n", type);
      break;
  }
  
  if (cursors[1] + 1 <= 10)
  {
    cursors[1]++;
    on_cursor_effect_event(1, cursors[1]);
  }
}


void DownFirstParamBeatBass(Track track, int indexMark)
{
  FMOD_DSP_TYPE type = track.activeEffect[indexMark];
  switch(type)
  {
    case FMOD_DSP_TYPE_CHORUS:
      TryUpParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_DRYMIX, BEATBASS_CHORUS_DRY_STEP, BEATBASS_CHORUS_DRY_MAX);
      TryDownParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_WETMIX1, BEATBASS_CHORUS_WET1_STEP, BEATBASS_CHORUS_WET1_MIN);
      TryDownParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_WETMIX2, BEATBASS_CHORUS_WET2_STEP, BEATBASS_CHORUS_WET2_MIN);
      break;
    case FMOD_DSP_TYPE_FLANGE:
      TryUpParameter(track.effects.dspflange, FMOD_DSP_FLANGE_DRYMIX, BEATBASS_FLANGE_DRY_STEP, BEATBASS_FLANGE_DRY_MAX);
      TryDownParameter(track.effects.dspflange, FMOD_DSP_FLANGE_WETMIX, BEATBASS_FLANGE_WET_STEP, BEATBASS_FLANGE_WET_MIN);
      break;
    case FMOD_DSP_TYPE_HIGHPASS:
      TryDownParameter(track.effects.dsphighpass, FMOD_DSP_HIGHPASS_CUTOFF, BEATBASS_HIGHPASS_CUTOFF_STEP, BEATBASS_HIGHPASS_CUTOFF_MIN);
      break;
    case FMOD_DSP_TYPE_LOWPASS:
      TryDownParameter(track.effects.dsphighpass, FMOD_DSP_LOWPASS_CUTOFF, BEATBASS_LOWPASS_CUTOFF_STEP, BEATBASS_LOWPASS_CUTOFF_MIN);
      break;
    default:
      //printf("effect '%d' not supported\n", type);
      break;
  }
  
  if (cursors[0] - 1 >= 0)
  {
    cursors[0]--;
    on_cursor_effect_event(0, cursors[0]);
  }
}

void UpFirstParamBeatBass(Track track, int indexMark)
{
  FMOD_DSP_TYPE type = track.activeEffect[indexMark];
  switch(type)
  {
    case FMOD_DSP_TYPE_CHORUS:
      TryDownParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_DRYMIX, BEATBASS_CHORUS_DRY_STEP, BEATBASS_CHORUS_DRY_MIN);
      TryUpParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_WETMIX1, BEATBASS_CHORUS_WET1_STEP, BEATBASS_CHORUS_WET1_MAX);
      TryUpParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_WETMIX2, BEATBASS_CHORUS_WET2_STEP, BEATBASS_CHORUS_WET2_MAX);
      break;
    case FMOD_DSP_TYPE_FLANGE:
      TryDownParameter(track.effects.dspflange, FMOD_DSP_FLANGE_DRYMIX, BEATBASS_FLANGE_DRY_STEP, BEATBASS_FLANGE_DRY_MIN);
      TryUpParameter(track.effects.dspflange, FMOD_DSP_FLANGE_WETMIX, BEATBASS_FLANGE_WET_STEP, BEATBASS_FLANGE_WET_MAX);
      break;
    case FMOD_DSP_TYPE_HIGHPASS:
      TryUpParameter(track.effects.dsphighpass, FMOD_DSP_HIGHPASS_CUTOFF, BEATBASS_HIGHPASS_CUTOFF_STEP, BEATBASS_HIGHPASS_CUTOFF_MAX);
      break;
    case FMOD_DSP_TYPE_LOWPASS:
      TryUpParameter(track.effects.dsphighpass, FMOD_DSP_LOWPASS_CUTOFF, BEATBASS_LOWPASS_CUTOFF_STEP, BEATBASS_LOWPASS_CUTOFF_MAX);
      break;
    default:
      //printf("effect '%d' not supported\n", type);
      break;
  }
  if (cursors[0] + 1 <= 10)
  {
    cursors[0]++;
    on_cursor_effect_event(0, cursors[0]);
  }
}


/// TUNE ///
void DownTirdthParamTune(Track track, int indexMark)
{
  FMOD_DSP_TYPE type = track.activeEffect[indexMark];
  switch(type)
  {
    case FMOD_DSP_TYPE_FLANGE:
      TryDownParameter(track.effects.dspflange, FMOD_DSP_FLANGE_RATE, TUNE_FLANGE_RATE_STEP, TUNE_FLANGE_RATE_MIN);
      break;
    default:
      //printf("effect '%d' not supported\n", type);
      break;
  }
  
  if (cursors[5] - 1 >= 0)
  {
    cursors[5]--;
    on_cursor_effect_event(5, cursors[5]);
  }
}

void UpTirdthParamTune(Track track, int indexMark)
{
  FMOD_DSP_TYPE type = track.activeEffect[indexMark];
  switch(type)
  {
    case FMOD_DSP_TYPE_FLANGE:
      TryUpParameter(track.effects.dspflange, FMOD_DSP_FLANGE_RATE, TUNE_FLANGE_RATE_STEP, TUNE_FLANGE_RATE_MAX);
      break;
    default:
      //printf("effect '%d' not supported\n", type);
      break;
  }
  if (cursors[5] + 1 <= 10)
  {
    cursors[5]++;
    on_cursor_effect_event(5, cursors[5]);
  }
}

void DownSecondParamTune(Track track, int indexMark)
{
  FMOD_DSP_TYPE type = track.activeEffect[indexMark];
  switch(type)
  {
    case FMOD_DSP_TYPE_FLANGE:
      TryDownParameter(track.effects.dspflange, FMOD_DSP_FLANGE_DEPTH, TUNE_FLANGE_DEPTH_STEP, TUNE_FLANGE_DEPTH_MIN);
      break;
    case FMOD_DSP_TYPE_HIGHPASS:
      TryDownParameter(track.effects.dsphighpass, FMOD_DSP_HIGHPASS_RESONANCE, TUNE_HIGHPASS_RESONANCE_STEP, TUNE_HIGHPASS_RESONANCE_MIN);
      break;
    case FMOD_DSP_TYPE_ECHO:
      TryDownParameter(track.effects.dspecho, FMOD_DSP_ECHO_DECAYRATIO, TUNE_ECHO_DELAY, TUNE_ECHO_DECAY_MIN);
      break;
    case FMOD_DSP_TYPE_LOWPASS:
      TryDownParameter(track.effects.dsphighpass, FMOD_DSP_LOWPASS_RESONANCE, TUNE_LOWPASS_RESONANCE_STEP, TUNE_LOWPASS_RESONANCE_MIN);
      break;
    default:
      //printf("effect '%d' not supported\n", type);
      break;
  }
  
  if (cursors[4] - 1 >= 0)
  {
    cursors[4]--;
    on_cursor_effect_event(4, cursors[4]);
  }
}

void UpSecondParamTune(Track track, int indexMark)
{
  FMOD_DSP_TYPE type = track.activeEffect[indexMark];
  switch(type)
  {
    case FMOD_DSP_TYPE_FLANGE:
      TryUpParameter(track.effects.dspflange, FMOD_DSP_FLANGE_DEPTH, TUNE_FLANGE_DEPTH_STEP, TUNE_FLANGE_DEPTH_MAX);
      break;
    case FMOD_DSP_TYPE_HIGHPASS:
      TryUpParameter(track.effects.dsphighpass, FMOD_DSP_HIGHPASS_RESONANCE, TUNE_HIGHPASS_RESONANCE_STEP, TUNE_HIGHPASS_RESONANCE_MAX);
      break;
    case FMOD_DSP_TYPE_ECHO:
      TryUpParameter(track.effects.dspecho, FMOD_DSP_ECHO_DECAYRATIO, TUNE_ECHO_DELAY, TUNE_ECHO_DECAY_MAX);
      break;
    case FMOD_DSP_TYPE_LOWPASS:
      TryUpParameter(track.effects.dsphighpass, FMOD_DSP_LOWPASS_RESONANCE, TUNE_LOWPASS_RESONANCE_STEP, TUNE_LOWPASS_RESONANCE_MAX);
      break;
    default:
      //printf("effect '%d' not supported\n", type);
      break;
  }
  
  if (cursors[4] + 1 <= 10)
  {
    cursors[4]++;
    on_cursor_effect_event(4, cursors[4]);
  }
}


void DownFirstParamTune(Track track, int indexMark)
{
  FMOD_DSP_TYPE type = track.activeEffect[indexMark];
  switch(type)
  {
    case FMOD_DSP_TYPE_CHORUS:
      TryUpParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_DRYMIX, TUNE_CHORUS_DRY_STEP, TUNE_CHORUS_DRY_MAX);
      TryDownParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_WETMIX1, TUNE_CHORUS_WET1_STEP, TUNE_CHORUS_WET1_MIN);
      TryDownParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_WETMIX2, TUNE_CHORUS_WET2_STEP, TUNE_CHORUS_WET2_MIN);
      break;
    case FMOD_DSP_TYPE_FLANGE:
      TryUpParameter(track.effects.dspflange, FMOD_DSP_FLANGE_DRYMIX, TUNE_FLANGE_DRY_STEP, TUNE_FLANGE_DRY_MAX);
      TryDownParameter(track.effects.dspflange, FMOD_DSP_FLANGE_WETMIX, TUNE_FLANGE_WET_STEP, TUNE_FLANGE_WET_MIN);
      break;
    case FMOD_DSP_TYPE_HIGHPASS:
      TryDownParameter(track.effects.dsphighpass, FMOD_DSP_HIGHPASS_CUTOFF, TUNE_HIGHPASS_CUTOFF_STEP, TUNE_HIGHPASS_CUTOFF_MIN);
      break;
    case FMOD_DSP_TYPE_ECHO:
      TryUpParameter(track.effects.dspecho, FMOD_DSP_ECHO_DRYMIX, TUNE_ECHO_DRY_STEP, TUNE_ECHO_DRY_MAX);
      TryDownParameter(track.effects.dspecho, FMOD_DSP_ECHO_WETMIX, TUNE_ECHO_WET_STEP, TUNE_ECHO_WET_MIN);
      break;
    case FMOD_DSP_TYPE_LOWPASS:
      TryDownParameter(track.effects.dsphighpass, FMOD_DSP_LOWPASS_CUTOFF, TUNE_LOWPASS_CUTOFF_STEP, TUNE_LOWPASS_CUTOFF_MIN);
      break;
    default:
      //printf("effect '%d' not supported\n", type);
      break;
  }
  
  if (cursors[3] - 1 >= 0)
  {
    cursors[3]--;
    on_cursor_effect_event(3, cursors[3]);
  }
}

void UpFirstParamTune(Track track, int indexMark)
{
    FMOD_DSP_TYPE type = track.activeEffect[indexMark];
    switch(type)
    {
      case FMOD_DSP_TYPE_CHORUS:
        TryDownParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_DRYMIX, TUNE_CHORUS_DRY_STEP, TUNE_CHORUS_DRY_MIN);
        TryUpParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_WETMIX1, TUNE_CHORUS_WET1_STEP, TUNE_CHORUS_WET1_MAX);
        TryUpParameter(track.effects.dspchorus, FMOD_DSP_CHORUS_WETMIX2, TUNE_CHORUS_WET2_STEP, TUNE_CHORUS_WET2_MAX);
        break;
      case FMOD_DSP_TYPE_FLANGE:
        TryDownParameter(track.effects.dspflange, FMOD_DSP_FLANGE_DRYMIX, TUNE_FLANGE_DRY_STEP, TUNE_FLANGE_DRY_MIN);
        TryUpParameter(track.effects.dspflange, FMOD_DSP_FLANGE_WETMIX, TUNE_FLANGE_WET_STEP, TUNE_FLANGE_WET_MAX);
        break;
      case FMOD_DSP_TYPE_HIGHPASS:
        TryUpParameter(track.effects.dsphighpass, FMOD_DSP_HIGHPASS_CUTOFF, TUNE_HIGHPASS_CUTOFF_STEP, TUNE_HIGHPASS_CUTOFF_MAX);
        break;
      case FMOD_DSP_TYPE_ECHO:
        TryDownParameter(track.effects.dspecho, FMOD_DSP_ECHO_DRYMIX, TUNE_ECHO_DRY_STEP, TUNE_ECHO_DRY_MIN);
        TryUpParameter(track.effects.dspecho, FMOD_DSP_ECHO_WETMIX, TUNE_ECHO_WET_STEP, TUNE_ECHO_WET_MAX);
        break;
      case FMOD_DSP_TYPE_LOWPASS:
        TryUpParameter(track.effects.dsphighpass, FMOD_DSP_LOWPASS_CUTOFF, TUNE_LOWPASS_CUTOFF_STEP, TUNE_LOWPASS_CUTOFF_MAX);
        break;
      default:
        //printf("effect '%d' not supported\n", type);
        break;
    }
    
    if (cursors[3] + 1 <= 10)
    {
      cursors[3]++;
      on_cursor_effect_event(3, cursors[3]);
    }
}

void backTrack(Track track)
{
  FMOD_RESULT result;
  unsigned int position;
  
  result = FMOD_Channel_GetPosition(track.channel, &position, FMOD_TIMEUNIT_MS);
  ERRCHECK(result);
  
  unsigned int newPosition = position - STEP_ADVANCE;
  if (newPosition >= 0)
  {
    result = FMOD_Channel_SetPosition(track.channel, newPosition, FMOD_TIMEUNIT_MS);
    ERRCHECK(result);
  }
}

void backMusic(Music *music)
{
  backTrack(music->beatbass);
  backTrack(music->tune);
}

void advanceTrack(Track track)
{
  FMOD_RESULT result;
  unsigned int position;
  
  result = FMOD_Channel_GetPosition(track.channel, &position, FMOD_TIMEUNIT_MS);
  ERRCHECK(result);
  
  unsigned int length;
  result =  FMOD_Sound_GetLength(track.stream, &length, FMOD_TIMEUNIT_MS);
  
  unsigned int newPosition = position + STEP_ADVANCE;
  if (newPosition <= length)
  {
    result = FMOD_Channel_SetPosition(track.channel, newPosition, FMOD_TIMEUNIT_MS);
    ERRCHECK(result);
  }
}

void advanceMusic(Music *music)
{
  advanceTrack(music->beatbass);
  advanceTrack(music->tune);
}

void KeyDown (SDL_KeyboardEvent *key, Music *music, FMOD_SYSTEM *system)
{
  switch(key->keysym.sym)
  {
    case SDLK_a:
      ToLeftTrack(music->tune);
      break;
      
    case SDLK_z:
      ToRightTrack(music->tune);
      break;
      
    case SDLK_q:
      UpFirstParamTune(music->tune, music->indexMark);
      break;
      
    case SDLK_s:
      UpSecondParamTune(music->tune, music->indexMark);
      break;
      
    case SDLK_d:
      UpTirdthParamTune(music->tune, music->indexMark);
      break;
      
    case SDLK_w:
      DownFirstParamTune(music->tune, music->indexMark);
      break;
      
    case SDLK_x:
      DownSecondParamTune(music->tune, music->indexMark);
      break;
      
    case SDLK_c:
      DownTirdthParamTune(music->tune, music->indexMark);
      break;
      
    case SDLK_i:
      ToRightTrack(music->beatbass);
      break;
      
    case SDLK_u:
      ToLeftTrack(music->beatbass);
      break;
      
    case SDLK_g:
      UpFirstParamBeatBass(music->beatbass, music->indexMark);
      break;
      
    case SDLK_h:
      UpSecondParamBeatBass(music->beatbass, music->indexMark);
      break;
      
    case SDLK_j:
      UpTirdthParamBeatBass(music->beatbass, music->indexMark);
      break;
      
    case SDLK_v:
      DownFirstParamBeatBass(music->beatbass, music->indexMark);
      break;
      
    case SDLK_b:
      DownSecondParamBeatBass(music->beatbass, music->indexMark);
      break;
      
    case SDLK_n:
      DownTirdthParamBeatBass(music->beatbass, music->indexMark);
      break;
      
    case SDLK_RIGHT:
      advanceMusic(music);
      break;
      
    case SDLK_LEFT:
      backMusic(music);
      break;
    
    default:
      break;
  }
}

/**
 * EnterFrame when enter to a new frame
**/
int EnterFrame(FMOD_SYSTEM *system, Music *music)
{
  int running = 1;

  if (music->downingEffects == TRUE)
  {
    Uint32 count = ((SDL_GetTicks() - music->downingStartTime) / music->downingStep) - music->downingCount;
    Uint32 i;
    if (count > 0)
    {
     for (i = 0; i < count; i++)
     {
      DownFirstParamTune(music->tune, music->indexMark);
      DownSecondParamTune(music->tune, music->indexMark);
      DownTirdthParamTune(music->tune, music->indexMark);
      
      DownFirstParamBeatBass(music->beatbass, music->indexMark);
      DownSecondParamBeatBass(music->beatbass, music->indexMark);
      DownTirdthParamBeatBass(music->beatbass, music->indexMark);
      
       music->downingCount++;
     }
    }
  }
  
  FMOD_System_Update(system);
  return running;
}

void sound_key_down(SDL_Event *event)
{
  if (music.downingEffects == FALSE)
  {
    KeyDown(&event->key, &music, sound_system);
  }
}

float last_pass = 0;

void update_music()
{
  if (music.downingEffects == FALSE)
  {
    float now = SDL_GetTicks();
    if (now > last_pass + MIN_DELAY_CHANGE)
    {
      last_pass = now;
      /// tune effect
      if (wiimote1->gforce.z > 0)
      {
        UpFirstParamTune(music.tune, music.indexMark);
      }
      
      if (wiimote1->gforce.z < 0)
      {
        DownFirstParamTune(music.tune, music.indexMark);
      }
      
      if (wiimote1->gforce.x > 0)
      {
        UpSecondParamTune(music.tune, music.indexMark);
      }
      
      if (wiimote1->gforce.x < 0)
      {
        DownSecondParamTune(music.tune, music.indexMark);
      }
      
      if (wiimote1->gforce.y > 0)
      {
        UpTirdthParamTune(music.tune, music.indexMark);
      }
      
      if (wiimote1->gforce.y < 0)
      {
        DownTirdthParamTune(music.tune, music.indexMark);
      }
      
      /// beatbass effect
      if (wiimote1->exp.nunchuk.gforce.z > 0)
      {
        UpFirstParamBeatBass(music.beatbass, music.indexMark);
      }
      
      if (wiimote1->exp.nunchuk.gforce.z < 0)
      {
        DownFirstParamBeatBass(music.beatbass, music.indexMark);
      }
      
      if (wiimote1->exp.nunchuk.gforce.x > 0)
      {
        UpSecondParamBeatBass(music.beatbass, music.indexMark);
      }
      if (wiimote1->exp.nunchuk.gforce.x < 0)
      {
        DownSecondParamBeatBass(music.beatbass, music.indexMark);
      }
      
      if (wiimote1->exp.nunchuk.gforce.y > 0)
      {
        UpTirdthParamBeatBass(music.beatbass, music.indexMark);
      }
      
      if (wiimote1->exp.nunchuk.gforce.y < 0)
      {
        DownTirdthParamBeatBass(music.beatbass, music.indexMark);
      }
      
      /// pan
      if (isfinite(wiimote1->exp.nunchuk.js.ang))
      {
        float numshockx = wiimote1->exp.nunchuk.js.mag * sin(wiimote1->exp.nunchuk.js.ang / 180.0 * M_PI);
        SetPanTrack(music.beatbass, numshockx * 0.8);
      }
    }
  }
}

void initEffectBeatBassOfMark(Track *track, int indexMark)
{
  FMOD_DSP_TYPE type = track->activeEffect[indexMark];
  
  switch(type)
  {
    case FMOD_DSP_TYPE_CHORUS:
      UpDspTrack(track, track->effects.dspchorus);
      SetChrorus(track->effects.dspchorus, BEATBASS_CHORUS_DRY_MAX, BEATBASS_CHORUS_WET1_MIN, BEATBASS_CHORUS_WET2_MIN, BEATBASS_CHORUS_WET3, BEATBASS_CHORUS_DELAY_MAX, BEATBASS_CHORUS_RATE_MIN, BEATBASS_CHORUS_DEPTH, BEATBASS_CHORUS_FEEDBACK);
      break;
    case FMOD_DSP_TYPE_FLANGE:
      UpDspTrack(track, track->effects.dspflange);
      SetFlange(track->effects.dspflange, BEATBASS_FLANGE_DRY_MAX, BEATBASS_FLANGE_WET_MIN, BEATBASS_FLANGE_DEPTH_MIN, BEATBASS_FLANGE_RATE_MIN);
      break;
    case FMOD_DSP_TYPE_HIGHPASS:
      UpDspTrack(track, track->effects.dsphighpass);
      SetHighpass(track->effects.dsphighpass, BEATBASS_HIGHPASS_CUTOFF_MAX, BEATBASS_HIGHPASS_RESONANCE_MIN);
      break;
    case FMOD_DSP_TYPE_LOWPASS:
      UpDspTrack(track, track->effects.dsplowpass);
      SetLowpass(track->effects.dsphighpass, BEATBASS_LOWPASS_CUTOFF_MAX, BEATBASS_LOWPASS_RESONANCE_MIN);
      break;
    default:
      //printf("effect '%d' not supported\n", type);
      break;
  }
  
  cursors[0] = 0;
  cursors[1] = 0;
  cursors[2] = 0;
  
  on_cursor_effect_event(0, 0);
  on_cursor_effect_event(1, 0);
  on_cursor_effect_event(2, 0);
}

void initEffectTuneOfMark(Track *track, int indexMark)
{
  FMOD_DSP_TYPE type = track->activeEffect[indexMark];
  
  switch(type)
  {
    case FMOD_DSP_TYPE_CHORUS:
      UpDspTrack(track, track->effects.dspchorus);
      SetChrorus(track->effects.dspchorus, TUNE_CHORUS_DRY_MAX, TUNE_CHORUS_WET1_MIN, TUNE_CHORUS_WET2_MIN, TUNE_CHORUS_WET3, TUNE_CHORUS_DELAY, TUNE_CHORUS_RATE, TUNE_CHORUS_DEPTH, TUNE_CHORUS_FEEDBACK);
      break;
    case FMOD_DSP_TYPE_FLANGE:
      UpDspTrack(track, track->effects.dspflange);
      SetFlange(track->effects.dspflange, TUNE_FLANGE_DRY_MAX, TUNE_FLANGE_WET_MIN, TUNE_FLANGE_DEPTH_MIN, TUNE_FLANGE_RATE_MIN);
      break;
    case FMOD_DSP_TYPE_HIGHPASS:
      UpDspTrack(track, track->effects.dsphighpass);
      SetHighpass(track->effects.dsphighpass, TUNE_HIGHPASS_CUTOFF_MAX, TUNE_HIGHPASS_RESONANCE_MIN);
      break;
    case FMOD_DSP_TYPE_ECHO:
      UpDspTrack(track, track->effects.dspecho);
      SetEcho(track->effects.dspecho, TUNE_ECHO_DELAY, TUNE_ECHO_DECAY_MIN, TUNE_ECHO_WET_MIN, TUNE_ECHO_DRY_MAX);
      break;
    case FMOD_DSP_TYPE_LOWPASS:
      UpDspTrack(track, track->effects.dsplowpass);
      SetLowpass(track->effects.dsphighpass, TUNE_LOWPASS_CUTOFF_MAX, TUNE_LOWPASS_RESONANCE_MIN);
      break;
    default:
      //printf("effect '%d' not supported\n", type);
    break;
  }
  
  cursors[3] = 0;
  cursors[4] = 0;
  cursors[5] = 0;
  on_cursor_effect_event(3, 0);
  on_cursor_effect_event(4, 0);
  on_cursor_effect_event(5, 0);
}

void disableAllEffects(EffectSet *effects)
{
  DownDsp(effects->dspchorus);
  DownDsp(effects->dspflange);
  DownDsp(effects->dsphighpass);
  DownDsp(effects->dspecho);
  DownDsp(effects->dsplowpass);
}

FMOD_RESULT F_CALLBACK OnSync( FMOD_CHANNEL *  channel, FMOD_CHANNEL_CALLBACKTYPE type, int command, unsigned int commanddata1, unsigned int commanddata2)
{
  Music *music= (Music*) command;
  unsigned int numsync = commanddata1;
  
  //printf("OnSync %d\n", numsync);
  
  unsigned int mark = numsync + 1;
  unsigned int nextMark = mark + 1;
  int new_part = FALSE;
  int just_before_new_part = FALSE;
  
  // on new mark
  if (mark == music->marks[music->indexMark + 1])
  {
    printf("new part\n");
    music->downingEffects = FALSE;
    music->downingCount = 0;
    disableAllEffects(&music->beatbass.effects);
    disableAllEffects(&music->tune.effects);
    
    music->indexMark ++;
    
    initEffectTuneOfMark(&music->tune, music->indexMark);
    initEffectBeatBassOfMark(&music->beatbass, music->indexMark);
    new_part = TRUE;
  }
  
  // last sync of mark
  if (nextMark == music->marks[music->indexMark + 1])
  {
    printf("just_before_new_part\n");
    music->downingEffects = TRUE;
    music->downingCount = 0;
    music->downingStartTime = SDL_GetTicks();
    just_before_new_part = TRUE;
  }
  
  // process event in main.c
  on_music_event(mark, new_part, just_before_new_part);
  
  return FMOD_OK;
}

void initEffectSet(FMOD_SYSTEM *system, EffectSet *effects)
{
  FMOD_RESULT result;
  
  result = FMOD_System_CreateDSPByType(system, FMOD_DSP_TYPE_CHORUS, &effects->dspchorus);
  ERRCHECK(result);
  
  result = FMOD_System_CreateDSPByType(system, FMOD_DSP_TYPE_FLANGE, &effects->dspflange);
  ERRCHECK(result);
  
  result = FMOD_System_CreateDSPByType(system, FMOD_DSP_TYPE_HIGHPASS, &effects->dsphighpass);
  ERRCHECK(result);
  
  result = FMOD_System_CreateDSPByType(system, FMOD_DSP_TYPE_ECHO, &effects->dspecho);
  ERRCHECK(result);
  
  result = FMOD_System_CreateDSPByType(system, FMOD_DSP_TYPE_LOWPASS, &effects->dsplowpass);
  ERRCHECK(result);
}

void initTrack(FMOD_SYSTEM *system, Track *track, char *filename)
{
  FMOD_RESULT result;
  
  //FMOD_SOUND      *stream;
  result = FMOD_System_CreateStream(system, filename, FMOD_HARDWARE | FMOD_LOOP_OFF | FMOD_2D, 0, &track->stream );
  ERRCHECK(result);
  printf("... track loaded\n");
  
  //float            volume;
  track->volume = 1;
  
  //FMOD_CHANNEL    *channel;
  result = FMOD_System_PlaySound(system, FMOD_CHANNEL_FREE, track->stream, 0, &track->channel);
  ERRCHECK(result);
  printf("... track playing\n");
  
  initEffectSet(system, &track->effects);
  printf("... init effects\n");
  
  //FMOD_DSP_TYPE    activeEffect[6];
  if (strcmp(filename, TRACKS_BEATBASS_FILENAME) == 0)
  {
    track->activeEffect[0] = FMOD_DSP_TYPE_HIGHPASS;
    track->activeEffect[1] = FMOD_DSP_TYPE_CHORUS;
    track->activeEffect[2] = FMOD_DSP_TYPE_LOWPASS;
    track->activeEffect[3] = FMOD_DSP_TYPE_FLANGE;
    track->activeEffect[4] = FMOD_DSP_TYPE_CHORUS;
  }
  else if (strcmp(filename, TRACKS_TUNE_FILENAME) == 0) 
  {
    track->activeEffect[0] = FMOD_DSP_TYPE_CHORUS;
    track->activeEffect[1] = FMOD_DSP_TYPE_FLANGE;
    track->activeEffect[2] = FMOD_DSP_TYPE_ECHO;
    track->activeEffect[3] = FMOD_DSP_TYPE_HIGHPASS;
    track->activeEffect[4] = FMOD_DSP_TYPE_ECHO;
  }
  else
  {
    printf("tracks '%s' haven't effects\n", filename);
  }
}

void initSyncPoints(Music *music, FMOD_SOUND *sound)
{
  FMOD_RESULT result;
  unsigned int length;
  result = FMOD_Sound_GetLength(sound, &length, FMOD_TIMEUNIT_MS);
  ERRCHECK(result);
  
  int step = length / 120;
  printf("La piste fait %d ms, marqueurs sont placés tous les %d ms.\n", length, step);
  music->downingStep = step / EFFECT_COUNT_STEP;
  
  printf("Le pas de decrease sera %d ms.\n", music->downingStep);

  int i;
  FMOD_SYNCPOINT *point[120];
  for (i = 0; i < length; i += step)
  {
    char *name = "warfkeur";
    result = FMOD_Sound_AddSyncPoint(sound, i, FMOD_TIMEUNIT_MS, name, point);
    ERRCHECK(result);
  }
}

void initScenario(Music *music)
{
  music->marks[0] = 1;
  music->marks[1] = 33;
  music->marks[2] = 65;
  music->marks[3] = 81;
  music->marks[4] = 105;
  music->marks[5] = 121;
  
  music->indexMark = 0;
  
  initEffectTuneOfMark(&music->tune, music->indexMark);
  initEffectBeatBassOfMark(&music->beatbass, music->indexMark);
  
  music->downingEffects = FALSE;
}

void initMusic(FMOD_SYSTEM *system, Music *music)
{
  FMOD_RESULT result;
  
  initTrack(system, &music->beatbass, TRACKS_BEATBASS_FILENAME);
  initTrack(system, &music->tune, TRACKS_TUNE_FILENAME);
  printf("... tracks inited\n");
  
  initSyncPoints(music, music->beatbass.stream);
  result = FMOD_Channel_SetCallback(music->beatbass.channel, FMOD_CHANNEL_CALLBACKTYPE_SYNCPOINT, OnSync, (int) music);
  ERRCHECK(result);
  printf("... sync points inited\n");
  
  initScenario(music);
  printf("... scenario inited\n");
  
  // effets sur la globalité du son 
  result = FMOD_System_CreateDSPByType(system, FMOD_DSP_TYPE_COMPRESSOR, &music->dspcompressor);
  ERRCHECK(result);
  
  result = FMOD_System_AddDSP(system, music->dspcompressor, 0);
  ERRCHECK(result);
  
  SetCompressor(music->dspcompressor, MUSIC_COMPRESSOR_THRESHOLD, MUSIC_COMPRESSOR_ATTACK, MUSIC_COMPRESSOR_RELEASE, MUSIC_COMPRESSOR_GAIN);
  printf("... main compressor inited\n");
}

/**
 * prepare all resources
**/
void init_music()
{
  initMusic(sound_system, &music);
  printf("... music inited\n");
}

/**
 * close drivers and hardware
 **/
void end(FMOD_SYSTEM *system)
{
  FMOD_System_Close(system);
}

/**
 * init drivers and hardware
**/
static int init(FMOD_SYSTEM **system)
{
  FMOD_RESULT result;
  printf("init...\n");
  result = FMOD_System_Create(system);
  ERRCHECK(result);
  printf("... FMOD system\n");
  result = FMOD_System_SetOutput(*system, FMOD_OUTPUTTYPE_ESD);
  ERRCHECK(result);
  printf("... FMOD output\n");
  result = FMOD_System_Init(*system, 2, FMOD_INIT_NORMAL, NULL);
  ERRCHECK(result);
  printf("... FMOD inited\n");
  return SDL_Init(SDL_INIT_VIDEO);
}
 
void init_sound()
{
  init(&sound_system);
}

void step_sound()
{
  EnterFrame(sound_system, &music);
}

void shutdown_sound()
{
  end(sound_system);
}

