#ifndef _PARALLAX_FXH_
#define _PARALLAX_FXH_


float2 ComputeParallaxOffset(float3 tViewDir, float bumpHeight) {
	float2 tParallaxDirection = normalize(tViewDir.xy);
	float len = length(tViewDir);
	float parallaxLength = sqrt(len*len - tViewDir.z*tViewDir.z) / tViewDir.z; 
	float2 tParallaxOffset = tParallaxDirection * parallaxLength;
	tParallaxOffset *= bumpHeight;
	return tParallaxOffset;
}


float2 ComputeParallaxOclusionMapping(
		sampler2D normalMap, float2 texCoord,
		float2 tParallaxOffset, float3 wViewDir,
		float3 wNormal, float bumpHeight,
		int lodThreashold, int minSamples, int maxSamples) {
	float2 dimension = float2(textureSize, textureSize);
	float2 texSample = texCoord;
	float2 fTexCoordsPerSize = texCoord * dimension;
	float2 dxSize, dySize;
	float2 dx, dy;
	float4( dxSize, dx ) = ddx( float4( fTexCoordsPerSize, texCoord ) );
	float4( dySize, dy ) = ddy( float4( fTexCoordsPerSize, texCoord ) );
	float  fMipLevel;      
	float  fMipLevelInt;
	float  fMipLevelFrac;
	float  fMinTexCoordDelta;
	float2 dTexCoords;
	dTexCoords = dxSize * dxSize + dySize * dySize;
	fMinTexCoordDelta = max( dTexCoords.x, dTexCoords.y );
	fMipLevel = max( 0.5 * log2( fMinTexCoordDelta ), 0 );
	
	if (fMipLevel <= (float)lodThreashold) {
		int nNumSteps = (int) lerp(maxSamples, minSamples, dot(wViewDir, wNormal));
		float fCurrHeight = 0.0;
		float fStepSize   = 1.0 / (float) nNumSteps;
		float fPrevHeight = 1.0;
		float fNextHeight = 0.0;
		int    nStepIndex = 0;
		bool   bCondition = true;
		float2 vTexOffsetPerStep = fStepSize * tParallaxOffset;
		float2 vTexCurrentOffset = texCoord;
		float  fCurrentBound     = 1.0;
		float  fParallaxAmount   = 0.0;
		float2 pt1 = 0;
		float2 pt2 = 0;
		float2 texOffset2 = 0;
		
		while (nStepIndex < nNumSteps) {
			vTexCurrentOffset -= vTexOffsetPerStep;
			fCurrHeight = tex2Dgrad(normalMap, vTexCurrentOffset, dx, dy).a;
			fCurrentBound -= fStepSize;
			if (fCurrHeight > fCurrentBound) {   
				pt1 = float2( fCurrentBound, fCurrHeight );
				pt2 = float2( fCurrentBound + fStepSize, fPrevHeight );
				texOffset2 = vTexCurrentOffset - vTexOffsetPerStep;
				nStepIndex = nNumSteps + 1;
				fPrevHeight = fCurrHeight;
			}
			else {
				nStepIndex++;
				fPrevHeight = fCurrHeight;
			}
		}   
		float fDelta2 = pt2.x - pt2.y;
		float fDelta1 = pt1.x - pt1.y;
		float fDenominator = fDelta2 - fDelta1;
		if (fDenominator == 0.0f)
			fParallaxAmount = 0.0f;
		else
			fParallaxAmount = (pt1.x * fDelta2 - pt2.x * fDelta1 ) / fDenominator;
		float2 vParallaxOffset = tParallaxOffset * (1 - fParallaxAmount);
		float2 texSampleBase = texCoord - vParallaxOffset;
		texSample = texSampleBase;

		if (fMipLevel > (float)(lodThreashold - 1)) {
			fMipLevelFrac = modf( fMipLevel, fMipLevelInt );
			texSample = lerp(texSampleBase, texCoord, fMipLevelFrac);
		}
	}
	
	return texSample;
}

#endif
