// world parameters
float4x4 matWorldToView;
float4x4 matViewToProj;
float3 cameraPosition;

// object parameters
bool swapTriangles;
float4x4 matObjectToWorld;
float4x4 matWorldToObject;

// material parameters
texture2D materialDiffuseMap;
texture2D materialNormalMap;
texture2D materialSpecularMap;
float2 materialNormalSwap = float2(1, 1);
float textureSize = 256;
float bumpDepth = 0.2;

// light parameters
float3 lightPosition;
float3 lightColor;
float lightRadius;
float volumeLightIntensity;
textureCUBE depthShadowMap;
textureCUBE lightCube;
float4x4 matWorldToCubeFace;


#include "sphere.fxh"
#include "shadows.fxh"
#include "parallax.fxh"


// samples

sampler2D diffuseSamp = sampler_state {
	texture = <materialDiffuseMap>;
	magFilter = linear;
	minFilter = linear;
	mipFilter = linear;
};

sampler2D normalSamp = sampler_state {
	texture = <materialNormalMap>;
	magFilter = linear;
	minFilter = linear;
	mipFilter = linear;
};

sampler2D specularSamp = sampler_state {
	texture = <materialSpecularMap>;
	magFilter = linear;
	minFilter = linear;
	mipFilter = linear;
};

samplerCUBE depthShadowSamp = sampler_state {
	texture = <depthShadowMap>;
	magFilter = linear;
	minFilter = point;
	mipFilter = none;
};

samplerCUBE lightCubeSamp = sampler_state {
	texture = <lightCube>;
	magFilter = linear;
	minFilter = point;
	mipFilter = none;
};


struct VertexInput {
	float3 oPosition: POSITION;
	float3 oNormal: NORMAL;
	float3 oTangent: TANGENT;
	float3 oBinormal: BINORMAL;
	float2 texCoord: TEXCOORD;
};


struct PixelInput {
	float4 pPosition: POSITION;
	float2 texCoord: TEXCOORD0;
	float3 wPosition: TEXCOORD1;
	float3 wTangentBasis1: TEXCOORD2;
	float3 wTangentBasis2: TEXCOORD3;
	float3 wTangentBasis3: TEXCOORD4;
};


struct PixelOutput {
	float4 color: COLOR;
};


PixelInput LambertBlinnVS(VertexInput input) {
	// transform position
	float4 oPosition = float4(input.oPosition, 1);
	float4 wPosition = mul(oPosition, matObjectToWorld);
	float4 vPosition = mul(wPosition, matWorldToView);
	float4 pPosition = mul(vPosition, matViewToProj);
	
	float3 wTangent = mul(-input.oBinormal, (float4x3)matObjectToWorld);
	float3 wBinormal = mul(input.oTangent, (float4x3)matObjectToWorld);
	float3 wNormal = mul(input.oNormal, (float4x3)matObjectToWorld);
	
	// create output
	PixelInput output = (PixelInput)0;
	output.texCoord = input.texCoord;
	output.pPosition = pPosition;
	output.wPosition = wPosition;
	output.wTangentBasis1 = wTangent;
	output.wTangentBasis2 = wBinormal;
	output.wTangentBasis3 = wNormal;
	return output;
}


PixelOutput AmbientPS(PixelInput input) {
	PixelOutput output;
	output.color = 0;
	return output;
}


float ComputeAttenuation(float dist, float radius) {
	static const float a = 0.5;
	float c = ((1+a)/a-1)/(radius*radius);
	return (1+a)/(1+c*dist*dist)-a;
}


float ComputeVolumeLightShadow(samplerCUBE depthSamp, float3 wPosition) {
	float3 lightVec = lightPosition - wPosition;
	float d1 = distance(lightPosition, wPosition);
	float d2 = texCUBE(depthSamp, lightVec);
	return d1 < d2;
}


float4 CalculateCube(samplerCUBE lightCubeSamp, float3 position) {
	float3 lightVec = lightPosition - position;
	float4 d2 = texCUBE(lightCubeSamp, lightVec);
	return d2;
}


float3 ComputeVolumeLight(samplerCUBE depthSamp, samplerCUBE lightCubeSamp, float3 wPosition, uniform bool useLightCube) {
#ifdef SHOW_VOLUMES
	float3 wLightVec = lightPosition-wPosition;
	float3 wViewVec = cameraPosition-wPosition;
	
	// compute shadows
	static const int numSamples = volumeQuality;
	float shadowResult = 0;
	float4 cubeResult = 0;
	float3 wStart = cameraPosition;
	float3 wEnd = wPosition;
	for(int i = 1; i <= numSamples; i++){
		float l = (float)i/(numSamples+1);
		float3 e = lerp(wStart, wEnd, l);
		float dist = distance(e, lightPosition);
		float attenuation = ComputeAttenuation(dist, lightRadius);
		shadowResult += ComputeVolumeLightShadow(depthSamp, e) * attenuation;
		cubeResult += CalculateCube(lightCubeSamp, e);
	}
	shadowResult *= 1.0/numSamples;
	cubeResult *= 1.0/numSamples;
	if (!useLightCube)
		cubeResult = 1;
	return shadowResult * cubeResult * lightColor;
#else
	return 0;
#endif
}


PixelOutput DiffusePS(PixelInput input, uniform bool useParallax, uniform bool shadows, uniform bool volume, uniform bool useLightCube) {
	PixelOutput output;
	
	float dist = distance(lightPosition, input.wPosition);
	float attenuation = ComputeAttenuation(dist, lightRadius);
	attenuation = saturate(attenuation);
	float3 volumeResult = 0;
	if (volume)
		volumeResult += ComputeVolumeLight(depthShadowSamp, lightCubeSamp, input.wPosition, useLightCube) * lightColor * 0.8;
	if (!volume && attenuation == 0)
		discard;
	else if (volume && attenuation == 0) {
		output.color = float4(volumeResult, 1);
		return output;
	}
	
	float3 wLightDir = lightPosition - input.wPosition;
	float3 tLightDir;
	tLightDir.x = dot(wLightDir, input.wTangentBasis1);
	tLightDir.y = dot(wLightDir, input.wTangentBasis2);
	tLightDir.z = dot(wLightDir, input.wTangentBasis3);
	tLightDir = normalize(tLightDir);
	
	float3 wCameraDir = normalize(cameraPosition - input.wPosition);
	float3 tCameraDir;
	tCameraDir.x = dot(wCameraDir, input.wTangentBasis1);
	tCameraDir.y = dot(wCameraDir, input.wTangentBasis2);
	tCameraDir.z = dot(wCameraDir, input.wTangentBasis3);
	tCameraDir = normalize(tCameraDir);
	float3 tHalfVec = normalize(tCameraDir + tLightDir);
	
	float2 texCoord = input.texCoord;
	if (useParallax) {
		float2 tParallaxOffset = ComputeParallaxOffset(tCameraDir, bumpDepth);
		texCoord = ComputeParallaxOclusionMapping(
			normalSamp, input.texCoord,
			tParallaxOffset, wCameraDir,
			input.wTangentBasis3, bumpDepth,
			g_nLODThreshold, g_nMinSamples, g_nMaxSamples);
	}
	
	float4 diffuseColor = tex2D(diffuseSamp, texCoord);
	float4 specularColor = tex2D(specularSamp, texCoord);
//	float3 tNormal = normalize(tex2D(normalSamp, texCoord)*2-1);
	float3 tNormal = tex2D(normalSamp, texCoord)*2-1;
	tNormal.xy *= materialNormalSwap;
	
	float light = saturate(dot(tLightDir, tNormal));
	float specular = pow(saturate(dot(tHalfVec, tNormal)), 10);
	float3 lightCubeColor = 1;
	if (useLightCube)
		lightCubeColor = texCUBE(lightCubeSamp, wLightDir);
	
	float3 result = 0;
	result += light * attenuation * lightColor * diffuseColor * lightCubeColor;
	result += specular * attenuation * lightColor * specularColor * lightCubeColor;
	if (shadows)
		result *= CalculateSoftShadow(depthShadowSamp, wLightDir, dist);
	if (volume)
		result += volumeResult;

	output.color = float4(result, 1);
	
	
	if (output.color.r < 0) output.color.r = 0;
	if (output.color.g < 0) output.color.g = 0;
	if (output.color.b < 0) output.color.b = 0;
	return output;
}
