//--------------------------------------------------------------------------------------
// Global variables
//--------------------------------------------------------------------------------------
float4 ambient;
float4 diffuse;
float4 specular;
float glossiness;				// larger values make a smaller specular shine
float shininess;				// larger values make a stronger specular shine

float3 g_LightPos;               // Light's direction in world space
float4 g_LightDiffuse;           // Light's diffuse color

float focalPoint=100.0;
float focalFactor=200.0;

texture texture0;
texture texture1;

float    time;
float4x4 worldMatrix;                  // World matrix for object
float4x4 worldViewProjection;    // World * View * Projection matrix
float4 viewPos;



//--------------------------------------------------------------------------------------
// Texture samplers
//--------------------------------------------------------------------------------------
sampler tex0 = 
sampler_state
{
    Texture = <texture0>;
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

sampler tex1 = 
sampler_state
{
    Texture = <texture1>;
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};


//--------------------------------------------------------------------------------------
// Vertex shader output structure
//--------------------------------------------------------------------------------------
struct VS_OUTPUT
{
    float4 Position   : POSITION;   // vertex position 
    float2 TextureUV  : TEXCOORD0;  // vertex texture coords 
    float3 Normal: TEXCOORD1;
    float3 LightDir: TEXCOORD2;
    float3 ViewDir: TEXCOORD3;
    float3 vertPos: TEXCOORD4;
};


//--------------------------------------------------------------------------------------
// This shader computes standard transform and lighting
//--------------------------------------------------------------------------------------
VS_OUTPUT RenderSceneVS( float4 vPos : POSITION, 
                         float3 vNormal : NORMAL,
                         float2 vTexCoord0 : TEXCOORD0 )
{
    VS_OUTPUT Output=(VS_OUTPUT)0;
    float3 vNormalWorldSpace;
  
    Output.Position = mul(vPos, worldViewProjection);
    Output.Normal = normalize(mul(vNormal, (float3x3)worldMatrix));

	Output.LightDir=g_LightPos-Output.Position;
	Output.ViewDir=viewPos-Output.Position;
    
    Output.TextureUV = vTexCoord0; 
    Output.vertPos=Output.Position;
    
    return Output;    
}



//--------------------------------------------------------------------------------------
// Pixel shader output structure
//--------------------------------------------------------------------------------------
struct PS_OUTPUT
{
    float4 RGBColor : COLOR0;  // Pixel color    
};

//--------------------------------------------------------------------------------------
// This shader outputs the pixel's color by modulating the texture's
//       color with diffuse material color
//--------------------------------------------------------------------------------------
PS_OUTPUT RenderScenePS( VS_OUTPUT In ) 
{ 
	PS_OUTPUT Output=(PS_OUTPUT)0;
	float4 color=tex2D(tex0, In.TextureUV);
	float4 bump=tex2D(tex1, In.TextureUV);
	
	float3 lightNorm=normalize(In.LightDir);
	float3 normalNorm=normalize(In.Normal)+bump-0.5;
	
	float4 diff=saturate(dot(lightNorm, normalNorm));
	float3 reflect=normalize(2*diff*normalNorm-lightNorm);
    float4 spec=pow(saturate(dot(reflect, normalize(In.ViewDir))), glossiness);
	
	Output.RGBColor=saturate((ambient+diffuse*g_LightDiffuse*diff)*color+spec*shininess*specular);
	Output.RGBColor.r=saturate(abs(In.vertPos.z-focalPoint)/focalFactor);

	return Output;
}


//--------------------------------------------------------------------------------------
// Renders scene to render target
//--------------------------------------------------------------------------------------
technique tester
{
    pass P0
    {          
        VertexShader = compile vs_2_0 RenderSceneVS( );
        PixelShader  = compile ps_2_0 RenderScenePS( );
    }
}
