
// initialization
#ifndef VIDF
#define LIGHTPOSITION Position
#define LIGHTCOLOR LightColor
#endif


// Engine Parameters
float4x4 mat_Object_to_World : WORLD;
float4x4 mat_World_to_View : VIEW;
float4x4 mat_View_to_Screen : PROJECTION;
float4x4 mat_World_to_Object : WORLDI;
float3 w_cameraPosition : WORLDCAMERAPOSITION;

float3 w_lightPosition : LIGHTPOSITION <
	string UIName = "Light Position"; 
	string Object = "OmniLight";
	int RefID = 0;
>;

float4 col_lightColor : LIGHTCOLOR <
	int LightRef = 0;
>;

#ifndef VIDF
static const float lightRadius = 100000;
#else
float lightRadius : LIGHTRADIUS;
#endif

float4 col_ambient : AMBIENT;


// Material parameters

float lightSmooth <
> = 0.6;

float separation <
> = 0.8;


float4 diffuseColor <
	string UIName = "Diffuse Color";
	string UItype = "ColorSwatch";
	bool VIExport = true;
> = float4(1,1,1,1);


float glowIntensity <
	string UIName = "Glow intensity";
	bool VIExport = true;
> = 1.0;


texture glowTex < 
	string UIName = "Glow Map";
	int Texcoord = 0;
	int MapChannel = 1;
	bool VIExport = true;
>;


texture lightLookupTex < 
	string UIName = "Light lookup texture";
	bool VIExport = true;
>;


texture hatch1Tex < 
	string UIName = "Hatch1 texture";
	int Texcoord = 0;
	int MapChannel = 1;
	bool VIExport = true;
>;
texture hatch2Tex < 
	string UIName = "Hatch2 texture";
	int Texcoord = 0;
	int MapChannel = 1;
	bool VIExport = true;
>;
texture hatch3Tex < 
	string UIName = "Hatch3 texture";
	int Texcoord = 0;
	int MapChannel = 1;
	bool VIExport = true;
>;



// Samples

sampler glowSamp = sampler_state{ 
	Texture = (glowTex);
	MipFilter = LINEAR;
	MinFilter = LINEAR;
	MagFilter = LINEAR;
};


sampler lightLookupSamp = sampler_state{ 
	Texture = (lightLookupTex);
	MipFilter = LINEAR;
	MinFilter = LINEAR;
	MagFilter = LINEAR;
	addressu = clamp;
	addressv = clamp;
};


sampler hatch1Samp = sampler_state{ 
	Texture = (hatch1Tex);
	MipFilter = LINEAR;
	MinFilter = LINEAR;
	MagFilter = LINEAR;
};
sampler hatch2Samp = sampler_state{ 
	Texture = (hatch2Tex);
	MipFilter = LINEAR;
	MinFilter = LINEAR;
	MagFilter = LINEAR;
};
sampler hatch3Samp = sampler_state{ 
	Texture = (hatch3Tex);
	MipFilter = LINEAR;
	MinFilter = LINEAR;
	MagFilter = LINEAR;
};


// Streams

struct VertexInput{
	float4 o_position : POSITION;
	float3 o_normal   : NORMAL;
	float2 tex_Coord0 : TEXCOORD0;
};

struct VertexOutput{
	float4 s_posRast  : POSITION;
	float2 tex_Coord0 : TEXCOORD0;
	float4 w_position : TEXCOORD1;
	float3 w_normal   : TEXCOORD2;
	float4 s_position : TEXCOORD3;
};

// Vertex Shader

VertexOutput VertexShader(VertexInput input){
	VertexOutput output;
	
	// transfom vectors
	float4 w_position = mul(input.o_position, mat_Object_to_World);
	float4 v_position = mul(w_position, mat_World_to_View);
	float4 s_position = mul(v_position, mat_View_to_Screen);
	
	float3 w_normal = normalize( mul(input.o_normal, (float3x3)mat_Object_to_World) );
	
	// write data
	output.s_posRast = s_position;
	output.s_position = s_position;
	output.w_position = w_position;
	output.w_normal = w_normal;
	output.tex_Coord0 = input.tex_Coord0;
	return output;
}

// Pixel Shader

float CalculateAttenuation(float lightRadius, float lightDistance){
	return 1 - lightDistance*(1.0/lightRadius);
}


float4 AmbientPixelShader(VertexOutput input):COLOR0{
	float3 col_final = col_ambient.rgb * diffuseColor;
	
	return float4(col_final, 1);
}


float4 ShadePixelShader(VertexOutput input):COLOR0{
	input.w_normal = normalize(input.w_normal);
	float3 w_lightDirection = w_lightPosition - input.w_position;
	float lightDist = length(w_lightDirection);
	w_lightDirection = normalize(w_lightDirection);
	float3 w_cameraDirection = normalize( w_cameraPosition - input.w_position );
	float attenuation = CalculateAttenuation(lightRadius, lightDist);
	
	// calculate ndotl
	float ndotl = saturate( dot(input.w_normal, w_lightDirection) );
	ndotl = tex2D(lightLookupSamp, float2(ndotl, 0.5));
	
	// hatch
	float hatchFactor = ndotl * attenuation;
	float4 weight0 = 0.0;
	if(hatchFactor < 0.25){
		weight0.x = hatchFactor*4;
	}
	else if(hatchFactor < 0.5){
		weight0.x = -hatchFactor*4 + 2;
		weight0.y = hatchFactor*4-1;
	}
	else if(hatchFactor < 0.75){
		weight0.y = -hatchFactor*4 + 3;
		weight0.z = hatchFactor*4-2;
	}
	else{
		weight0.z = -hatchFactor*4 + 4;
		weight0.w = hatchFactor*4-3;
	}
	
	float h1 = tex2D(hatch1Samp, input.tex_Coord0) * weight0.x;
	float h2 = tex2D(hatch2Samp, input.tex_Coord0) * weight0.y;
	float h3 = tex2D(hatch3Samp, input.tex_Coord0) * weight0.z;
	float h4 = 1.0 * weight0.w;
	ndotl = h1 + h2 + h3 + h4;
	
	// final composition
	float3 col_diffuse = ndotl * col_lightColor;
	float3 col_final = col_diffuse * diffuseColor;
	
	return float4(col_final, 1);
}


float4 EmissivePixelShader(VertexOutput input):COLOR0{
	float4 col_glowSamp = tex2D(glowSamp, input.tex_Coord0);
	float3 col_final = col_glowSamp;
	
	float i = pow((col_final.r+col_final.g+col_final.b)*0.333, 0.25) * glowIntensity;
	
	return float4(col_final, i);
}


float4 DepthDistPixelShader(VertexOutput input):COLOR0{
	float3 normal = input.w_normal*0.5+0.5;
	float dist = input.s_position.z * 0.001;
	
	return float4(normal, dist);
}


float4 MaxPixelShader(VertexOutput input):COLOR0{
	float4 res = 0;
	res += ShadePixelShader(input);
	res += EmissivePixelShader(input);
	return res;
}


// Techniques

#ifndef VIDF

technique Max{
	pass p0{
		alphaBlendEnable = false;
		vertexShader = compile vs_1_1 VertexShader();
		pixelShader = compile ps_2_0 MaxPixelShader();
	}
}

#else

technique VIDF{
	pass ambient{
		vertexShader = compile vs_1_1 VertexShader();
		pixelShader = compile ps_2_0 AmbientPixelShader();
		
		zwriteenable = true;
	}
	
	pass shade{
		vertexShader = compile vs_1_1 VertexShader();
		pixelShader = compile ps_2_0 ShadePixelShader();
		
		alphablendenable = true;
		srcblend = one;
		destblend = one;
		
		zwriteenable = false;
	}
	
	pass emissive{
		vertexShader = compile vs_1_1 VertexShader();
		pixelShader = compile ps_2_0 EmissivePixelShader();
		
		alphablendenable = true;
		srcblend = one;
		destblend = one;
		
		zwriteenable = false;
	}
	
	pass depthdist{
		vertexShader = compile vs_1_1 VertexShader();
		pixelShader = compile ps_2_0 DepthDistPixelShader();
	}
}

#endif
