using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Text;
using System.Security.Cryptography;

namespace StarifficEditor
{
    public class MeshFace
    {
        public Color color;
        public float z;
        public int[] pts;

        public MeshFace(Color col, float _z, int pt1, int pt2, int pt3)
        {
            color = col;
            z = _z;
            pts = new int[3];
            pts[0] = pt1;
            pts[1] = pt2;
            pts[2] = pt3;
        }
    }

    public class Mesh
    {
        public String name;
        public List<PointF> vertices;
        public List<MeshFace> faces;

        public int Id
        {
            get { return TextUtil.GetStringId(name); }
        }

        public Mesh(String fname)
        {
            try
            {
                name = Path.GetFileNameWithoutExtension(fname).ToLower();
                vertices = new List<PointF>();
                faces = new List<MeshFace>();

                using (TextReader fin = new StreamReader(fname))
                {
                    ReadMesh(fin, fname);
                    fin.Close();
                }

                // Sort the faces to draw order
                faces.Sort(delegate(MeshFace f1, MeshFace f2) { return Compare(f1.z, f2.z); });
            }
            catch (Exception e)
            {
                throw new Exception("Error loading mesh: " + e);
            }
        }

        private static int Compare(float f1, float f2)
        {
            if (f1 < f2) return -1;
            if (f1 > f2) return 1;
            return 0;
        }

        public RectangleF GetBoundingBox(float scale, float angle)
        {
            if (faces.Count <= 0)
                return new RectangleF();

            double ang_rad = Math.PI * angle / 180.0;
            float c = (float)Math.Cos(ang_rad);
            float s = (float)Math.Sin(ang_rad);

            float minx = 0, miny = 0, maxx = 0, maxy = 0;
            bool first = true;
            foreach (MeshFace face in faces)
            {
                foreach (int pt_idx in face.pts)
                {
                    PointF pt = vertices[pt_idx];
                    float x = (pt.X*c - pt.Y*s) * scale;
                    float y = (pt.X*s + pt.Y*c) * scale;

                    if (first)
                    {
                        first = false;
                        minx = maxx = x;
                        miny = maxy = y;
                    }
                    else
                    {
                        minx = Math.Min(minx, x);
                        miny = Math.Min(miny, y);
                        maxx = Math.Max(maxx, x);
                        maxy = Math.Max(maxy, y);
                    }
                }
            }

            return new RectangleF(minx, miny, maxx - minx, maxy - miny);
        }

        public void Draw(Graphics g, float dest_x, float dest_y, float scale, float angle, bool wireframe)
        {
            PointF[] pts = new PointF[3];

            Brush brush = null;
            Pen pen = null;
            Color color = Color.White;

            double ang_rad = Math.PI * angle / 180.0;
            float c = (float)Math.Cos(ang_rad);
            float s = (float)Math.Sin(ang_rad);

            foreach (MeshFace face in faces)
            {
                // Process the coords
                for (int i = 0; i < 3; ++i)
                {
                    PointF pt = vertices[face.pts[i]];
                    pts[i] = new PointF(dest_x + (pt.X * c - pt.Y * s) * scale, dest_y + (pt.X * s + pt.Y * c) * scale);
                }

                // Draw it
                if (brush == null || color != face.color)
                {
                    color = face.color;
                    brush = new SolidBrush(color);
                    pen = new Pen(brush);
                }

                if (wireframe)
                    g.DrawPolygon(pen, pts);
                else
                    g.FillPolygon(brush, pts);
            }
        }

        public void Export(JavaBinaryWriter fout)
        {
            fout.Write(Id);
            fout.Write(name);

            fout.Write((short)vertices.Count);
            foreach (PointF pt in vertices)
            {
                fout.WriteFixedScaled(pt.X);
                fout.WriteFixedScaled(pt.Y);
            }

            fout.Write((short)faces.Count);
            foreach (MeshFace f in faces)
            {
                fout.Write(f.color.ToArgb());
                fout.Write((short)f.pts[0]);
                fout.Write((short)f.pts[1]);
                fout.Write((short)f.pts[2]);
            }
        }

        public override string ToString()
        {
            return name;
        }

        private void ReadMesh(TextReader fin, String fname)
        {
            Dictionary<String, Color> mat_map = new Dictionary<String, Color>();
            List<float> vertex_z = new List<float>();

            Color cur_col = Color.White;

            while (true)
            {
                String line = fin.ReadLine();
                if (line == null)
                    break;
                line = line.Trim();
                if (line.Length <= 0 || line[0] == '#')
                    continue;

                // Command string
                String[] cmd = line.Split(' ');
                if (cmd.Length <= 0)
                    continue;

                switch (cmd[0])
                {
                    case "mtllib":
                        ReadMaterial(mat_map, fname, cmd[1]);
                        break;

                    case "usemtl":
                        cur_col = Color.White;
                        if (mat_map.ContainsKey(cmd[1]))
                            cur_col = mat_map[cmd[1]];
                        break;

                    case "v":
                        vertices.Add(new PointF(TextUtil.ParseFloat(cmd[1], 0), TextUtil.ParseFloat(cmd[3], 0)));
                        vertex_z.Add(TextUtil.ParseFloat(cmd[2], 0));
                        break;

                    case "f":
                        if (true)
                        {
                            for (int i = 2; i < (cmd.Length - 1); ++i)
                            {
                                int v1 = ParseFaceIdx(cmd[1]) - 1;
                                int v2 = ParseFaceIdx(cmd[i]) - 1;
                                int v3 = ParseFaceIdx(cmd[i+1]) - 1;

                                float z = (vertex_z[v1] + vertex_z[v2] + vertex_z[v3]) / 3.0f;

                                if (vertices[v1] == null || vertices[v2] == null || vertices[v3] == null)
                                    throw new Exception("Invalid vertex index!");

                                faces.Add(new MeshFace(cur_col, z, v1, v2, v3));
                            }
                        }
                        break;
                }
            }
        }

        private int ParseFaceIdx(String idx_str)
        {
            idx_str = idx_str.Split('/')[0];
            return int.Parse(idx_str);
        }

        private void ReadMaterial(Dictionary<String, Color> mat, String base_name, String name)
        {
            String mat_name = Path.Combine(Path.GetDirectoryName(base_name), name);

            using (TextReader fin = new StreamReader(mat_name))
            {
                String cur_mat = null;
                Color cur_color = Color.White;

                while (true)
                {
                    String line = fin.ReadLine();
                    if (line == null)
                        break;
                    line = line.Trim();
                    if (line.Length <= 0 || line[0] == '#')
                        continue;

                    // Command string
                    String[] cmd = line.Split(' ');
                    if (cmd.Length <= 0)
                        continue;

                    switch (cmd[0])
                    {
                        case "newmtl":
                            if (cur_mat != null)
                                mat[cur_mat] = cur_color;
                            cur_mat = cmd[1];
                            cur_color = Color.White;
                            break;

                        case "Kd":
                            if (true)
                            {
                                float r = TextUtil.ParseFloat(cmd[1], 0);
                                float g = TextUtil.ParseFloat(cmd[2], 0);
                                float b = TextUtil.ParseFloat(cmd[3], 0);
                                cur_color = Color.FromArgb(ToRGB(r), ToRGB(g), ToRGB(b));
                            }
                            break;
                    }
                }

                // Store the final material
                if (cur_mat != null)
                    mat[cur_mat] = cur_color;

                fin.Close();
            }
        }

        private static int ToRGB(float v)
        {
            int ret = (int)(v * 255.0f);
            if (ret < 0)
                return 0;
            if (ret > 255)
                return 255;
            return ret;
        }
    }
}
