using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.IO;
using System.Text;
using System.Xml;

namespace StarifficEditor
{
    public enum LevelObjectType
    {
        Start = 0,
        End = 1,
        Rectangle = 10,
        Sphere = 11,
        Mesh = 20,
        Item = 30,
    }

    public enum ObjectSpecialMode
    {
        Normal = 0,
        NoCollisions = 1,
        Bumper = 2,
        Death = 3,
    }

    public abstract class LevelObject : ICloneable
    {
        public string name = "Default";
        public string Name
        {
            get { return name; }
            set { name = value; }
        }

        public float x = 10.0f;
        public float X
        {
            get { return x; }
            set { x = value; }
        }

        public float y = 10.0f;
        public float Y
        {
            get { return y; }
            set { y = value; }
        }

        [Browsable(false)]
        public virtual RectangleF BoundingBox
        {
            get { return new RectangleF(x, y, 0, 0); }
        }

        [Browsable(false)]
        public virtual bool IsStatic
        {
            get { return true; }
        }

        [Browsable(false)]
        public virtual bool Deletable
        {
            get { return true; }
        }

        [Browsable(false)]
        public virtual LevelObjectType Type
        {
            get
            {
                String type = GetType().ToString();
                String[] parts = type.Split(new String[] { "LevelObject" }, StringSplitOptions.None);
                if (parts.Length != 2)
                    throw new Exception("Invalid object type name!");
                type = parts[1];

                return (LevelObjectType) Enum.Parse(typeof(LevelObjectType), type, true);
            }
        }

        public LevelObject()
        {
            Name = "Obj" + (DateTime.Now.Ticks / 10000000L);
        }

        public abstract void Draw(Graphics g, bool selected, bool wireframe);

        public virtual bool IsPointOnObject(PointF pt)
        {
            return BoundingBox.Contains(pt);
        }

        public override string ToString()
        {
            return Type + ": " + name;
        }

        public virtual void Load(XmlElement elem)
        {
			name = TextUtil.GetString(elem["name"], "value", name);
			x = TextUtil.GetFloat(elem["x"], "value", x);
			y = TextUtil.GetFloat(elem["y"], "value", y);
		}

		public virtual void Save(XmlElement elem)
		{
			TextUtil.AddStringElement(elem, "name", "value", name);
			TextUtil.AddStringElement(elem, "x", "value", "" + x);
			TextUtil.AddStringElement(elem, "y", "value", "" + y);
		}

        public virtual void Export(JavaBinaryWriter fout)
		{
            RectangleF bbox = BoundingBox;
            fout.WriteFixedScaled(bbox.X);
            fout.WriteFixedScaled(bbox.Y);
            fout.WriteFixedScaled(bbox.Width);
            fout.WriteFixedScaled(bbox.Height);

            fout.WriteFixedScaled(x);
            fout.WriteFixedScaled(y);
        }

        protected Color GetColor(Color c, bool selected)
        {
            if (selected)
                return c;

            int col = c.ToArgb();
            int new_a = (int) (0.6f * ((col>>24)&0xff));
            col = (new_a<<24) | (col & 0x00ffffff);
            return Color.FromArgb(col);
        }

        protected Brush GetBrush(Color c, bool selected)
        {
            return new SolidBrush(GetColor(c, selected));
        }

        protected Pen GetPen(Color c, bool selected)
        {
            return new Pen(GetBrush(c, selected));
        }

        public object Clone()
        {
            return MemberwiseClone();
        }
    }

    public class LevelObjectStart : LevelObject
    {
        private static readonly float Radius = 10.0f;

        public override RectangleF BoundingBox
        {
            get { return new RectangleF(x - Radius, y - Radius, 2 * Radius, 2 * Radius); }
        }

        public override bool Deletable
        {
            get { return false; }
        }

        public override bool IsStatic
        {
            get { return false; }
        }

        public LevelObjectStart()
        {
            name = "Player";
        }

        public override void Draw(Graphics g, bool selected, bool wireframe)
        {
            Pen pen = GetPen(Color.YellowGreen, selected);
            for (int i = 0; i < 5; ++i)
            {
                float ang = (float) (i * 2.0f * Math.PI / 5.0f);
                g.DrawLine(pen, x, y, x + Radius * (float)Math.Sin(ang), y - Radius * (float)Math.Cos(ang));
            }

            g.FillRectangle(GetBrush(Color.Yellow, selected), x - 1, y - 2, 3, 3);
        }

		public override void Load(XmlElement elem)
		{
			base.Load(elem);
		}

		public override void Save(XmlElement elem)
		{
			base.Save(elem);
		}

        public override void Export(JavaBinaryWriter fout)
        {
            base.Export(fout);
        }
    }

    public class LevelObjectEnd : LevelObject
    {
        private static readonly float Radius = 10.0f;

        public override RectangleF BoundingBox
        {
            get { return new RectangleF(x - Radius, y - Radius, 2 * Radius, 2 * Radius); }
        }

        public override bool Deletable
        {
            get { return false; }
        }

        public LevelObjectEnd()
        {
            name = "Goal";
        }

        public override void Draw(Graphics g, bool selected, bool wireframe)
        {
            Pen pen = GetPen(Color.YellowGreen, selected);
            pen.Width = 3.0f;

            RectangleF bbox = BoundingBox;
            
            if (wireframe)
                g.DrawRectangle(GetPen(Color.Yellow, selected), bbox.Left, bbox.Top, bbox.Width, bbox.Height);
            else
                g.FillRectangle(GetBrush(Color.Yellow, selected), bbox.Left, bbox.Top, bbox.Width, bbox.Height);

            g.DrawLine(pen, bbox.X + bbox.Width / 2, bbox.Y, bbox.X + bbox.Width / 2, bbox.Y + bbox.Height);
            g.DrawLine(pen, bbox.X, bbox.Y + bbox.Height / 2, bbox.X + bbox.Width, bbox.Y + bbox.Height / 2);
        }

        public override void Load(XmlElement elem)
        {
            base.Load(elem);
        }

        public override void Save(XmlElement elem)
        {
            base.Save(elem);
        }

        public override void Export(JavaBinaryWriter fout)
        {
            base.Export(fout);
        }
    }

    public class LevelObjectRectangle : LevelObject
    {
        public Color color1 = Color.White;
        public Color Color1
        {
            get { return color1; }
            set { color1 = value; }
        }

        public SizeF size = new SizeF(50, 50);
        public SizeF Size
        {
            get { return size; }
            set { size = value; }
        }

        public float angle = 0.0f;
        public float Angle
        {
            get { return angle; }
            set { angle = value; }
        }

        public ObjectSpecialMode mode = ObjectSpecialMode.Normal;
        public ObjectSpecialMode Mode
        {
            get { return mode; }
            set { mode = value; }
        }

        public float bump_power = 10.0f;
        [Category("Bumper")]
        public float BumpPower
        {
            get { return bump_power; }
            set { bump_power = value; }
        }

        [Browsable(false)]
        public override RectangleF BoundingBox
        {
            get
            {
                float minx = 0, miny = 0, maxx = 0, maxy = 0;
                bool first = true;
                foreach ( PointF pt in Points)
                {
                    if (first)
                    {
                        minx = maxx = pt.X;
                        miny = maxy = pt.Y;
                        first = false;
                    }
                    else
                    {
                        minx = Math.Min(minx, pt.X);
                        miny = Math.Min(miny, pt.Y);
                        maxx = Math.Max(maxx, pt.X);
                        maxy = Math.Max(maxy, pt.Y);
                    }
                }
                return new RectangleF(minx, miny, maxx - minx, maxy - miny);
            }
        }

        private PointF[] Points
        {
            get
            {
                PointF[] pts = new PointF[4];
                for (int i = 0; i < 4; ++i)
                    pts[i] = GetPoint(i);
                return pts;
            }
        }

        public override void Draw(Graphics g, bool selected, bool wireframe)
        {
            PointF[] pts = Points;

            if (wireframe)
            {
                Pen pen = GetPen(color1, selected);
                g.DrawPolygon(pen, pts);
                if (selected)
                {
                    g.DrawLine(pen, pts[0], pts[2]);
                    g.DrawLine(pen, pts[1], pts[3]);
                }
            }
            else
                g.FillPolygon(GetBrush(color1, selected), pts);
        }

        public override bool IsPointOnObject(PointF pt)
        {
            if (!base.IsPointOnObject(pt))
                return false;

            float w2 = size.Width / 2.0f;
            float h2 = size.Height / 2.0f;

            PointF d = PointF.Subtract( pt, new SizeF(X, Y) );

            for (int i = 0; i < 2; ++i)
            {
                PointF axis = GetAxis(i == 0);

                float dist = d.X * axis.X + d.Y * axis.Y;
                float test = (i == 0) ? w2 : h2;

                if (dist > test) return false;
                if (dist < -test) return false;
            }

            return true;
        }

		public override void Load(XmlElement elem)
		{
			base.Load(elem);
			color1 = Color.FromArgb(TextUtil.GetInt(elem["color1"], "value", color1.ToArgb()));
			size.Width = TextUtil.GetFloat(elem["size"], "x", size.Width);
            size.Height = TextUtil.GetFloat(elem["size"], "y", size.Height);
            angle = TextUtil.GetFloat(elem["angle"], "value", angle);
            mode = TextUtil.GetEnum<ObjectSpecialMode>(elem["mode"], "value", mode);
            bump_power = TextUtil.GetFloat(elem["bump_power"], "value", bump_power);
		}

		public override void Save(XmlElement elem)
		{
			base.Save(elem);
			TextUtil.AddStringElement(elem, "color1", "value", "" + color1.ToArgb());
			TextUtil.AddStringElement(elem, "size", "x", "" + size.Width);
			TextUtil.AddStringElement(elem, "size", "y", "" + size.Height);
            TextUtil.AddStringElement(elem, "angle", "value", "" + angle);
            TextUtil.AddStringElement(elem, "mode", "value", "" + mode);
            TextUtil.AddStringElement(elem, "bump_power", "value", "" + bump_power);
        }

        public override void Export(JavaBinaryWriter fout)
        {
            base.Export(fout);
            fout.Write((byte)mode);
            fout.WriteFixedScaled(bump_power);

            fout.Write(color1.ToArgb());

            bool is_rotated = Math.Abs(angle % 360.0f) > 0.0001f;
            fout.Write((byte)(is_rotated ? 1 : 0));

            foreach (PointF pt in Points)
            {
                fout.WriteFixedScaled(pt.X);
                fout.WriteFixedScaled(pt.Y);
            }

            PointF axis_x = GetAxis(true);
            fout.WriteFixed(axis_x.X);
            fout.WriteFixed(axis_x.Y);

            PointF axis_y = GetAxis(false);
            fout.WriteFixed(axis_y.X);
            fout.WriteFixed(axis_y.Y);

            fout.WriteFixedScaled(size.Width / 2.0f);
            fout.WriteFixedScaled(size.Height / 2.0f);
        }

        private PointF GetPoint(int idx)
        {
            float w2 = size.Width / 2.0f;
            float h2 = size.Height / 2.0f;

            float x, y;
            switch (idx)
            {
                case 0: x = -w2; y = -h2; break;
                case 1: x =  w2; y = -h2; break;
                case 2: x =  w2; y =  h2; break;
                case 3: x = -w2; y =  h2; break;
                default: throw new Exception("Invalid index!");
            }

            double ang_rad = Math.PI * angle / 180.0;
            float c = (float)Math.Cos(ang_rad);
            float s = (float)Math.Sin(ang_rad);

            return new PointF(X + (x * c - y * s), Y + (x * s + y * c));
        }

        private PointF GetAxis(bool is_x_axis)
        {
            double ang_rad = Math.PI * angle / 180.0;
            float c = (float)Math.Cos(ang_rad);
            float s = (float)Math.Sin(ang_rad);

            float x = 0, y = 0;
            if (is_x_axis)
                x = 1;
            else
                y = 1;

            return new PointF(x * c - y * s, x * s + y * c);
        }
    }

    public class LevelObjectSphere : LevelObject
    {
        public Color color1 = Color.White;
        public Color Color1
        {
            get { return color1; }
            set { color1 = value; }
        }

        public float radius = 50.0f;
        public float Radius
        {
            get { return radius; }
            set { radius = value; }
        }

        public ObjectSpecialMode mode = ObjectSpecialMode.Normal;
        public ObjectSpecialMode Mode
        {
            get { return mode; }
            set { mode = value; }
        }

        public float bump_power = 10.0f;
        [Category("Bumper")]
        public float BumpPower
        {
            get { return bump_power; }
            set { bump_power = value; }
        }

        [Browsable(false)]
        public override RectangleF BoundingBox
        {
            get { return new RectangleF(x-radius, y-radius, radius*2.0f, radius*2.0f); }
        }

        public override void Draw(Graphics g, bool selected, bool wireframe)
        {
            if (wireframe)
            {
                Pen pen = GetPen(color1, selected);
                g.DrawEllipse(pen, x - radius, y - radius, radius * 2.0f, radius * 2.0f);
                if (selected)
                {
                    g.DrawLine(pen, x - radius, y, x + radius, y);
                    g.DrawLine(pen, x, y - radius, x, y + radius);
                }
            }
            else
                g.FillEllipse(GetBrush(color1, selected), x - radius, y - radius, radius * 2.0f, radius * 2.0f);
        }

        public override bool IsPointOnObject(PointF pt)
        {
            if (!base.IsPointOnObject(pt))
                return false;
            
            // Distance check
            float dx = pt.X - X;
            float dy = pt.Y - Y;
            float dist = (float)Math.Sqrt(dx * dx + dy * dy);
            return (dist <= radius);
        }

        public override void Load(XmlElement elem)
		{
			base.Load(elem);
			color1 = Color.FromArgb(TextUtil.GetInt(elem["color1"], "value", color1.ToArgb()));
			radius = TextUtil.GetFloat(elem["radius"], "value", radius);
            mode = TextUtil.GetEnum<ObjectSpecialMode>(elem["mode"], "value", mode);
            bump_power = TextUtil.GetFloat(elem["bump_power"], "value", bump_power);
        }

		public override void Save(XmlElement elem)
		{
			base.Save(elem);
			TextUtil.AddStringElement(elem, "color1", "value", "" + color1.ToArgb());
            TextUtil.AddStringElement(elem, "radius", "value", "" + radius);
            TextUtil.AddStringElement(elem, "mode", "value", "" + mode);
            TextUtil.AddStringElement(elem, "bump_power", "value", "" + bump_power);
        }

        public override void Export(JavaBinaryWriter fout)
        {
            base.Export(fout);
            fout.Write((byte)mode);
            fout.WriteFixedScaled(bump_power);
            fout.Write(color1.ToArgb());
            fout.WriteFixedScaled(radius);
        }
    }

    public class LevelObjectMesh : LevelObject
    {
        public float scale = 1.0f;
        public float Scale
        {
            get { return scale; }
            set { scale = value; }
        }

        public float angle = 0.0f;
        public float Angle
        {
            get { return angle; }
            set { angle = value; }
        }

        public String mesh = "";
        [Editor(typeof(MeshDesignEditor), typeof(System.Drawing.Design.UITypeEditor))]
        public String Mesh
        {
            get { return mesh; }
            set { mesh = value; }
        }

        [Browsable(false)]
        public Mesh MeshObj
        {
            get
            {
                Mesh ret;
                if (!Meshes.global_meshes.meshes.TryGetValue(mesh, out ret))
                    return null;
                return ret;
            }
        }

        [Browsable(false)]
        public override RectangleF BoundingBox
        {
            get
            {
                Mesh m = MeshObj;
                if (m == null)
                    return new RectangleF(X - 5, Y - 5, 10, 10);
                RectangleF bbox = m.GetBoundingBox(scale, angle);
                bbox.Offset(X, Y);
                return bbox;
            }
        }

        public override void Draw(Graphics g, bool selected, bool wireframe)
        {
            Mesh m = MeshObj;

            if (m == null)
            {
                g.FillRectangle(Brushes.Red, X - 5, Y - 5, 10, 10);
            }
            else
            {
                m.Draw(g, X, Y, scale, angle, wireframe);
            }
        }

        public override void Load(XmlElement elem)
        {
            base.Load(elem);
            scale = TextUtil.GetFloat(elem["scale"], "value", scale);
            angle = TextUtil.GetFloat(elem["angle"], "value", angle);
            mesh = TextUtil.GetString(elem["mesh"], "value", mesh);
        }

        public override void Save(XmlElement elem)
        {
            base.Save(elem);
            TextUtil.AddStringElement(elem, "scale", "value", "" + scale);
            TextUtil.AddStringElement(elem, "angle", "value", "" + angle);
            TextUtil.AddStringElement(elem, "mesh", "value", mesh);
        }

        public override void Export(JavaBinaryWriter fout)
        {
            base.Export(fout);
            fout.WriteFixed(scale);

            double ang_rad = Math.PI * angle / 180.0;
            fout.WriteFixed((float)ang_rad);

            fout.Write(TextUtil.GetStringId(mesh));
        }
    }

    public class LevelObjectItem : LevelObject
    {
        private static readonly float Radius = 10.0f;

        public enum ItemTypeEnum
        {
            GravityDir = 0,
            Collectible = 1,
        }

        public enum DirEnum
        {
            Right, Down, Left, Up,
        }

        public ItemTypeEnum item_type = ItemTypeEnum.GravityDir;
        public ItemTypeEnum ItemType
        {
            get { return item_type; }
            set { item_type = value; }
        }

        public DirEnum gravity_dir = DirEnum.Down;
        [Category("Gravity direction")]
        public DirEnum GravityDir
        {
            get { return gravity_dir; }
            set { gravity_dir = value; }
        }

        public override RectangleF BoundingBox
        {
            get { return new RectangleF(x - Radius, y - Radius, 2 * Radius, 2 * Radius); }
        }

        public LevelObjectItem()
        {
        }

        public override void Draw(Graphics g, bool selected, bool wireframe)
        {
            RectangleF bbox = BoundingBox;

            if (item_type == ItemTypeEnum.GravityDir)
            {
                float dx = (float)Math.Cos((int)gravity_dir * Math.PI / 2.0);
                float dy = (float)Math.Sin((int)gravity_dir * Math.PI / 2.0);
                float nx = -dy;
                float ny = dx;

                PointF[] poly = new PointF[3];
                poly[0] = new PointF(X + dx * 6, Y + dy * 6);
                poly[1] = new PointF(X - dx * 6 + nx * 4, Y - dy * 6 + ny * 4);
                poly[2] = new PointF(X - dx * 6 - nx * 4, Y - dy * 6 - ny * 4);

                g.DrawRectangle(GetPen(Color.LightBlue, selected), bbox.Left, bbox.Top, bbox.Width, bbox.Height);

                if (wireframe)
                    g.DrawPolygon(GetPen(Color.LightBlue, selected), poly);
                else
                    g.FillPolygon(GetBrush(Color.LightBlue, selected), poly);
            }

            if (item_type == ItemTypeEnum.Collectible)
            {
                Font f = new Font( FontFamily.GenericMonospace, 10.0f );
                g.FillEllipse(GetBrush(Color.LightGoldenrodYellow, selected), bbox);
                g.DrawString("C", f, Brushes.Black, new PointF(bbox.X, bbox.Y));
            }
        }

        public override void Load(XmlElement elem)
        {
            base.Load(elem);
            item_type = TextUtil.GetEnum<ItemTypeEnum>(elem["item_type"], "value", item_type);
            gravity_dir = TextUtil.GetEnum<DirEnum>(elem["gravity_dir"], "value", gravity_dir);
        }

        public override void Save(XmlElement elem)
        {
            base.Save(elem);
            TextUtil.AddStringElement(elem, "item_type", "value", "" + item_type);
            TextUtil.AddStringElement(elem, "gravity_dir", "value", "" + gravity_dir);
        }

        public override void Export(JavaBinaryWriter fout)
        {
            base.Export(fout);
            fout.Write((byte)item_type);

            fout.Write((byte)gravity_dir);
        }
    }

    public sealed class LevelObjectFactory
    {
        public static LevelObject CreateObject(LevelObjectType type)
        {
            try
            {
                Type obj_type = Type.GetType("StarifficEditor.LevelObject" + type);
                return (LevelObject) obj_type.GetConstructor(new Type[0]).Invoke(new Object[0]);
            }
            catch (Exception)
            {
            }
            return null;
        }

        private LevelObjectFactory()
        { }
    }
}
