

/**
 * Routines for a 2D vector (worked in arrays). 
 */
public final class Vec2
{
    /**
     * Creates a new empty 2D vector.
     * 
     * @return New vector (zero).
     */
    public static final int[] create()
    {
        return create( 0, 0 );
    }
    
    /**
     * Creates a new 2D vector.
     * 
     * @param x The x component.
     * @param y The y component.
     * @return New vector.
     */
    public static final int[] create( int x, int y )
    {
        return new int[]{ x, y };
    }
    
    /**
     * Sets the vectors values.
     * 
     * @param vec The vector to set.
     * @param x The x component.
     * @param y The y component.
     */
    public static final void set( int[] vec, int x, int y )
    {
        vec[ 0 ] = x;
        vec[ 1 ] = y;
    }

    /**
     * Sets the vectors values.
     * 
     * @param vec The vector to set.
     * @param src The source vector.
     */
    public static final void set( int[] vec, int[] src )
    {
        vec[ 0 ] = src[ 0 ];
        vec[ 1 ] = src[ 1 ];
    }
    
    /**
     * Sets the vectors components as integers.
     * 
     * @param dest The destination vector.
     * @param vec The source vector.
     */
    public static final void getInt( int[] dest, int[] vec )
    {
        dest[ 0 ] = FP.toInt( vec[ 0 ] );
        dest[ 1 ] = FP.toInt( vec[ 1 ] );
    }
    
    /**
     * Sets the vectors components as integers, after scaling and adding (intvec = (fpvec - top) * scale).
     * 
     * @param dest The destination vector.
     * @param vec The source vector.
     */
    public static final void getScaledInt( int[] dest, int[] vec, int top_x, int top_y, int scale_x, int scale_y )
    {
        dest[ 0 ] = FP.toInt( FP.mul(vec[ 0 ] - top_x, scale_x) );
        dest[ 1 ] = FP.toInt( FP.mul(vec[ 1 ] - top_y, scale_y) );
    }
    
    /**
     * Makes the vector it's normal.
     * 
     * @param vec The vector to modify.
     */
    public static final void makeNormal( int[] vec )
    {
        int x = vec[ 0 ];
        int y = vec[ 1 ];
        
        vec[ 0 ] = -y;
        vec[ 1 ] = x;
    }
    
    /**
     * Does "dest = a + b * b_scale".
     */
    public static final void addScaled( int[] dest, int[] a, int[] b, int b_scale )
    {
        dest[ 0 ] = a[ 0 ] + FP.mul( b[ 0 ], b_scale );
        dest[ 1 ] = a[ 1 ] + FP.mul( b[ 1 ], b_scale );
    }
    
    /**
     * Does a "dest += src".
     * 
     * @param dest The destination vector.
     * @param src The value vector.
     */
    public static final void add( int[] dest, int[] src )
    {
        dest[ 0 ] += src[ 0 ];
        dest[ 1 ] += src[ 1 ];
    }
    
    /**
     * Does a "dest = a+b".
     * 
     * @param dest The destination vector (can be one of a and b).
     * @param a The first vector.
     * @param b The second vector.
     */
    public static final void add( int[] dest, int[] a, int[] b )
    {
        dest[ 0 ] = a[ 0 ] + b[ 0 ];
        dest[ 1 ] = a[ 1 ] + b[ 1 ];
    }
    
    /**
     * Does a "dest -= src".
     * 
     * @param dest The destination vector.
     * @param src The value vector.
     */
    public static final void sub( int[] dest, int[] src )
    {
        dest[ 0 ] -= src[ 0 ];
        dest[ 1 ] -= src[ 1 ];
    }
    
    /**
     * Does a "dest = a-b".
     * 
     * @param dest The destination vector (can be one of a and b).
     * @param a The first vector.
     * @param b The second vector.
     */
    public static final void sub( int[] dest, int[] a, int[] b )
    {
        dest[ 0 ] = a[ 0 ] - b[ 0 ];
        dest[ 1 ] = a[ 1 ] - b[ 1 ];
    }
    
    /**
     * Scales the vector.
     * 
     * @param dest The vector to scale.
     * @param mul The scale factor.
     */
    public static final void scale( int[] dest, int mul )
    {
        dest[ 0 ] = FP.mul( dest[ 0 ], mul );
        dest[ 1 ] = FP.mul( dest[ 1 ], mul );
    }
    
    /**
     * Does a linear interpolation between two vectors.
     * 
     * @param dest The destination vector.
     * @param vec1 Source vector 1.
     * @param vec2 Source vector 2.
     * @param t The fixed point lerp factor (0-1).
     */
    public static final void lerp( int[] dest, int[] vec1, int[] vec2, int t )
    {
    	dest[0] = vec1[0] + FP.mul( t, vec2[0] - vec1[0] );
    	dest[1] = vec1[1] + FP.mul( t, vec2[1] - vec1[1] );
    }
    
    /**
     * Returns the length of the vector.
     * 
     * @param vec The vector to check.
     * @return Returns the length of the vector.
     */
    public static final int length( int[] vec )
    {
        return FP.sqrt( FP.mul( vec[ 0 ], vec[ 0 ] ) + FP.mul( vec[ 1 ], vec[ 1 ] ) );
    }

    /**
     * Dot product between two vectors.
     * 
     * @param v1 The first vector.
     * @param v2 The second vector.
     * @return Dot product between two vectors.
     */
    public static final int dot( int[] v1, int[] v2 )
    {
        return FP.mul( v1[0], v2[0] ) + FP.mul( v1[1], v2[1] ); 
    }
    
    /**
     * Returns the distance between the two vectors.
     * 
     * @param v1 Vector #1.
     * @param v2 Vector #2.
     * @return The distance.
     */
    public static final int dist( int[] v1, int[] v2 )
    {
        int p0 = v2[0] - v1[0];
        int p1 = v2[1] - v1[1];

        return FP.sqrt( FP.mul( p0, p0 ) + FP.mul( p1, p1 ) );
    }
    
    /**
     * Normalizes the vector.
     * 
     * @param vec The vector to normalize.
     */
    public static final int normalize( int[] vec )
    {
        int len = length( vec );
        if ( len <= 0 )
        {
            set( vec, 0, 0 );
            return 0;
        }
        
        vec[ 0 ] = FP.div( vec[ 0 ], len );
        vec[ 1 ] = FP.div( vec[ 1 ], len );
        return len;
    }
    
    public static String toString( int[] vec )
    {
        return "(" + FP.toString(vec[0]) + ", " + FP.toString(vec[1]) + ")"; 
    }
    
    private Vec2()
    { }
}
