#include "libfhi_premodel.h"

#include "libfhi_configfile.h"
#include "libfhi_mesh.h"
#include "libfhi_texture.h"

#include <float.h>
#include <iostream>

using namespace libfhi;

//############################################################################
// Constructor / destructor ##################################################
//############################################################################

/** Empty constructor.
 */
PreModel::PreModel()
  : Attributable(DEFAULT_ATTR)
{
  // Do nothing.
}

/** Default constructor.
 * @param filename Read the model from this file if possible.
 */
PreModel::PreModel(const char *filename)
{
  ConfigFile filu(filename);

  if(!filu.is_ok())
  {
    std::cerr << "Error opening model: \"" << filename << "\"\n";
    return;
  }

  // Loader information.
  float scale[3] = { 1.0f, 1.0f, 1.0f };
  Matrix44 rm;
  bool rotate = false;
  int scaleto = -1;
  float scaleto_val = 1.0f;
  std::vector<Color4> colors;

  // Loop until at end of file.
  while(filu.advance())
  {
    if(filu.has_id("scale"))
    {
      switch(filu.get_num_arg())
      {
	case 3:
	  scale[0] = filu.get_float(0);
	  scale[1] = filu.get_float(1);
	  scale[2] = filu.get_float(2);
	  break;

	case 1:
	  scale[0] = scale[1] = scale[2] = filu.get_float(0);
	  break;

	default:
	  filu.warn("illegal definition");
	  break;
      }
    }
    else if(filu.has_id_arg("scaleto", 2))
    {
      scaleto = filu.get_int(0);
      scaleto_val = filu.get_float(1);
    }
    else if(filu.has_id_arg("rotate", 3))
    {
      rotate = true;
      rm.identity();
      rm.rotation_zxy33(filu.get_int(0), filu.get_int(1), filu.get_int(2));
    }
    else if(filu.has_id_arg("texture", 1))
    {
      this->texture_file = filu.get_string(0);
    }
    else if(filu.has_id("color"))
    {
      switch(filu.get_num_arg())
      {
	case 3: // Allow for non-alpha definition.
	  colors.push_back(Color4(filu.get_float(0), filu.get_float(1),
  		filu.get_float(2), 1.0f));
	  break;

	case 4:
	  colors.push_back(Color4(filu.get_float(0), filu.get_float(1),
		filu.get_float(2), filu.get_float(3)));
	  break;

	default:
	  filu.warn("illegal definition");
	  break;
      }
    }
    else if(filu.has_id("vertex"))
    {
      Vector2 tex(0.0f, 0.0f);
      Vector3 ver;

      switch(filu.get_num_arg())
      {
	case 6: // Texture included.
	  tex.set(filu.get_float(4), filu.get_float(5));
	  // Fallback.

	case 4:
	  ver.set(filu.get_float(0) * scale[0],
	      filu.get_float(1) * scale[1],
	      filu.get_float(2) * scale[2]);
	  // Rotate if need.
	  if(rotate)
	  {
	    ver = rm * ver;
	  }
	  this->add_vertex(ver,
	      colors[filu.get_int(3)],
	      tex);
	  break;

	default:
	  filu.warn("illegal definition");
	  break;
      }
    }
    else if(filu.has_id_arg("voxel", 1))
    {
      this->add_face(filu.get_int(0));
    }
    else if(filu.has_id_arg("line", 2))
    {
      this->add_face(filu.get_int(0), filu.get_int(1));
    }
    else if(filu.has_id_arg("triangle", 3))
    {
      this->add_face(filu.get_int(0), filu.get_int(1), filu.get_int(2));
    }
    else if(filu.has_id_arg("quad", 4))
    {
      this->add_face(filu.get_int(0), filu.get_int(1), filu.get_int(2),
	  filu.get_int(3));
    }
    else
    {
      filu.warn_empty();
    }
  }

  // Now, if scale to x length has been specified, walk through all vertices,
  // determine real scale, then scale to approperiate lenght.
  if((scaleto >= 0) && (scaleto <= 2))
  {
    this->scale_to(scaleto, scaleto_val);
  }

  // Echo success.
  std::cout << "Loaded model \"" << filename << "\" with ";
  if(this->texture_file.length() > 0)
  {
    std::cout << "T(\"" << this->texture_file << "\") ";
  }
  std::cout << "V(" << static_cast<uint32_t>(this->get_num_vertex()) <<
    ") F(" << static_cast<uint32_t>(this->get_num_face()) << ")\n";
}

/** Default destructor. */
PreModel::~PreModel()
{
  // Clear all faces and vertices.
  for(std::vector<ComponentFace*>::iterator i = this->face_array.begin(),
      e = this->face_array.end(); (i != e); ++i)
  {
    delete *i;
  }
  // Delete all edges.
  for(EDGEMAP::iterator i = this->edge_map.begin(),
      e = this->edge_map.end(); (i != e); ++i)
  {
    delete (*i).second;
  }
  // Delete all vertices.
  for(std::vector<ComponentVertex*>::iterator i = this->vertex_array.begin(),
      e = this->vertex_array.end(); (i != e); ++i)
  {
    delete *i;
  }
}

//############################################################################
// Inline ####################################################################
//############################################################################

/** Add an element from an array - priority queue -combo.
 * @param array Reference to array.
 * @param queue Reference to the queue.
 * @param op Pointer to the element to add.
 */
template <class Type>
static inline void component_add(std::vector<Type> &array,
    std::priority_queue<size_t> &queue, Type op)
{
  // If queue points anywhere, insert stuff there.
  if(!queue.empty())
  {
    size_t idx = queue.top();

    array[idx] = op;
    op->set_idx(idx);

    queue.pop();
  }

  // Otherwise just add to the array.
  array.push_back(op);
  op->set_idx(array.size() - 1);
}

/** Remove an element from the array - priority queue -combo.
 * @param array Reference to array.
 * @param queue Reference to the queue.
 * @param op Index of the element to remove.
 */
template <class Type>
static inline void component_remove(std::vector<Type*> &array,
    std::priority_queue<size_t> &queue, size_t idx)
{
  Type *comp = array[idx];

  // If the last, remove from the array
  if(idx == array.size() - 1)
  {
    array.pop_back();

    // If was also marked in the queue, remove from there too.
    if(queue.top() == idx)
    {
      queue.pop();
    }

    // Remove while queue matches end.
    while((!queue.empty()) && (queue.top() == (array.size() - 1)))
    {
      array.pop_back();
      queue.pop();
    }
  }
  // Otherwise insert null and push position to queue.
  else
  {
    array[idx] = NULL;
    queue.push(idx);
  }

  // Delete in the end.
  delete comp;
}

//############################################################################
// Searching #################################################################
//############################################################################

/** Find edge by two face indices.
 * @param idx1 Vertex index 1.
 * @param idx2 Vertex index 2.
 * @return The iterator or end of edge map if not found.
 */
PreModel::EDGEMAP::iterator PreModel::find_edge(size_t idx1, size_t idx2)
{
  ComponentVertex *ver1, *ver2;
  std::pair<EDGEMAP::iterator, EDGEMAP::iterator> range;

  cmpswap(idx1, idx2);
  ver1 = get_vertex(idx1);
  ver2 = get_vertex(idx2);

  // If either not found.
  if((!ver1) || (!ver2))
  {
    return edge_map.end();
  }

  // Get range
  range = edge_map.equal_range(reinterpret_cast<size_t>(ver1));

  // Initialize to begin, then iterate.
  for(EDGEMAP::iterator iter = range.first; (iter != range.second); ++iter)
  {
    ComponentEdge *edge = (*iter).second;

    // If the iterator is found, return it.
    if(edge->has_children(ver1, ver2))
    {
      return iter;
    }
  }

  return edge_map.end();
}

//############################################################################
// Removal ###################################################################
//############################################################################

/** Remove one vertex.
 * @param idx Vertex index.
 */
void PreModel::remove_vertex(size_t idx)
{
  ComponentVertex *comp = get_vertex(idx);

  if(!comp)
  {
    return;
  }

  // Remove all parents of the vertex.
  while(comp->has_parents())
  {
    Component *parent = comp->get_parent(comp->num_parents() - 1);

    // Removal depends on type, note that removing these will also remove
    // the parent data from this vertex.
    if(parent->is_face())
    {
      this->remove_face(parent->get_idx());
    }
    // Edges are tricky.
    else if(parent->get_type() == COMPONENT_EDGE)
    {
      ComponentEdge *edge = static_cast<ComponentEdge*>(parent);
      this->remove_edge(edge->get_child(0)->get_idx(),
	  edge->get_child(1)->get_idx());
    }
    else
    {
      std::cerr << "Error: Illegal vertex parent.\n";
    }
  }

  component_remove(vertex_array, vertex_queue, idx);
}

/** Remove one edge.
 * @param idx1 Vertex index 1.
 * @param idx2 Vertex index 2.
 */
void PreModel::remove_edge(size_t idx1, size_t idx2)
{
  EDGEMAP::iterator iter = find_edge(idx1, idx2);

  if(iter == edge_map.end())
  {
    return;
  }

  // Remove all parents of the
  ComponentEdge *edge = (*iter).second;
  while(edge->has_parents())
  {
    Component *parent = edge->get_parent(edge->num_parents() - 1);

    // Note that this will also remove the parent data from the edge.
    if(parent->is_face())
    {
      this->remove_face(parent->get_idx());
    }
    else
    {
      std::cerr << "Error: Illegal edge parent.\n";
    }
  }

  delete edge;
  edge_map.erase(iter);
}

/** Remove one face. Also remove it from the face forget list if it was there.
 * @param idx [in] Face index.
 */
void PreModel::remove_face(size_t idx)
{
  if(get_face(idx))
  {
    component_remove(face_array, face_queue, idx);

    // If we were forgetting this face, we need not do so anymore.
    HASHMAP<size_t, bool>::iterator iter = this->forget.find(idx);
    if(iter != this->forget.end())
    {
      this->forget.erase(iter);
    }
  }
}

/** Clean up unneccessary data accumulated in this premodel.
 * Basically, removes unneccessary edges and removes gaps by moving stuff
 * back in the arrays.
 */
void PreModel::cleanup()
{
  // Remove all forgotten faces.
  while(!this->forget.empty())
  {
    HASHMAP<size_t, bool>::iterator iter = this->forget.begin();
    this->remove_face((*iter).first);
  }

  // Clean up the face array.
  while(!face_queue.empty())
  {
    this->face_array.erase(this->face_array.begin() +
	this->face_queue.top());
    this->face_queue.pop();
  }
  for(size_t i = 0; (i < this->face_array.size()); ++i)
  {
    this->face_array[i]->set_idx(i);
  }

  // Remove all unneccessary edges.
  for(EDGEMAP::iterator iter = this->edge_map.begin();
      (iter != this->edge_map.end());)
  {
    EDGEMAP::iterator curr = iter;
    ++iter;

    if(!(*curr).second->has_parents())
    {
      delete (*curr).second;
      this->edge_map.erase(curr);
    }
  }

  // Remove vertices with no parents (since edges are already removed, this
  // only leaves vertices that had no face parents to begin with).
  for(size_t i = 0; (i < this->vertex_array.size()); ++i)
  {
    ComponentVertex *ver = this->vertex_array[i];

    if((ver) && (!ver->has_parents()))
    {
      component_remove(this->vertex_array, this->vertex_queue, i);
    }
  }

  // Clean up the vertex array.
  while(!this->vertex_queue.empty())
  {
    this->vertex_array.erase(this->vertex_array.begin() +
	this->vertex_queue.top());
    this->vertex_queue.pop();
  }
  for(size_t i = 0; (i < this->vertex_array.size()); ++i)
  {
    this->vertex_array[i]->set_idx(i);
  }

}

//############################################################################
// Additions #################################################################
//############################################################################

/** Add one vertex to this model.
 * @param pos Position.
 * @param col Color.
 * @return The vertex added.
 */
ComponentVertex* PreModel::add_vertex(const Vector3& pos, const Color4& col,
    const Vector2 &tex)
{
  ComponentVertex *ver = new ComponentVertex(pos, col, tex);

  component_add(vertex_array, vertex_queue, ver);

  return ver;
}

/** Add a new edge to this model.
 * @param idx1 First vertex index.
 * @param idx2 Second vertex index.
 * @return The added edge or NULL on error.
 */
ComponentEdge* PreModel::add_edge(size_t idx1, size_t idx2)
{
  ComponentVertex *ver1, *ver2;
  ComponentEdge *edg;
  std::pair<size_t, ComponentEdge*> pair;
 
  // Try to find edge.
  edg = get_edge(idx1, idx2);
  if(edg)
  {
    return edg;
  }

  // If not found, get vertices.
  cmpswap(idx1, idx2);
  ver1 = get_vertex(idx1);
  ver2 = get_vertex(idx2);

  // If either vertex not found, abort.
  if((ver1 == NULL) || (ver2 == NULL))
  {
    return NULL;
  }

  // Since everything is okay, insert the edge.
  edg = new ComponentEdge(ver1, ver2);
  pair.first = reinterpret_cast<size_t>(ver1);
  pair.second = edg;
  edge_map.insert(pair);

  return edg;
}

/** Add a new face (voxel) to this model.
 * @param idx The reference vertex.
 * @return The added face or NULL on error.
 */
ComponentFace* PreModel::add_face(size_t idx)
{
  ComponentVertex *ver = get_vertex(idx);
  ComponentFace *fac;

  if(ver == NULL)
  {
    return NULL;
  }

  fac = new ComponentFace(ver);
  component_add(face_array, face_queue, fac);

  if(this->has_forget())
  {
    this->forget[fac->get_idx()] = true;
  }

  return fac;
}

/** Add a new face (line) to this model.
 * @param idx1 First vertex index.
 * @param idx2 Second vertex index.
 * @return The added face or NULL on error.
 */
ComponentFace* PreModel::add_face(size_t idx1, size_t idx2)
{
  ComponentVertex *ver1 = get_vertex(idx1),
		  *ver2 = get_vertex(idx2);
  ComponentEdge *edg;
  ComponentFace *fac;

  if((ver1 == NULL) || (ver2 == NULL))
  {
    return NULL;
  }

  edg = add_edge(idx1, idx2);
  fac = new ComponentFace(ver1, ver2);
  fac->set_edge(edg);
  component_add(face_array, face_queue, fac);

  if(this->has_forget())
  {
    this->forget[fac->get_idx()] = true;
  }

  return fac;
}

/** Add a new face (triangle) to this model.
 * @param idx1 First vertex index.
 * @param idx2 Second vertex index.
 * @param idx3 Third vertex index.
 * @return The added face or NULL on error.
 */
ComponentFace* PreModel::add_face(size_t idx1, size_t idx2, size_t idx3)
{
  ComponentVertex *ver1 = get_vertex(idx1),
		  *ver2 = get_vertex(idx2),
		  *ver3 = get_vertex(idx3);
  ComponentEdge *edg1, *edg2, *edg3;
  ComponentFace *fac;

  if((ver1 == NULL) || (ver2 == NULL) || (ver3 == NULL))
  {
    return NULL;
  }

  edg1 = add_edge(idx1, idx2);
  edg2 = add_edge(idx2, idx3);
  edg3 = add_edge(idx3, idx1);
  fac = new ComponentFace(ver1, ver2, ver3);
  fac->set_edge(edg1, edg2, edg3);
  component_add(face_array, face_queue, fac);

  if(this->has_forget())
  {
    this->forget[fac->get_idx()] = true;
  }

  return fac;
}

/** Add a new face (quad) to this model.
 * @param idx1 First vertex index.
 * @param idx2 Second vertex index.
 * @param idx3 Third vertex index.
 * @param idx4 Fourth vertex index.
 * @return The added face or NULL on error.
 */
ComponentFace* PreModel::add_face(size_t idx1, size_t idx2, size_t idx3,
    size_t idx4)
{
  ComponentVertex *ver1 = get_vertex(idx1),
		  *ver2 = get_vertex(idx2),
		  *ver3 = get_vertex(idx3),
		  *ver4 = get_vertex(idx4);
  ComponentEdge *edg1, *edg2, *edg3, *edg4;
  ComponentFace *fac;

  if((ver1 == NULL) || (ver2 == NULL) || (ver3 == NULL) || (ver4 == NULL))
  {
    return NULL;
  }

  edg1 = add_edge(idx1, idx2);
  edg2 = add_edge(idx2, idx3);
  edg3 = add_edge(idx3, idx4);
  edg4 = add_edge(idx4, idx1);
  fac = new ComponentFace(ver1, ver2, ver3, ver4);
  fac->set_edge(edg1, edg2, edg3, edg4);
  component_add(face_array, face_queue, fac);

  if(this->has_forget())
  {
    this->forget[fac->get_idx()] = true;
  }

  return fac;
}
   
//############################################################################
// Other big methods #########################################################
//############################################################################

/** Scales the whole model (all vertices of) to a given length in given axis.
 * @param axis Axis to scale according to.
 * @param scaleto Value to scale to.
 */
void PreModel::scale_to(int axis, float scaleto)
{
  float maxv = -FLT_MAX,
	minv = FLT_MAX;

  // Loop and get minimum and maximum.
  for(std::vector<ComponentVertex*>::iterator i = this->vertex_array.begin(),
      e = this->vertex_array.end(); (i != e); ++i)
  {
    float val = (*i)->get_pos().f[axis];

    maxv = stdmax(maxv, val);
    minv = stdmin(minv, val);
  }

  // Determine real scaling factor.
  scaleto /= (maxv - minv);

  // Loop again.
  for(std::vector<ComponentVertex*>::iterator i = this->vertex_array.begin(),
      e = this->vertex_array.end(); (i != e); ++i)
  {
    libfhi::Vector3 spos = (*i)->get_pos() * scaleto;

    (*i)->set_pos(spos);
  }
}

//############################################################################
// Compiler ##################################################################
//############################################################################

/** Write vertex data into target mesh. Will also precalculate normals and
 * stuff like that. If the given name is empty, dies not save the produced
 * mesh in the canonical database.
 *
 * To ease the generation of larger objects and allow for objects that take
 * lighting data also from polygons that are not included into the final mesh
 * (for example, when tiling), the compilation allows for a 'forget' list.
 * This forget list contains names of all the faces not to be included in the
 * compiled object. Those faces will also be destroyed from this model and
 * removed from the forget list, which means that further compilations will
 * produce the same mesh, but with different lighting.
 *
 * PreModels are not intended to be compiled multiple times.
 * @param str The name string to be used for canonical database.
 * @return New mesh object compiled from the data or NULL if no data.
 */
Mesh* PreModel::compile(const char *str)
{
  Mesh *ret;
  MESH_TYPEDEF_INDEX piles[MESH_ELEMENT_COUNT] = { 0, 0, 0, 0, 0, 0, 0 },
	  	     *epos[MESH_ELEMENT_COUNT];

  // Check if no vertices or no faces, in this case, just destroy the mesh.
  if((vertex_array.size() <= 0) || (face_array.size() <= 0))
  {
    return NULL;
  }

  // Get the mesh from canondb if string exists.
  if(str != NULL)
  {
    ret = Mesh::canon_new(str);
  }
  else
  {
    ret = new Mesh();
  }

  // Exit on error.
  if(ret == NULL)
  {
    return NULL;
  }

  // Set the texture on the mesh. If open fails, no harm done.
  if(this->texture_file.length() > 0)
  {
    ret->set_texture(texture_new(this->texture_file.c_str()));
  }

  // Calculate the normal for all faces.
  for(std::vector<ComponentFace*>::iterator i = this->face_array.begin(),
      e = this->face_array.end(); (i != e); ++i)
  {
    (*i)->calculate_normal();
  }

  // Iterate through vertices, calculate their normals and copy them to
  // destination data.
  for(std::vector<ComponentVertex*>::iterator i = this->vertex_array.begin(),
      e = this->vertex_array.end(); (i != e); ++i)
  {
    (*i)->calculate_normal();
  }

  // Perform cleanup. Will also remove forgotten faces and useless vertices.
  this->cleanup();

  // Reserve vertex data.
  ret->reserve_vertex(vertex_array.size());

  // Get the newly reserved arrays.
  Color4 *ar_col = ret->get_array_color();
  Vector3 *ar_ver = ret->get_array_vertex(),
	  *ar_vnor = ret->get_array_vnormal();
  Vector2 *ar_tex = ret->get_array_texcoord();

  // Write all remaining vertices into the mesh.
  for(size_t i = 0; (i < vertex_array.size()); ++i)
  {
    ComponentVertex *ver = this->vertex_array[i];

    ar_ver[i] = ver->get_pos();
    ar_col[i] = ver->get_col();
    ar_vnor[i] = ver->get_nor();

    // Only fill if texture was successfully set.
    if(ret->has_texture())
    {
      ar_tex[i] = ver->get_tex();
    }
  }

  // Perform attribute check for all faces.
  for(std::vector<ComponentFace*>::iterator i = this->face_array.begin(),
      e = this->face_array.end(); (i != e); ++i)
  {
    (*i)->calculate_attributes();
  }

  // Perform attribute check for all edges.
  for(EDGEMAP::iterator i = this->edge_map.begin(),
      e = this->edge_map.end(); (i != e); ++i)
  {
    (*i).second->calculate_attributes();
  }

  // Count all faces and radix sort into piles.
  if(!this->has_wireframe())
  {
    for(std::vector<ComponentFace*>::iterator i = this->face_array.begin(),
	e = this->face_array.end(); (i != e); ++i)
    {
      int idx, cnt = 0;

      switch((*i)->get_type())
      {
	case COMPONENT_VOXEL:
	  idx = MESH_POINTS;
	  cnt = 1;
	  break;
	case COMPONENT_LINE:
	  idx = ((*i)->has_flat() && !this->has_denyflat()) ?
	    MESH_FLAT_LINES : MESH_LINES;
	  cnt = 2;
	  break;
	case COMPONENT_TRIANGLE:
	  idx = ((*i)->has_flat() && !this->has_denyflat()) ?
	    MESH_FLAT_TRIANGLES : MESH_TRIANGLES;
	  cnt = 3;
	  break;
	case COMPONENT_QUAD:
	  idx = ((*i)->has_flat() && !this->has_denyflat()) ?
	    MESH_FLAT_QUADS : MESH_QUADS;
	  cnt = 4;
	  break;
	default:
	  std::cerr << "Warning: Invalid face detected: " <<
	    static_cast<unsigned>((*i)->get_idx()) << "\n";
	  break;
      }

      // Only add on legal faces.
      if(cnt > 0)
      {
	piles[idx] += cnt;
      }
    }
  }
  // If need for wireframe, instead romp through edges.
  else
  {
    // Add edges.
    for(EDGEMAP::iterator i = this->edge_map.begin(),
	e = this->edge_map.end(); (i != e); ++i)
    {
      int idx = MESH_LINES;

      if((*i).second->has_flat() && !this->has_denyflat())
      {
	idx = MESH_FLAT_LINES;
      }

      piles[idx] += 2;
    }
  }

  // Reserve according to pile sizes.
  ret->reserve_face(piles);
  for(int i = 0; (i < static_cast<int>(MESH_ELEMENT_COUNT)); ++i)
  {
    epos[i] = ret->get_array_elem(static_cast<ElementType>(i)) + 1;
  }

  // Second pass to write into index arrays.
  if(!this->has_wireframe())
  {
    for(std::vector<ComponentFace*>::iterator i = this->face_array.begin(),
	e = this->face_array.end(); (i != e); ++i)
    {
      int idx, cnt = 0;

      switch((*i)->get_type())
      {
	case COMPONENT_VOXEL:
	  idx = MESH_POINTS;
	  cnt = 1;
	  break;
	case COMPONENT_LINE:
	  idx = ((*i)->has_flat() && !this->has_denyflat()) ?
	    MESH_FLAT_LINES : MESH_LINES;
	  cnt = 2;
	  break;
	case COMPONENT_TRIANGLE:
	  idx = ((*i)->has_flat() && !this->has_denyflat()) ?
	    MESH_FLAT_TRIANGLES : MESH_TRIANGLES;
	  cnt = 3;
	  break;
	case COMPONENT_QUAD:
	  idx = ((*i)->has_flat() && !this->has_denyflat()) ?
	    MESH_FLAT_QUADS : MESH_QUADS;
	  cnt = 4;
	  break;
	default:
	  std::cerr << "Warning: Invalid face detected: " <<
	    static_cast<unsigned>((*i)->get_idx()) << "\n";
	  break;
      }

      // Copy indexes themselves
      if(cnt > 0)
      {
	int j = 0;
	do {
	  *(epos[idx]++) = static_cast<MESH_TYPEDEF_INDEX>(
		(*i)->get_vertex(j++)->get_idx());
	} while(--cnt);
      }
    }
  }
  // If writing in wireframe, write edges instead.
  else
  {
    // Add edges.
    for(EDGEMAP::iterator i = this->edge_map.begin(),
	e = this->edge_map.end(); (i != e); ++i)
    {
      int idx = MESH_LINES;
      ComponentEdge *edg = (*i).second;

      if(edg->has_flat() && !this->has_denyflat())
      {
	idx = MESH_FLAT_LINES;
      }

      *(epos[idx]++) = static_cast<MESH_TYPEDEF_INDEX>(
	  edg->get_vertex(0)->get_idx());
      *(epos[idx]++) = static_cast<MESH_TYPEDEF_INDEX>(
	  edg->get_vertex(1)->get_idx());
    }
  }

  return ret;
}

//############################################################################
// Debug #####################################################################
//############################################################################

#ifdef LIBFHI_DEBUG

/** Print this to given strea.
 * @param s Stream to output to.
 */
std::ostream& PreModel::print(std::ostream &s) const
{
  s << "Model, " << static_cast<uint32_t>(get_num_face()) << " faces:";

  for(size_t i = 0; i < get_num_face(); i++)
  {
    s << "\n" << *(face_array[i]);
  }

  return s;
}

#endif

//############################################################################
// Loppu #####################################################################
//############################################################################

