#ifndef libfhi_singleton_include
#define libfhi_singleton_include

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi_default.h"

namespace libfhi {

//############################################################################
// Class #####################################################################
//############################################################################

/** Singleton describes a singleton class that does not exist until referenced
 * and fetched from get_instance(). All singleton classes must have a default
 * empty constructor (that is called upon the first fetch). The instance is
 * public, it can be referenced directly if need be, however, this must be
 * done ONLY when it is certain that the instace has already been reserved
 * (and only on very, very time-critical situations). For the most part, using
 * get_instance() is just fine.
 *
 * Another note, please do not call instance_get when wishing to generate a
 * new instance to a type derieved from a type inheriting singleton, please
 * do not use instance_get, it will generate a new instance of the base class.
 * Instead, use instance_new with a new object parameter. After this, you
 * may use instance_get as usual.
 */
template <class Type>
class Singleton
{
  public:
    static Type *instance;

  public:
    static inline void instance_delete();
    static inline Type* instance_get();
    static inline Type* instance_new(Type*);
};

//############################################################################
// Globals ###################################################################
//############################################################################

template <class Type>
Type *Singleton<Type>::instance = NULL;

//############################################################################
// Methods ###################################################################
//############################################################################

/** Destroy the instance. Should not normally be neccessary.
 */
template <class Type>
void Singleton<Type>::instance_delete()
{
  zdel(Singleton<Type>::instance);
}

/** Get the instance, if it does not yet exist, create it.
 * @return Pointer to the instance.
 */
template <class Type>
Type* Singleton<Type>::instance_get()
{
  if(!Singleton<Type>::instance)
  {
    Singleton<Type>::instance = new Type();
  }

  return Singleton<Type>::instance;
}

/** Store a new instance into the instance slot, this can be of a type
 * derieved from the original.
 * @param op The new instance.
 * @return The new instance.
 */
template <class Type>
Type* Singleton<Type>::instance_new(Type *op)
{
  delete Singleton<Type>::instance;

  Singleton<Type>::instance = op;

  return op;
}

//############################################################################
// End #######################################################################
//############################################################################

}
#endif

