#include "libfhi_font.h"
#include "libfhi_misc.h"
#include "libfhi_surface.h"

#include <iostream>

// Freetype
#include <ft2build.h>
#include FT_FREETYPE_H
#include FT_GLYPH_H

namespace libfhi {

//############################################################################
// Glyph construction ########################################################
//############################################################################

/** Default constructor.
 * @param op Void pointer to the glyph structure in a FT_Face.
 */
Glyph::Glyph(void *op)
{
  FT_GlyphSlot glp = static_cast<FT_GlyphSlot>(op);
  int data_size;

  // Get parameters.
  this->w = glp->bitmap.width;
  this->h = glp->bitmap.rows;
  this->lt = glp->bitmap_left;
  this->tp = glp->bitmap_top;
  this->ax = glp->advance.x >> 6;
  this->ay = glp->advance.y >> 6;

  data_size = this->w * this->h;
  if(data_size > 0)
  {
    this->data = new uint8_t[data_size];

    for(int i = 0; i < this->w; ++i)
    {
      for(int j = 0; j < this->h; ++j)
      {
	uint8_t *dst = this->data + j * this->w + i;

	if(glp->bitmap.buffer[j * glp->bitmap.pitch + i / 8] & (128 >> i % 8))
	{
	  *dst = 0x01;
	}
	else
	{
	  *dst = 0x00;
	}
      }
    }
  }
  else
  {
    this->data = NULL;
  }

#ifdef LIBFHI_OPENGL
  // If OpenGl is enabled, each glyph has one texture representing it. Mark
  // the texture unused upon initialization.
  this->texture_name = 0;
#endif
}

/** Default destructor. */
Glyph::~Glyph()
{
  // Free glyph picture data.
  if(this->data)
  {
    delete this->data;
  }

#ifdef LIBFHI_OPENGL
  // Free texture name if it was used.
  if(this->texture_name)
  {
    glDeleteTextures(1, &(this->texture_name));
  }
#endif
}

//############################################################################
// Glyph conditional methods #################################################
//############################################################################

#ifdef LIBFHI_OPENGL

/** Generate a new OpenGL texture for this Glyph. This method is intended to be
 * run in a 'lazy' evaluation mode where characters are cached into textures
 * whenever they are needed. Whatever slowdown happens, happens only once, thus,
 * this method is allowed to be slow.
 * @return Name of the newly generated texture.
 */
GLuint Glyph::generate_texture()
{
  // We need to find the next exponent of 2 to expand the texture correctly into
  // 2^n -dimensions.
  float lx = static_cast<float>(this->w),
	ly = static_cast<float>(this->h);
  int nx, ny;

  // Assing integer values.
  if(this->w == 1)
  {
    nx = 2;
  }
  else
  {
    nx = static_cast<int>(ceil(log(static_cast<double>(lx)) / log(2.0)));
    nx = static_cast<int>(powf(2.0f, static_cast<float>(nx)));
  }

  if(this->h == 1)
  {
    ny = 2;
  }
  else
  {
    ny = static_cast<int>(ceil(log(static_cast<double>(ly)) / log(2.0)));
    ny = static_cast<int>(powf(2.0f, static_cast<float>(ny)));
  }

  if((this->w < 1) || (nx < 2) || (this->h < 1) || (ny < 2))
  {
    return 0;
  }

  // Set the maximum texcoords in order for GL_LINEAR to work correctly.
  this->tmx = static_cast<float>(this->w) / static_cast<float>(nx);
  this->tmy = static_cast<float>(this->h) / static_cast<float>(ny);

  // Reserve rgba data array.
  uint32_t *rgbadata = new uint32_t[nx * ny];

  // Copy the font into the upper right corner of the bitmap.
  for(int j = 0; (j < ny); ++j)
  {
    for(int i = 0; (i < nx); ++i)
    {
      int idxl = j * nx + i;

      if((i < this->w) && (j < this->h))
      {
	int idxs = j * this->w + i;

	if(this->data[idxs] != 0)
	{
	  rgbadata[idxl] = 0xFFFFFFFF;
	}
	else
	{
	  rgbadata[idxl] = 0x00000000;
	}
      }
      else
      {
	rgbadata[idxl] = 0x00000000;
      }
    }
  }

  // If we arrive at this point, there really is no texture, safly create new.
  glEnable(GL_TEXTURE_2D);
  glGenTextures(1, &(this->texture_name));
  glBindTexture(GL_TEXTURE_2D, this->texture_name);

  // If strange errors appear, try uncommenting lines below.
  glPixelStorei(GL_PACK_ALIGNMENT, 1);
  glPixelStorei(GL_PACK_ROW_LENGTH, nx);
  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
  glPixelStorei(GL_UNPACK_ROW_LENGTH, nx);
  
  // Set mipmapping.
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

  // Create a new texture.
  gluBuild2DMipmaps(GL_TEXTURE_2D, 4, nx, ny, GL_RGBA, GL_UNSIGNED_BYTE,
      rgbadata);

  // Can not disable textures, as the textout might still be on. However,
  // free the data used to generate the texture.
  delete[] rgbadata;

  return this->texture_name;
}

#endif

//############################################################################
// Variables #################################################################
//############################################################################

/** Handle to the FreeType library. */
FT_Library ft_handle;

//############################################################################
// Font ######################################################################
//############################################################################

/** Default constructor. */
Font::Font()
{
  this->face = NULL;
}

/** Default destructor. */
Font::~Font()
{
  // Delete all glyphs.
  this->clear_cache();

  // Delete face.
  if(this->face)
  {
    FT_Done_Face(static_cast<FT_Face>(this->face));
  }
}

/** Clear accumulated glyph data.
 */
void Font::clear_cache()
{
  for(HASHMAP<int, Glyph*>::iterator i = glyphs.begin(), e = glyphs.end();
      (i != e); ++i)
  {
    delete (*i).second;
  }

  this->glyphs.clear();
}

/** Get a given glyph from this font structure. Load it if does not exist.
 * Note that the bitmaps are cleared when the font size changes, so this does
 * not have to check for correct size, if the glyph is found, return it.
 * @param idx Unicode index of the glyph.
 * @return Libfhi Glyph struct as void pointer.
 */
Glyph* Font::get_glyph(int idx)
{
  HASHMAP<int, Glyph*>::iterator iter =
    this->glyphs.find(idx);

  // If found, return right away.
  if(iter != this->glyphs.end())
  {
    return (*iter).second;
  }

  // Otherwise, generate.
  FT_Face ftface = static_cast<FT_Face>(this->face);

  int glyph_index = FT_Get_Char_Index(ftface, idx);
  Glyph *ret;
    
  // Glyph not found.
  if(FT_Load_Glyph(ftface, glyph_index, FT_LOAD_DEFAULT))
  {
    std::cerr << "Warning: Could not locate glyph " << idx << ".\n";
    return NULL;
  }

  // Render the glyph.
  if(FT_Render_Glyph(ftface->glyph, FT_RENDER_MODE_MONO))
  {
    std::cerr << "Warning: Could not render glyph " << idx << ".\n";
  }

  // Construct a new libfhi Glyph object, store and return it.
  ret = new Glyph(ftface->glyph);
  this->glyphs[idx] = ret;
  return ret;
}

/** Load a font into this. You should delete this Font on error.
 * @param filename Name of the file.
 * @return True on success, false on failure.
 */
bool Font::load(const char *filename)
{
  int error = FT_New_Face(ft_handle, filename, 0,
      reinterpret_cast<FT_Face*>(&(this->face)));

  // Process FreeType error.
  switch(error)
  {
    case 0: // All ok.
      break;
    case FT_Err_Unknown_File_Format:
      std::cerr << "Error: Font \"" << filename <<
	"\" of unknown format.\n";
      return false;
    default:
      std::cerr << "Error: Font \"" << filename <<
	"\" cannot be found or broken.\n";
      return false;
  }

  return true;
}

/** Get the space used by a certain string.
 * @param text Text string.
 */
int Font::get_string_width(const char *text)
{
  char *str = const_cast<char*>(text);
  int cumulative_width = 0;

  while(*str)
  {
    int unicode = strdecodeutf8(str);
    Glyph *glyph = this->get_glyph(unicode);

    if(!glyph)
    {
      continue;
    }

    cumulative_width += glyph->get_advance_x();
  }

  return cumulative_width;
}

/** Set the font size. If different from current, will clear all glyph data.
 * @param w Width.
 * @param h Heigth.
 * @return True on success, false on failure.
 */
bool Font::set_size(int w, int h)
{
  FT_Face ftface = static_cast<FT_Face>(this->face);

  int oldw = ftface->size->metrics.x_scale,
      oldh = ftface->size->metrics.y_scale;

  if(FT_Set_Pixel_Sizes(ftface, w, h))
  {
    return false;
  }

  // Clear cache when changing size.
  if((oldw != ftface->size->metrics.x_scale) ||
      (oldh != ftface->size->metrics.y_scale))
  {
    this->clear_cache();
  }

  return true;  
}

//############################################################################
// Font loading ##############################################################
//############################################################################

/** Loads a font and saves it into the array of fonts.
 * @param filename Name of font to load.
 * @return Pointer to newly reserved font object or NULL on error.
 */
Font* font_new(const char *filename)
{
  Font *ret = new Font();

  // Try to load.
  if(!ret->load(filename))
  {
    delete ret;
    return NULL;
  }

  // Echo if debug.
#ifdef LIBFHI_DEBUG
  std::cout << "Font loaded from \"" << filename << "\"\n";
#endif
  return ret;
}

/** Initialize the fonts.
 * @return True if all went well, false otherwise.
 */
bool font_init()
{
  // Initialize FreeType.
  if(FT_Init_FreeType(&ft_handle))
  {
    std::cerr << "Error: Could not init FreeType.";
    return false;
  }

  return true;
}

/** Shuts down the font subsystem.
 * @return True if all went well, false otherwise.
 */
bool font_quit()
{
  Font::canon_clear();

  // Done with FreeType
  FT_Done_FreeType(ft_handle);

  return true;
}

//############################################################################
// Loppu #####################################################################
//############################################################################

}

