#include "libfhi_component.h"

#include <iostream>

using namespace libfhi;

//############################################################################
// Component (basic) #########################################################
//############################################################################

/** Default constructor.
 * @param op_type [in] Type of the new component.
 */
Component::Component(ComponentType op_type)
: type(op_type)
{
  return;
}

/** Refined constructor.
 * @param op_pos [in] Position of the new component.
 * @param op_type [in] Type of the new component.
 */
Component::Component(const Vector3& op_pos, ComponentType op_type)
: type(op_type)
{
  set_pos(op_pos);
}

/** Destructor. */
Component::~Component()
{
  for(std::vector<Component*>::iterator i = this->parents.begin(),
      e = this->parents.end(); (i != e); ++i)
  {
    (*i)->remove_child(this);
  }
}

//############################################################################
// Vertex ####################################################################
//############################################################################

/** Default constructor. Alpha needs not be specified (defaults to 1.0f).
 * @param op_pos Position.
 * @param op_col Color.
 * @param op_tex Texture coordinates.
 */
ComponentVertex::ComponentVertex(const Vector3& op_pos, const Color4& op_col,
    const Vector2 &op_tex)
  : Component(op_pos, COMPONENT_VERTEX), col(op_col), texcoord(op_tex)
{
  // Do nothing.
}

/** Destructor.
 */
ComponentVertex::~ComponentVertex()
{
  // Do nothing.
}

/** Calculate the normal of this vertex. Requires all faces to have their
 * normal calculated already.
 */
void ComponentVertex::calculate_normal()
{
  std::vector<Vector3> array;

  // Iterate all faces linked to this vertex.
  for(std::vector<Component*>::iterator i = this->parents.begin(),
      e = this->parents.end(); (i != e); ++i)
  {
    bool found = false;
    ComponentFace *par;
    Vector3 nor;
      
    // Skip if not face.
    if(!(*i)->is_face())
    {
      continue;
    }
    par = static_cast<ComponentFace*>(*i);

    // Fetch normal from lookup table.
    nor = par->get_nor();

    // Search for similar normal in the normal list of this vertex.
    for(std::vector<Vector3>::iterator j = array.begin(),
	e2 = array.end(); (j != e2); ++j)
    {
      if((*j).is_same_direction(nor))
      {
	found = true;
	break;
      }
    }

    // If normal was not there, add it into the list.
    if(!found)
    {
      array.push_back(nor);
    }
  }

  // Sum the normal from all connected face normals.
  normal.set(0.0f, 0.0f, 0.0f);
  for(std::vector<Vector3>::iterator i = array.begin(),
      e = array.end(); (i != e); ++i)
  {
    normal += *i;
  }

  // Unit the vector.
  if(!normal.is_null())
  {
    normal.unit();
  }
  // Set to default if cannot unit.
  else
  {
    normal.set(0.0f, 1.0f, 0.0f);
  }
}

/** Vertices have no children, devoid.
 * @param Devoid.
 * @return False;
 */
bool ComponentVertex::has_child(const Component*) const
{
  return false;
}

/** Vertices have no children, devoid.
 * @param Devoid.
 * @return False.
 */
bool ComponentVertex::has_child(size_t) const
{
  return false;
}

/** Vertices have no children, devoid.
 * @param Devoid.
 */
void ComponentVertex::remove_child(Component*)
{
  return;
}

/** Vertices have nothing to update, devoid.
 */
void ComponentVertex::update()
{
  return;
}

//############################################################################
// Edge ######################################################################
//############################################################################

/** Default constructor.
 * Assumes order of vertices correct.
 * @param op1 [in] First vertex.
 * @param op2 [in] Second vertex.
 */
ComponentEdge::ComponentEdge(ComponentVertex *op1, ComponentVertex *op2)
: Component(COMPONENT_EDGE), Attributable(DEFAULT_ATTR)
{
  vertex[0] = op1;
  vertex[1] = op2;
  vertex[0]->add_parent(this);
  vertex[1]->add_parent(this);
  update();
}

/** Destructor.
 */
ComponentEdge::~ComponentEdge()
{
  if(vertex[0])
  {
    vertex[0]->remove_parent(this);
  }
  if(vertex[1])
  {
    vertex[1]->remove_parent(this);
  }
}

/** Recalculate the attributes of this edge.
 * Must be after all normals (for faces) are calculated.
 */
void ComponentEdge::calculate_attributes()
{
  // Set flatness.
  if((vertex[0]->get_col() == vertex[1]->get_col()) &&
      (vertex[0]->get_nor().is_same_direction(vertex[1]->get_nor())))
  {
    attr_or(ATTR_FLAT);
  }
  else
  {
    attr_not(ATTR_FLAT);
  }
}

/** Tell if this edge has given child.
 * @param op [in] Pointer to potential child.
 * @return True if yes, false if no.
 */
bool ComponentEdge::has_child(const Component *op) const
{
  return ((vertex[0] == op) || (vertex[1] == op));
}

/** Tell if this edge has child with given index.
 * @param child_idx [in] Potential child index.
 * @return True if yes, false if no.
 */
bool ComponentEdge::has_child(size_t child_idx) const
{
  return (vertex[0]->has_idx(child_idx) || vertex[1]->has_idx(child_idx));
}

/** Remove a given child from this edge.
 * @param op [in] Pointer to edge.
 */
void ComponentEdge::remove_child(Component *op)
{
  if(vertex[0] == op)
  {
    vertex[0]->remove_parent(this);
    vertex[0] = NULL;
  }
  else if(vertex[1] == op)
  {
    vertex[1]->remove_parent(this);
    vertex[1] = NULL;
  }
}

/** Update the position of this edge based on the average position of child
 * vertices.
 */
void ComponentEdge::update()
{
  pos.average(vertex[0]->get_pos(), vertex[1]->get_pos());
}

//############################################################################
// Face ######################################################################
//############################################################################

/** Constructor for voxel.
 * @param op1 [in] Vertex.
 */
ComponentFace::ComponentFace(ComponentVertex *op1)
: Component(COMPONENT_VOXEL), Attributable(DEFAULT_ATTR)
{
  edge[0] = edge[1] = edge[2] = edge[3] = NULL;
  vertex[1] = vertex[2] = vertex[3] = NULL;

  vertex[0] = op1;
  vertex[0]->add_parent(this);

  update();
}

/** Constructor for line.
 * @param op1 [in] Vertex 1.
 * @param op2 [in] Vertex 2.
 */
ComponentFace::ComponentFace(ComponentVertex *op1, ComponentVertex *op2)
: Component(COMPONENT_LINE), Attributable(DEFAULT_ATTR)
{
  edge[0] = edge[1] = edge[2] = edge[3] = NULL;
  vertex[2] = vertex[3] = NULL;

  vertex[0] = op1;
  vertex[1] = op2;
  vertex[0]->add_parent(this);
  vertex[1]->add_parent(this);

  update();
}

/** Constructor for triangle.
 * @param op1 [in] Vertex 1.
 * @param op2 [in] Vertex 2.
 * @param op3 [in] Vertex 3.
 */
ComponentFace::ComponentFace(ComponentVertex *op1, ComponentVertex *op2,
    ComponentVertex *op3)
: Component(COMPONENT_TRIANGLE), Attributable(DEFAULT_ATTR)
{
  edge[0] = edge[1] = edge[2] = edge[3] = NULL;
  vertex[3] = NULL;

  vertex[0] = op1;
  vertex[1] = op2;
  vertex[2] = op3;
  vertex[0]->add_parent(this);
  vertex[1]->add_parent(this);
  vertex[2]->add_parent(this);
  
  update();
}

/** Constructor for quad.
 * @param op1 [in] Vertex 1.
 * @param op2 [in] Vertex 2.
 * @param op3 [in] Vertex 3.
 */
ComponentFace::ComponentFace(ComponentVertex *op1, ComponentVertex *op2,
    ComponentVertex *op3, ComponentVertex *op4)
: Component(COMPONENT_QUAD), Attributable(DEFAULT_ATTR)
{
  edge[0] = edge[1] = edge[2] = edge[3] = NULL;

  vertex[0] = op1;
  vertex[1] = op2;
  vertex[2] = op3;
  vertex[3] = op4;
  vertex[0]->add_parent(this);
  vertex[1]->add_parent(this);
  vertex[2]->add_parent(this);
  vertex[3]->add_parent(this);

  update();
}

/** Destructor
 */
ComponentFace::~ComponentFace()
{
  for(int i = 0; (i < 4); ++i)
  {
    if(edge[i] != NULL)
    {
      edge[i]->remove_parent(this);
    }
    if(vertex[i] != NULL)
    {
      vertex[i]->remove_parent(this);
    }
  }
}

/** Set edge data for this.
 * @param op [in] Edge.
 */
void ComponentFace::set_edge(ComponentEdge *op)
{
  edge[0] = op;
  edge[0]->add_parent(this);
}

/** Set edge data for this.
 * @param op1 [in] Edge 1.
 * @param op2 [in] Edge 2.
 * @param op3 [in] Edge 3.
 */
void ComponentFace::set_edge(ComponentEdge *op1, ComponentEdge *op2,
    ComponentEdge *op3)
{
  edge[0] = op1;
  edge[1] = op2;
  edge[2] = op3;
  edge[0]->add_parent(this);
  edge[1]->add_parent(this);
  edge[2]->add_parent(this);
}

/** Set edge data for this.
 * @param op1 [in] Edge 1.
 * @param op2 [in] Edge 2.
 * @param op3 [in] Edge 3.
 * @param op4 [in] Edge 4.
 */
void ComponentFace::set_edge(ComponentEdge *op1, ComponentEdge *op2,
    ComponentEdge *op3, ComponentEdge *op4)
{
  edge[0] = op1;
  edge[1] = op2;
  edge[2] = op3;
  edge[3] = op4;
  edge[0]->add_parent(this);
  edge[1]->add_parent(this);
  edge[2]->add_parent(this);
  edge[3]->add_parent(this);
}

/** Tell if this face has the given child by address.
 * @param op [in] The child to search for.
 * @return True if yes, false if no.
 */
bool ComponentFace::has_child(const Component *op) const
{
  for(int i = 0; i < 4; ++i)
  {
    if((edge[i] == op) || (vertex[i] == op))
    {
      return true;
    }
  }

  return false;
}

/** Tell if this face has the given child by index.
 * @param child_idx [in] The index to search for.
 * @return True if found, false if not.
 */
bool ComponentFace::has_child(size_t child_idx) const
{
  for(int i = 0; i < 4; ++i)
  {
    if(edge[i] != NULL)
    {
      if(edge[i]->get_idx() == child_idx)
      {
	return true;
      }
    }
    if(vertex[i] != NULL)
    {
      if(vertex[i]->get_idx() == child_idx)
      {
	return true;
      }
    }
  }

  return false;
}

/** Remove the given child from this face.
 * @param op [in] Pointer to removed child.
 */
void ComponentFace::remove_child(Component *op)
{
  for(int i = 0; i < 4; ++i)
  {
    if(edge[i] == op)
    {
      edge[i]->remove_parent(op);
      edge[i] = NULL;
      return;
    }

    if(vertex[i] == op)
    {
      vertex[i]->remove_parent(op);
      vertex[i] = NULL;
      return;
    }
  }
}

/** Update the position of this face by calculating the average from the
 * vertices.
 */
void ComponentFace::update()
{
  switch(type)
  {
    case COMPONENT_VOXEL:
      pos = vertex[0]->get_pos();
      return;
    case COMPONENT_LINE:
      pos.average(vertex[0]->get_pos(), vertex[1]->get_pos());
      return;
    case COMPONENT_TRIANGLE:
      pos.average(vertex[0]->get_pos(), vertex[1]->get_pos(),
	  vertex[2]->get_pos());
      return;
    case COMPONENT_QUAD:
      pos.average(vertex[0]->get_pos(), vertex[1]->get_pos(),
	  vertex[2]->get_pos(), vertex[3]->get_pos());
      return;
    default:
      return;
  }
}

/** Recalculate the attributes of this face.
 * Must be after all normals are calculated.
 */
void ComponentFace::calculate_attributes()
{
  int cnt = 0;

  // Flatness check.
  switch(this->get_type())
  {
    case COMPONENT_VOXEL:
      break;
    case COMPONENT_LINE:
      cnt = 2;
      break;
    case COMPONENT_TRIANGLE:
      cnt = 3;
      break;
    case COMPONENT_QUAD:
      cnt = 4;
      break;
    default:
      std::cerr << "Error: Face " << static_cast<unsigned>(this->get_idx())
	<< " has incomplete type.\n";
      break;
  }

  // Set flatness to true, remove it if need be.
  attr_or(ATTR_FLAT);
  for(int i = 1; (i < cnt); ++i)
  {
    if((vertex[0]->get_col() != vertex[i]->get_col()) ||
	(!vertex[0]->get_nor().is_same_direction(vertex[i]->get_nor())))
    {
      attr_not(ATTR_FLAT);
      break;
    }
  }
}

/** Calculate the normal of this face.
 */
void ComponentFace::calculate_normal()
{
  Vector3 tv1, tv2;

  // Act according to the type.
  switch(this->get_type())
  {
    // One corner.
    case COMPONENT_VOXEL:
      if(vertex[0]->get_pos().is_null())
      {
	normal.set(0.0f, 1.0f, 0.0f);
      }
      normal = vertex[0]->get_pos();
      normal.unit();
      break;

    // Two corners.
    case COMPONENT_LINE:
      tv1 = vertex[0]->get_pos();
      tv2 = vertex[1]->get_pos();
      if(tv1.is_null())
      {
	tv1 = Vector3(0.0f, 1.0f, 0.0f);
      }
      if(tv2.is_null())
      {
	tv2 = Vector3(0.0f, 1.0f, 0.0f);
      }
      tv1.unit();
      tv2.unit();
      normal = tv1 + tv2;
      if(normal.is_null())
      {
	normal.set(0.0f, 1.0f, 0.0f);
      }
      normal.unit();
      break;

    // Three or four corners.
    // If quad is nonplanar, will cause problems somewhere else.
    case COMPONENT_TRIANGLE:
    case COMPONENT_QUAD:
      tv1 = vertex[1]->get_pos() - vertex[0]->get_pos();
      tv2 = vertex[2]->get_pos() - vertex[0]->get_pos();
      if(tv1.is_null() || tv2.is_null())
      {
	normal.set(0.0f, 1.0f, 0.0f);
      }
      normal.product_cross(tv1, tv2);
      if(normal.is_null())
      {
	normal.set(0.0f, 1.0f, 0.0f);
      }
      normal.unit();
      break;

    // Error
    default:
      std::cerr << "Error: Face " << static_cast<unsigned>(this->get_idx())
	<< " has incomplete type.\n";
      break;
  }
}


//############################################################################
// Debug #####################################################################
//############################################################################

#ifdef LIBFHI_DEBUG

/** Print this vertex into given stream.
 * @param s [in] Stream to print to.
 * @param spaces [in] Spaces before each line.
 * @return Modified stream.
 */
std::ostream& ComponentVertex::print(std::ostream &s, int spaces) const
{
  this->print_spaces(s, spaces);
  s << "Vertex " << static_cast<unsigned>(this->idx) << " (" <<
    static_cast<unsigned>(this->parents.size()) << " parents):\n";
  this->print_spaces(s, spaces);
  s << "Pos: " << this->pos << " Col: " << this->col;
  return s;
}

/** Print this edge into given stream.
 * @param s [in] Stream to print to.
 * @param spaces [in] Spaces before each line.
 * @return Modified stream.
 */
std::ostream& ComponentEdge::print(std::ostream &s, int spaces) const
{
  this->print_spaces(s, spaces);
  s << "Edge " << static_cast<unsigned>(this->idx) << " (" <<
    static_cast<unsigned>(this->parents.size()) << " parents):\n";
  this->vertex[0]->print(s, spaces + 2);
  s << "\n";
  this->vertex[1]->print(s, spaces + 2);
  return s;
}

/** Print this face into given stream.
 * @param s [in] Stream to print to.
 * @param spaces [in] Spaces before each line.
 * @return Modified stream.
 */
std::ostream& ComponentFace::print(std::ostream &s, int spaces) const
{
  int cornernum = 0;
  this->print_spaces(s, spaces);

  // Print the type
  switch(type)
  {
    case COMPONENT_VOXEL:
      s << "Voxel ";
      cornernum = 1;
      break;
    case COMPONENT_LINE:
      s << "Line ";
      cornernum = 2;
      break;
    case COMPONENT_TRIANGLE:
      s << "Triangle ";
      cornernum = 3;
      break;
    case COMPONENT_QUAD:
      printf("Polygon ");
      cornernum = 4;
      break;
    default:
      break;
  }

  // Print the vertices and return
  s << static_cast<unsigned>(this->idx) << ":";
  for(int i = 0; i < cornernum; i++)
  {
    s << "\n";
    vertex[i]->print(s, spaces + 2);
  }
  return s;
}

#endif

//############################################################################
// Loppu #####################################################################
//############################################################################

