import copy
import os
import re

# Get the uname for tests
uname = (os.uname())[0]

##############################################################################
# Environment stuff ##########################################################
##############################################################################

# Get environment value from given environment
# @param env Environment to look in.
# @param op Variable name.
# @return Variables as array or empty stirng if not found.
def env_fetch(env, op):
  """Create environment variable if it does not exist."""
  if env['ENV'].has_key(op):
    return env['ENV'][op].split()
  return ''

# Replace environment variable with a given name by another with another
# given name.
# @param env Environment to do the exchange in.
# @param dst Destination variable name.
# @param src Source variable name.
def env_replace(env, dst, src):
  """Replace argument in environment by another."""
  if env['ENV'].has_key(src):
    env['ENV'][dst] = env['ENV'][src]

# Find a given flag in environment
# @param env Environment to search in.
# @param name Variable name.
# @param flag1 First flag to search.
# @param flag2 Second flag to search.
def env_find(env, name, flag1, flag2 = None):
  if env['ENV'].has_key(name):
    array = env['ENV'][name].split()
    for i in range(len(array)):
      if flag1 == array[i]:
	if flag2:
	  if flag2 == array[i + 1]:
	    return True
	else:
	  return True
  return False

# Append a C compiler flag to environment.
# @param env Environment to expand.
# @param flag Flag to add.
def env_append_cflag(env, flag):
  if not env_find(env, 'CCFLAGS', flag):
    env.Append(CCFLAGS = flag)
    if env['ENV'].has_key('CCFLAGS'):
      env['ENV']['CCFLAGS'] += ' ' + flag
    else:
      env['ENV']['CCFLAGS'] = flag
  if not env_find(env, 'CXXFLAGS', flag):
    env.Append(CXXFLAGS = flag)
    if env['ENV'].has_key('CXXFLAGS'):
      env['ENV']['CXXFLAGS'] += ' ' + flag
    else:
      env['ENV']['CXXFLAGS'] = flag

# Append a linker flag to environment.
# @param env Environment to expand.
# @param flag1 First flag.
# @param flag2 Second flag.
def env_append_ldflag(env, flag1, flag2 = None):
  if flag2:
    if not env_find(env, 'LINKFLAGS', flag1, flag2):
      env.Append(LINKFLAGS = flag1)
      env.Append(LINKFLAGS = flag2)
      if env['ENV'].has_key('LINKFLAGS'):
	env['ENV']['LINKFLAGS'] += ' ' + flag1 + ' ' + flag2
      else:
	env['ENV']['LINKFLAGS'] = flag1 + ' ' + flag2
  elif not env_find(env, 'LINKFLAGS', flag1):
    env.Append(LINKFLAGS = flag1)
    if env['ENV'].has_key('LINKFLAGS'):
      env['ENV']['LINKFLAGS'] += ' ' + flag1
    else:
      env['ENV']['LINKFLAGS'] = flag1

# Fix an erraneous input string to ParseConfig.
# @param command Command that generates a string.
# @return Fixed string.
def parseconfig_fix(env, command):
  # Open the command from operating system.
  str = os.popen(command).read()
  # Read things SCons ParseConfig cannot handle.
  str_array = str.split()
  for i in str_array:
    if i[:2] == '-D' and i[2:]:
      env_append_cflag(env, i)
  # Replace inconsistensies that could cause breakage.
  str = str.replace('-Wl,--rpath', '-Wl,-rpath')
  str = str.replace('"', '\"')
  return 'echo "' + str + '"'

##############################################################################
# Pkgconfig ##################################################################
##############################################################################

# Class pkgconfig is used to collect data for pkg-config file writing.
class PkgConfig:

  # Local variables.
  env = 0
  cflags = []
  lflags = []
  libs = []
  requirements = []
  description = "No description"
  dir_bin = None
  dir_inc = None
  dir_lib = None
  dir_prefix = "/usr/local"
  name = "Unknown"
  version = "0.0.0"

  ########################################
  # Generic methods ######################
  ########################################

  # Constructor.
  # @param environment The environment to use within this.
  def __init__(self, environment):
    self.env = environment.Copy()

  # Copy this pkgconfig.
  # @return copy of this.
  def Copy(self):
    ret = PkgConfig(self.env)
    ret.cflags = copy.deepcopy(self.cflags)
    ret.lflags = copy.deepcopy(self.lflags)
    ret.libs = copy.deepcopy(self.libs)
    ret.requirements = copy.deepcopy(self.requirements)
    ret.description = self.description
    ret.dir_bin = self.dir_bin
    ret.dir_inc = self.dir_include
    ret.dir_lib = self.dir_lib
    ret.dir_prefix = self.dir_prefix
    ret.name = self.name
    ret.version = self.version
    return ret

  # Add a C compiler flag to the pkgconfig data, also export to environment.
  # @param flag Flag to add.
  def add_cflag(self, flag):
    if flag != '':
      self.env.Append(CCFLAGS = [flag])
      self.env.Append(CXXFLAGS = [flag])
      self.cflags += [flag]

  # Add a linker flag to the pkgconfig data, also export to envirionment.
  # @param flag Flag to add.
  def add_lflag(self, flag):
    if flag != '':
      self.env.Append(LINKFLAGS = [flag])
      self.lflags += [flag]

  # Add a library to the pkgconfig data, also export to environment.
  # @param lib Library to add.
  def add_lib(self, lib):
    if lib != '':
      self.env.Append(LIBS = [lib])
      self.libs += [lib]

  # Add a requirement to the pkgconfig data.
  # @param op The requirement (a package within pkg-config).
  def add_requirement(self, op):
    if op != '':
      self.requirements += [op]

  # Get executable directory.
  # @return Directory string.
  def get_dir_bin(self):
    if self.dir_bin:
      return self.dir_bin
    return os.path.join(self.dir_prefix, 'bin')

  # Get include directory.
  # @param forpc True if required for .pc -file.
  # @return Directory string.
  def get_dir_inc(self, forpc = False):
    if self.dir_inc:
      return self.dir_inc
    if forpc:
      return '${prefix}/include'
    return os.path.join(self.dir_prefix, 'include')

  # Get library directory.
  # @param forpc True if required for .pc -file.
  # @return Directory string.
  def get_dir_lib(self, forpc = False):
    if self.dir_lib:
      return self.dir_lib
    if forpc:
      return '${exec_prefix}/lib'
    return os.path.join(self.dir_prefix, 'lib')

  # Get pkgconfig install directory.
  # @return Directory string.
  def get_dir_pkgconfig(self):
    return os.path.join(self.get_dir_lib(), 'pkgconfig')

  # Get the current environment out.
  # @return A copy of the current environment.
  def get_env(self):
    return self.env.Copy()

  # Set the name, version and description
  # @param n Name.
  # @param v Version.
  # @param d Description.
  def set_info(self, n, v, d):
    self.name = n
    self.version = v
    self.description = d

  # Set bindir (to other than prefix/bin).
  # @param str Library directory.
  def set_dir_bin(self, srt):
    self.dir_bin = str

  # Set includedir (to other than prefix/include).
  # @param str Library directory.
  def set_dir_include(self, srt):
    self.dir_include = str

  # Set libdir (to other than prefix/lib).
  # @param str Library directory.
  def set_dir_lib(self, srt):
    self.dir_lib = str

  # Set the installation prefix
  # @param str Prefix directory.
  def set_dir_prefix(self, str):
    self.dir_prefix = str

  # Write a pkgconfig .pc file
  # @param filename File to write to.
  def write(self, filename):
    file = open(filename, "wt")
    # Write prefixes.
    file.write("prefix=" + self.dir_prefix +"\n")
    file.write("exec_prefix=${prefix}\n")
    file.write("libdir=" + self.get_dir_lib(True) + "\n")
    file.write("includedir=" + self.get_dir_inc(True) + "\n")
    # Write info.
    file.write("\nName: " + self.name + "\nDescription: " + self.description +
	"\nVersion: " + self.version + "\nRequires:")
    # Write requirements.
    for i in self.requirements:
      file.write(" " + i)
    # No cflags if nothing specified.
    file.write("\nCflags: -I${includedir}")
    for i in self.cflags:
      file.write(" " + i)
    # No lflags if nothing specified.
    file.write("\nLibs: -L${libdir}")
    for i in self.lflags:
      file.write(" " + i)
    for i in self.libs:
      file.write(" " + "-l" + i)
    # Close and end.
    file.close()

  ########################################
  # Find framework #######################
  ########################################

  # Try to include a framework that works with a shell script, i.e.
  # pkg-config.
  # @param vcommand Version check command.
  # @param acommand Command to add flags.
  # @param name Name of the package (for checking).
  # @param cmp Comarison version (must be greater or equal to).
  # @param define Export this flag if successful.
  # @param obligatory If true, exit on error.
  # @return True if found, False if not.
  def find_compare(self, vcommand, acommand, name, cmp, define = '',
      obligatory = False):
    # Print searching text.
    print 'Checking for ' + name + ' version >= ' + cmp + ': ',
    # Read the version and prepare it (as the compared version)
    ver = os.popen(vcommand).read().strip().split('.')
    cmpver = cmp.split('.')
    # Initialize to false
    ret = False
    retstring = 'no'
    # Loop, if succeeds, ret will be true
    for i in range(len(cmpver)):
      vn = int(ver[i])
      cn = int(cmpver[i])
      if vn < cn:
	if obligatory:
	  print 'Error: no'
	  Exit(1)
	print 'no'
	return False
      elif vn > cn:
	break
    print 'yes'
    # Parse the config.
    self.env.ParseConfig(parseconfig_fix(self.env, acommand))
    # Manually add -D flag if defined.
    if define != '':
      self.add_cflag('-D' + define)
    return True

  # Search for a library and a header, add to this project if found.
  # @param lib Name of the library.
  # @param header Name of the header.
  # @param define Export this flag if successful.
  # @param lang Language (usually C).
  # @param obligatory If true, exit on error.
  # @param force If true, continue even if not found.
  # @return True on success, false on error.
  def find_libwithheader(self, lib, header, define = '', lang = 'C',
      obligatory = False, force = True, symbol = ''):
    # On Darwin, need to apply these additional directories.
    if uname == 'Darwin':
      env_append_cflag(self.env, '-I/usr/local/include')
      env_append_cflag(self.env, '-I/sw/include')
      env_append_ldflag(self.env, '-L/usr/local/lib')
      env_append_ldflag(self.env, '-L/sw/lib')
    # Begin configuration.
    conf = self.env.Configure()
    if symbol:
      success = conf.CheckLibWithHeader(lib, header, lang, symbol)
    else:
      success = conf.CheckLibWithHeader(lib, header, lang)
    if not success:
      if force:
	print "Warning: library " + lib + " not found, continuing anyway."
      elif obligatory:
	print "Error: library " + lib + " not found."
	Exit(1)
      else:
	print "Warning: library " + lib + " not found."
	return False
    self.env = conf.Finish()
    if define != '':
      self.add_cflag('-D' + define)
    if not success:
      self.env.Append(LIBS = lib)
    return True

  # Search for given header (default: C, not C++), append given define if
  # found, possibly abort if not found.
  # @param header Header name.
  # @param define Export this flag if successful.
  # @param lang Language (usually 'C').
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_header(self, header, define = '', lang = 'C', obligatory = False):
    conf = self.env.Configure()
    if lang == 'C':
      success = conf.CheckCHeader(header)
    else:
      success = conf.CheckCXXHeader(header)
    if not success and obligatory:
      print "Error: header " + header + " not found."
      Exit(1)
    self.env = conf.Finish()
    if success:
      if define:
	self.add_cflag('-D' + define)
      return True
    return False

  # Search for a given function, append given define if found, possibly abort
  # if not found.
  # @param func Function name.
  # @param header Header where the prototype is declared.
  # @param define Export this flag if successful.
  # @param lang Language (usually 'C').
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_function(self, func, header, define = '', lang = 'C',
      obligatory = False):
    conf = self.env.Configure()
    success = conf.CheckFunc(func, header, lang)
    self.env = conf.Finish()
    if success:
      if define:
	self.add_cflag('-D' + define)
      return True
    return False

  ########################################
  # Individual find methods ##############
  ########################################

  # Find GDlib.
  # @param cmp Minimum version.
  # @param define If set, define this on success.
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_gdlib(self, cmp,  define = '', obligatory = True):
    return self.find_compare('gdlib-config --version',
	'gdlib-config --cflags --libs ; echo "-lgd"', 'GDlib', cmp, define,
	obligatory)

  # Find GLEW.
  # @param define If set, define this on success.
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_glew(self, define = '', obligatory = True):
    return self.find_libwithheader('GLEW', 'GL/glew.h', define, 'C',
	obligatory)

  # Find GLUT.
  # @param define If set, define this on success.
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_glut(self, define = '', obligatory = True):
    return self.find_libwithheader('glut', 'GL/glut.h', define, 'C',
	obligatory)

  # Find GTK.
  # @param cmp Minimum version.
  # @param define If set, define this on success.
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_gtk(self, cmp,  define = '', obligatory = True):
    return self.find_compare('pkg-config --modversion gtk+-2.0',
	'pkg-config --cflags --libs gtk+-2.0', 'GTK', cmp, define,
	obligatory)

  # Find GTK.
  # @param cmp Minimum version.
  # @param define If set, define this on success.
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_gthread(self, cmp,  define = '', obligatory = True):
    return self.find_compare('pkg-config --modversion gthread-2.0',
	'pkg-config --cflags --libs gthread-2.0', 'GThread', cmp, define,
	obligatory)

  # Find Freetype.
  # @param cmp Minimum version.
  # @param define If set, define this on success.
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_freetype(self, cmp,  define = '', obligatory = True):
    return self.find_compare('freetype-config --version',
	'freetype-config --cflags --libs', 'FreeType', cmp, define,
	obligatory)

  # Find libfhi.
  # @param cmp Minimum version.
  # @param define If set, define this on success.
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_libfhi(self, cmp,  define = '', obligatory = True):
    return self.find_compare('pkg-config --modversion libfhi',
	'pkg-config --cflags --libs libfhi', 'libfhi', cmp, define,
	obligatory)

  # Find libpng.
  # @param cmp Minimum version.
  # @param define If set, define this on success.
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_libpng(self, cmp,  define = '', obligatory = True):
    return self.find_compare('pkg-config --modversion libpng12',
	'pkg-config --cflags --libs libpng12', 'libpng', cmp, define,
	obligatory)

  # Find libxml.
  # @param cmp Minimum version.
  # @param define If set, define this on success.
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_libxml(self, cmp,  define = '', obligatory = True):
    return self.find_compare('pkg-config --modversion libxml-2.0',
	'pkg-config --cflags --libs libxml-2.0', 'libxml', cmp, define,
	obligatory)

  # Find OpenGL, a little bit trickier this time.
  # @param define If set, define this on success.
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_opengl(self, define = '', obligatory = True):
    # Check for OSX, if yes, do a little add beforehand
    if uname == 'Darwin':
      if define != '':
	self.add_cflag('-D' + define)
      env_append_cflag(self.env, '-I/System/Library/Frameworks/OpenGL.framework/Headers')
      env_append_ldflag(self.env, '-L/System/Library/Frameworks/OpenGL.framework/Libraries')
      env_append_ldflag(self.env, '-framework', 'OpenGL')
      print 'Checking for OpenGL: -framework OpenGL'
      return True
    elif uname == 'FreeBSD':
      env_append_cflag(self.env, '-I/usr/X11R6/include')
      env_append_ldflag(self.env, '-L/usr/X11R6/lib')
    # Proceed.
    success = self.find_libwithheader('GL', 'GL/gl.h', define, 'C',
	obligatory)
    if success:
      success = self.find_libwithheader('GLU', 'GL/glu.h', '', 'C',
	  obligatory)
    return success

  # Find SDL.
  # @param cmp Minimum version.
  # @param define If set, define this on success.
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_sdl(self, cmp,  define = '', obligatory = True):
    # Check for OSX, if yes, use framework.
    if uname == 'Darwin':
      if define != '':
	self.add_cflag('-D' + define)
      env_append_cflag(self.env, '-I/Library/Frameworks/SDL.framework/Headers')
      env_append_cflag(self.env, '-D_REENTRANT')
      env_append_ldflag(self.env, '-framework', 'SDL')
      env_append_ldflag(self.env, '-framework', 'Cocoa')
      print 'Checking for SDL: -framework SDL'
      return True
    # Otherwise default.
    return self.find_compare('sdl-config --version',
	'sdl-config --cflags --libs', 'SDL', cmp, define,
	obligatory)

  # Find SDL_image.
  # @param define If set, define this on success.
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_sdl_image(self, define = '', obligatory = True):
    # Check for OSX, if yes, use framework.
    if uname == 'Darwin':
      if define != '':
	self.add_cflag('-D' + define)
      env_append_cflag(self.env, '-I/Library/Frameworks/SDL_image.framework/Headers')
      env_append_ldflag(self.env, '-framework', 'SDL_image')
      print 'Checking for SDL_image: -framework SDL_image'
      return True
    return self.find_libwithheader('SDL_image', 'SDL_image.h', define, 'C',
	obligatory)

  # Find SDL_mixer.
  # @param define If set, define this on success.
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_sdl_mixer(self, define = '', obligatory = True):
    # Check for OSX, if yes, use framework.
    if uname == 'Darwin':
      if define != '':
	self.add_cflag('-D' + define)
      env_append_cflag(self.env, '-I/Library/Frameworks/SDL_mixer.framework/Headers')
      env_append_ldflag(self.env, '-framework', 'SDL_mixer')
      print 'Checking for SDL_image: -framework SDL_mixer'
      return True
    return self.find_libwithheader('SDL_mixer', 'SDL_mixer.h', define, 'C',
	obligatory)

  # Find wxWidgets.
  # @param cmp Minimum version.
  # @param define If set, define this on success.
  # @param obligatory If true, exit on error.
  # @return True on success, false on error.
  def find_wxwidgets(self, cmp,  define = '', obligatory = True):
    return self.find_compare('wx-config --version',
	'wx-config --cflags --libs', 'wxWidgets', cmp, define,
	obligatory)

##############################################################################
# End ########################################################################
##############################################################################

