#ifndef particle_h
#define particle_h

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi.h"

//############################################################################
// Class #####################################################################
//############################################################################

/** Particle system consists of an immense number of individual quads, that
 * have small lifetimes and few varying attributes. This is 
 */
class Particle
{
  private:
    /** Color of this particle. */
    libfhi::Color4 color;

    /** Position of this particle. */
    libfhi::Vector2 pos;

    /** Movement vector of this particle. */
    libfhi::Vector2 mov;

    /** Decrement of the alpha component of this particle. */
    float decrement;

    /** Lifetime, after this reaches 0, the particle is destroyed. */
    int lifetime;

  public:
    inline Particle(const libfhi::Color4&, const libfhi::Vector2&,
	const libfhi::Vector2&, int);
    inline ~Particle();

  public:
    inline const libfhi::Color4& get_col() const;
    inline int get_lifetime() const;
    inline const libfhi::Vector2& get_pos() const;
    inline bool tick();
};

//############################################################################
// Particle inline ###########################################################
//############################################################################

/** Constructor for particle.
 * @param c Color to use.
 * @param p Position to start from.
 * @param d Direction (and movement).
 * @param l Lifetime.
 */
Particle::Particle(const libfhi::Color4 &c, const libfhi::Vector2 &p,
    const libfhi::Vector2 &d, int l)
  : color(c), pos(p), mov(d), lifetime(l)
{
  // Decrementing so, that size reaches 0 when lifetime would reach 0.
  this->decrement = this->color.af / static_cast<float>(l);
}

/** Default destructor.
 */
Particle::~Particle()
{
  // Do nothing.
}

/** Get color of this.
 * @return Color as reference.
 */
const libfhi::Color4& Particle::get_col() const
{
  return this->color;
}

/** Get current lifetime of this.
 * @return Lifetime as int.
 */
int Particle::get_lifetime() const
{
  return this->lifetime;
}

/** Get position of this.
 * @return Position as reference.
 */
const libfhi::Vector2& Particle::get_pos() const
{
  return this->pos;
}

/** Advance one frame in a particle.
 * @return True if particle still exists, false if not.
 */
bool Particle::tick()
{
  if((this->lifetime--) <= 0)
  {
    return false;
  }

  // Advance position.
  this->pos += this->mov;

  // Decrement scale.
  this->color.af -= this->decrement;

  return true;
}

//############################################################################
// End #######################################################################
//############################################################################

#endif

