#include "weaponslot.h"

#include "effect.h"
#include "mainmenu.h"
#include "sound.h"
#include "weapon.h"
#include "weaponmount.h"

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor.
 * @param op_pos Position relative to host.
 * @param op_dir Direction.
 * @param op_pri Priority to use.
 * @param op_mount Mount to link to.
 */
WeaponSlot::WeaponSlot(const libfhi::Vector2 &op_pos, uint16_t op_dir,
    WeaponMount *op_mount)
  : pos(op_pos), dir(op_dir), reload(0), ammo(op_mount->get_max_ammo()),
  queue_cnt(0), mount(op_mount)
{
  // Do nothing.
}

/** Copy constructor.
 * @param src Copy source.
 */
WeaponSlot::WeaponSlot(WeaponSlot *src)
  : pos(src->pos), dir(src->dir), reload(src->reload), ammo(src->ammo), 
  queue_cnt(0), mount(src->mount)
{
  // Check that something really stupid did not happen.
  if(this->mount == NULL)
  {
    std::cerr << "Error: Copied weapon slot with illegal weapon mount.\n";
  }

  // If weapon has fade, instead set ammo to 0.
  if(this->mount->has_fade())
  {
    this->ammo = 0;
  }
}

/** Default destructor.
 */
WeaponSlot::~WeaponSlot()
{
  // Do nothing.
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Fire one bullet of this.
 * @param faction Faction to fire under.
 * @param hpos Host position.
 * @param hmov Host movement.
 * @param hdir Host direction.
 * @param charge How long has a released button been pressed?
 */
void WeaponSlot::fire(int faction, const libfhi::Vector2 &hpos,
    const libfhi::Vector2 &hmov, uint16_t hdir, int charge_now,
    int charge_last)
{
  // Power and damage of fired shot.
  int power = 0,
      damage;

  // Get the weapon to use.
  Weapon *weapon = this->mount->get_weapon();

  // If weapon has fade and we have stuff coming.
  if(this->mount->has_fade() && (this->queue_cnt > 0))
  {
    if(this->reload > 0)
    {
      return;
    }

    // Remove one from queue.
    this->reload = this->mount->get_reload();
    --this->queue_cnt;
    damage = weapon->get_damage();
  }
  // Else if of charging type.
  else if(this->mount->has_charge())
  {
    // Do not fire unless we have released the button.
    if(charge_now != 0)
    {
      if(charge_now > EFFECT_CHARGE_TOLERANCE)
      {
	// But while the charge really is above zero, exhibit charge
	// particles.
	Effect *efu = weapon->get_charge();

	// Execute the effect if found.
	if(efu)
	{
	  efu->execute(&hpos, &hmov);
	}
      }

      return;
    }

    // The power of the charge depends on the amount of ticks it was held
    // down.
    power = this->mount->get_power(charge_last);

    // Abort if no power.
    if(power == 0)
    {
      return;
    }

    // If weapon has fade, set the fade to given value and reload to reload
    // time.
    if(this->mount->has_fade())
    {
      this->queue_cnt = this->mount->get_fade();
      this->reload = this->mount->get_reload();
    }

    damage = this->mount->get_damage(power);
  }
  // Otherwise of reload type. Only fire on single-shot presses or when there is
  // queue of bullets waiting.
  else
  {
    // Fade handles queue in a different manner.
    if(this->mount->has_fade())
    {
      if(charge_now != 1)
      {
	return;
      }

      if(this->ammo < this->mount->get_drain())
      {
	return;
      }

      // One less than fade.
      this->reload = this->mount->get_reload();
      this->queue_cnt = this->mount->get_fade();
      this->ammo -= this->mount->get_drain();
    }
    // Than normals.
    else
    {
      // Increase queue count if it is less or equal to the single queue.
      if((charge_now == 1) && (this->queue_cnt <= this->mount->get_burst()))
      {
	this->queue_cnt += this->mount->get_burst();
      }

      if(queue_cnt <= 0)
      {
	return;
      }

      // If no ammo, empty the queue and do not register.
      if(this->ammo < this->mount->get_drain())
      {
	this->queue_cnt = 0;
	return;
      }

      // Abort if no reload time.
      if(this->reload > 0)
      {
	return;
      }

      // Otherwise take one bullet from queue and shoot it.
      this->reload = this->mount->get_reload();
      --this->queue_cnt;
      this->ammo -= this->mount->get_drain();
    }

    // Finally, get damage.
    damage = weapon->get_damage();
  }

  // Rotate this position according to host direction.
  float pi = libfhi::uint2pi(hdir),
	cr = cosf(pi),
	sr = sinf(pi);

  libfhi::Vector2 npos;
  npos.rotate_translate(this->pos, cr, sr, hpos);

  // Get random deviation, then calculate new sine/cosine.
  int dirdiff = (rand() % (this->mount->get_spread() * 2 + 1)) -
    this->mount->get_spread();
  pi = libfhi::uint2pi(static_cast<uint16_t>(this->dir + hdir + dirdiff));
  cr = cosf(pi);
  sr = sinf(pi);

  // Add one bullet.
  weapon->add(faction,
      npos,
      hmov,
      cr,
      sr,
      power,
      damage);
}

/** Draw the charge of this.
 * @param y Y coordinate to start drawing from.
 * @param charge Charge level of the entity.
 * @param screen Screen to draw into.
 */
int WeaponSlot::draw(int y, int charge, libfhi::Surface *screen)
{
  static int reload_color = libfhi::Surface::makecol3(255, 200, 50),
	     charge_color = libfhi::Surface::makecol3(50, 200, 255);

  float percent;
  int col;

  if(this->mount->has_charge())
  {
    percent = static_cast<float>(stdmin(charge, this->mount->get_charge())) /
      static_cast<float>(this->mount->get_charge());
    col = charge_color;
  }
  else
  {
    percent = static_cast<float>(this->ammo) /
      static_cast<float>(this->mount->get_max_ammo());
    col = reload_color;
  }
  return Menu::draw_charge_bar(y, percent, col, this->mount->get_name(),
      screen);
}

//############################################################################
// End #######################################################################
//############################################################################

