#ifndef weaponmount_h
#define weaponmount_h

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi.h"

#include "enums.h"
#include "game.h"
#include "weapon.h"

//############################################################################
// Class #####################################################################
//############################################################################

/** WeaponMount describes the physical attributes of one weapon. It's damage,
 * muzzle speed, spread, lifetime, etc.
 */
class WeaponMount :
  public libfhi::Attributable,
  public libfhi::CanonicalDB<WeaponMount>
{
  public:
    /** Charging weapon type. */
    static const uint8_t ATTR_CHARGE = 0x01;

    /** Fading type. */
    static const uint8_t ATTR_FADE = 0x02;

    /** Default attributes. */
    static const uint8_t DEFAULT_ATTR = 0x00;

  private:
    /** The reference to weapon this is using. */
    Weapon *weapon;

    union
    {
      /** Burst, the number of shots to fire per each button mash. */
      int burst;

      /** Fade - how long the weapon is active after charge released. */
      int fade;
    };

    union
    {
      /** The count decremented by one shot. */
      int drain;

      /** Charge time of one level shot. */
      int charge;
    };

    /** Ammunition, firing one shot decrements this by drain variable. */
    int max_ammo;

    /** Reload time of bullets. */
    int reload;

    /** Spread of bullets spawned by this weapon. */
    int spread;

    /** String containing name. */
    std::string name;

    /** Name width on screen (precalculated). */
    int name_width;

  public:
    WeaponMount(const char*);
    ~WeaponMount();

    // Inline methods.
    inline int get_burst() const;
    inline int get_charge() const;
    inline int get_damage(int) const;
    inline int get_drain() const;
    inline int get_fade() const;
    inline int get_max_ammo() const;
    inline const char* get_name() const;
    inline int get_power(int) const;
    inline int get_reload() const;
    inline float get_speed() const;
    inline int get_spread() const;
    inline Weapon* get_weapon();
    inline bool has_charge() const;
    inline bool has_fade() const;
};

//############################################################################
// Inline methods ############################################################
//############################################################################

/** Get burst.
 * @return Burst as integer.
 */
int WeaponMount::get_burst() const
{
  return this->burst;
}

/** Get charge.
 * @return Chaerge as integer.
 */
int WeaponMount::get_charge() const
{
  return this->charge;
}

/** Get damage of this, assuming a charge time.
 * @param op Number of frames this has been charging.
 * @return Damage as integer.
 */
int WeaponMount::get_damage(int op) const
{
  return op * this->weapon->get_damage();
}

/** Get drain of one shot.
 * @return Drain as integer.
 */
int WeaponMount::get_drain() const
{
  return this->drain;
}

/** Get fade of one shot.
 * @return Fade as integer.
 */
int WeaponMount::get_fade() const
{
  return this->fade;
}

/** Get maximum ammo count.
 * @return Speed as float.
 */
int WeaponMount::get_max_ammo() const
{
  return this->max_ammo;
}

/** Get the name of this.
 * @return Name string.
 */
const char* WeaponMount::get_name() const
{
  return this->weapon->get_name();
}

/** Get power of shot that has been charged given amount of frames.
 * @param op Number of frames this has been charging.
 * @return Damage as integer.
 */
int WeaponMount::get_power(int op) const
{
  return (op / this->charge);
}

/** Get the reload time of this.
 * @return Reload time.
 */
int WeaponMount::get_reload() const
{
  return this->reload;
}

/** Get the speed of this.
 * @return Speed as float.
 */
float WeaponMount::get_speed() const
{
  return this->weapon->get_speed();
}

/** Get spread of this.
 * @return Spread as integer.
 */
int WeaponMount::get_spread() const
{
  return this->spread;
}

/** Get the weapon id of this (what kind of bullet to spawn).
 * @return Weapon id as enum.
 */
Weapon* WeaponMount::get_weapon()
{
  return this->weapon;
}

/** Tell if this has charge properties.
 * @return True if yes, false if no.
 */
bool WeaponMount::has_charge() const
{
  return this->has_attr(ATTR_CHARGE);
}

/** Tell if this has fade properties.
 * @return True if yes, false if no.
 */
bool WeaponMount::has_fade() const
{
  return this->has_attr(ATTR_FADE);
}

//############################################################################
// End #######################################################################
//############################################################################

#endif

