#include "weapon.h"

#include "data.h"
#include "effect.h"
#include "particlesystem.h"
#include "sound.h"

#include <sstream>

//############################################################################
// Statics ###################################################################
//############################################################################

float Weapon::slowdown = 1.0f;

//############################################################################
// Weapon construction #######################################################
//############################################################################

/** Default constructor.
 */
Weapon::Weapon()
  : Attributable(DEFAULT_ATTR)
{
  // Do nothing.
}

/** Load constructor. Open a file and load information from there.
 * @param type The id of this weapon.
 * @param filename File to open.
 */
Weapon::Weapon(const char *filename)
  : Attributable(DEFAULT_ATTR), mesh(NULL), model(NULL), sample_fire(NULL),
  sample_hit(NULL), sample_fade(NULL), hit(NULL), fade(NULL), charge(NULL),
  explosion_radius(0.0f), use_ranged_explosions(false)
{
  libfhi::ConfigFile filu(filename);

  if(!filu.is_ok())
  {
    std::cout << "ERROR: could not weapon from \"" << filename << "\"\n";
    return;
  }

  while(filu.advance())
  {
    // Attributes.
    if(filu.has_id_arg("attr", 1))
    {
      this->set_attr(filu.get_int(0));
    }
    // Name.
    else if(filu.has_id("name"))
    {
      if(filu.get_num_arg() < 1)
      {
	filu.warn("needs string argument.");
	this->name = "ERROR";
      }
      else
      {
	std::stringstream sstr;
	sstr << filu.get_string(0);

	for(size_t i = 1; (i < filu.get_num_arg()); ++i)
	{
	  sstr << " " << filu.get_string(i);
	}

	this->name = sstr.str();
      }
    }
    // Samples.
    else if(filu.has_id_arg("sample", 3))
    {
      std::string
	samplename_fire = filu.get_string(0).c_str(),
	samplename_hit = filu.get_string(1).c_str(),
	samplename_fade = filu.get_string(2).c_str();

      if(samplename_fire.compare("NONE") != 0)
      {
	this->sample_fire = Data::load_sample(samplename_fire.c_str());
      }
      if(samplename_hit.compare("NONE") != 0)
      {
	this->sample_hit = Data::load_sample(samplename_hit.c_str());
      }
      if(samplename_fade.compare("NONE") != 0)
      {
	this->sample_fade = Data::load_sample(samplename_fade.c_str());
      }
    }
    // Hits.
    else if(filu.has_id_arg("hit", 1))
    {
      this->hit = Data::load_eff(filu.get_string(0).c_str());
    }
    // Fadeouts.
    else if(filu.has_id_arg("fade", 1))
    {
      this->fade = Data::load_eff(filu.get_string(0).c_str());
    }
    // Charges.
    else if(filu.has_id_arg("charge", 1))
    {
      this->charge = Data::load_eff(filu.get_string(0).c_str());
    }
    // Absorb cap.
    else if(filu.has_id_arg("absorb", 1))
    {
      this->absorb_cap = filu.get_int(0);
    }
    // Explosion radius.
    else if(filu.has_id_arg("explosion_radius", 1))
    {
      this->explosion_radius = filu.get_float(0);
      this->use_ranged_explosions = (this->explosion_radius > 0.0f);
    }
    // Speed (for bullets).
    else if(filu.has_id_arg("speed", 1))
    {
      this->speed = filu.get_float(0);

      // Also tells us this is a bullet weapon.
      this->attr_not(ATTR_INSTANT);
    }
    // Length (for instants).
    else if(filu.has_id_arg("length", 1))
    {
      this->length = filu.get_float(0);

      // Also tells us this is an instant weapon.
      this->attr_or(ATTR_INSTANT);
    }
    // Damage.
    else if(filu.has_id_arg("damage", 1))
    {
      this->damage = filu.get_int(0);
    }
    // Lifetime.
    else if(filu.has_id_arg("lifetime", 1))
    {
      this->lifetime = filu.get_int(0);
    }
    // Rotation speed.
    else if(filu.has_id_arg("rotspeed", 1))
    {
      this->rotspeed = filu.get_int(0);
    }
    // Thurst.
    else if(filu.has_id_arg("thurst", 1))
    {
      this->thurst = filu.get_float(0);

      // Also tells us this is a homing weapon.
      this->attr_or(ATTR_HOMING);
    }
    // Drag coefficient.
    else if(filu.has_id_arg("drag", 2))
    {
      this->drag_v = filu.get_float(0);
      this->drag_h = filu.get_float(1);
    }
    // Mass.
    else if(filu.has_id_arg("mass", 1))
    {
      this->mass = filu.get_float(0);
    }
    // Collision map.
    else if(filu.has_id_arg("cmap", 1))
    {
      CollisionMap *ref = Data::load_cmap(filu.get_string(0).c_str());

      this->cmap = new CollisionMap(ref, NULL);
    }
    // Model.
    else if(filu.has_id_arg("msh", 2))
    {
      this->visible_count = filu.get_int(0);

      // Load reference mesh.
      this->mesh = Data::load_msh(filu.get_string(1).c_str());
      this->maxdist = this->mesh->get_maxdist();

      // Create the display model, along with the mesh. Then reserve it as
      // multiple of the reference mesh.
      this->model = new libfhi::PostModel(new libfhi::Mesh());
      this->model->get_mesh()->reserve_multiple(this->mesh,
	  this->visible_count);
    }
    else
    {
      filu.warn_empty();
    }
  }

  // If homing weapon, calculate some physical attributes.
  if(this->has_homing())
  {
    // Copy pasta from entity. TODO: FIX!
    this->mass = (this->mass * 0.5f) / powf(PHYSICS_STEP_SPACE, 4.0f);
    this->thurst = (this->thurst * 0.5f) / powf(PHYSICS_STEP_SPACE, 4.0f);
    this->drag_v /= powf(PHYSICS_STEP_SPACE, 2.0f);
    this->drag_h /= powf(PHYSICS_STEP_SPACE, 2.0f);
  }

  std::cout << "Loaded weapon \"" << filename << "\"\n";
}

/** Default destructor.
 */
Weapon::~Weapon()
{
  this->clear_munitions();

  // Destroy model and it's mesh.
  if(this->model)
  {
    delete this->model->get_mesh();
    delete this->model;
  }
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Addition for munitions.
 * @param faction Bullet faction.
 * @param pos Host position.
 * @param mov Host movement.
 * @param cr Cosine of direction rotation.
 * @param sr Sine of direction rotation.
 * @param power The power of shot fired.
 * @param dmg Damage done by bullet.
 */
void Weapon::add(int faction, const libfhi::Vector2 &pos,
    const libfhi::Vector2 &mov, float cr, float sr, int power, int dmg)
{
  // New munition and push it into the back.
  Munition *mun = new Munition();
  this->bullets.push_back(mun);

  // Play the sound.
  if(this->sample_fire)
  {
    this->sample_fire->Play(pos);
  }

  // Call correct init method depending on type.
  if(this->has_instant())
  {
    mun->init_instant(faction, pos, cr, sr, dmg, this->lifetime);
  }
  else
  {
    if(this->has_homing())
    {
      mun->init_homing(faction, pos, mov, cr, sr, dmg, this->lifetime,
	  this->get_speed());
    }
    else
    {
      mun->init_bullet(faction, pos, mov, cr, sr, power, dmg, this->lifetime,
	  this->get_speed());
    }
  }
}

/** Clear all munitions from this weapon (prepare for new game).
 */
void Weapon::clear_munitions()
{
  // Destroy all bullets.
  for(std::list<Munition*>::iterator i = this->bullets.begin(),
      e = this->bullets.end(); (i != e); ++i)
  {
    delete *i;
  }

  this->bullets.clear();
}

/** Create an explosion (that deals damage) to a position.
 * @param pos Position in absolute coordinates.
 * @param dir Direction of explosion (used if needed).
 * @param dmg Damage value.
 */
void Weapon::explode(const libfhi::Vector2 &pos, const libfhi::Vector2 &dir,
    int dmg)
{
  // This be static. If it fails, tough luck.
  static CollisionMap *explmap =
    Data::load_cmap("collisionmap/munition_explosion.cmap");

  // Play hit sound if exists.
  if(this->sample_hit)
  {
    this->sample_hit->Play(pos);
  }

  // Execute the effect.
  this->hit->execute(&pos, &dir);

  // Do the complex ranged explosion stuff. Prettu crude but works.
  if(this->use_ranged_explosions)
  {
    // Get the particle system and terrain.
    ParticleSystem *hit_system = this->hit->get_system();
    Terrain *terrain = Terrain::instance;

    // The list of objects we want to harm afterwards.
    std::list<EntityBase*> harm_list;

    // Get count of sparks.
    int cnt = this->hit->get_cnt();

    // Explosion map must have dot.
    CollisionMap::CollisionDot *dot = explmap->get_single_dot();

    float rand_radius = (this->explosion_radius - dot->r) * 0.001f;

    // Loop (note, 0 sparks does not crash).
    while(cnt--)
    {
      float pi = libfhi::uint2pi(static_cast<uint16_t>(rand() % 65536)),
	    spotr = (rand() % 1001) * rand_radius;

      // Generate hit position.
      libfhi::Vector2 hitpos =
	pos + libfhi::Vector2(spotr * cosf(pi), spotr * sinf(pi));

      // Save to dot.
      explmap->set_center(hitpos);
      dot->tp = hitpos;

      // Check for possible collision.
      CollisionSquare *sq = terrain->get_collision_square(explmap);
      CollisionMap *cm = sq->collides_single_dot(explmap);

      if(cm)
      {
	EntityBase *host = cm->get_host();

	for(std::list<EntityBase*>::iterator i = harm_list.begin(),
	    e = harm_list.end(); ((i != e) && host); ++i)
	{
	  if((*i) == host)
	  {
	    host = NULL;
	  }
	}

	if(host)
	{
	  harm_list.push_back(host);
	}
      }
    }

    // Give damage to everyone in list.
    int faction = this->cmap->get_faction();
    for(std::list<EntityBase*>::iterator i = harm_list.begin(),
	e = harm_list.end(); (i != e); ++i)
    {
      Entity *ent = (*i)->get_entity();

      if(ent->get_absorb_state() > 0)
      {
	ent->absorb_bullet(this->id);
      }
      else if(!is_faction_same((*i)->get_faction(), faction))
      {
	(*i)->take_damage(dmg);
      }
    }
  }
}

//############################################################################
// Branchers #################################################################
//############################################################################

/** Draw this weapon.
 * @param view View vector.
 */
void Weapon::draw(const libfhi::Vector2 &view)
{
  if(!this->has_instant())
  {
    this->draw_bullet(view);
  }
}

/** Tick this weapon.
 */
void Weapon::tick()
{
  if(this->has_instant())
  {
    this->tick_instant();
  }
  else
  {
    this->tick_bullet();
  }
}

//############################################################################
// Bullet-specific ###########################################################
//############################################################################

/** Specialized draw method.
 * @param view Camera position.
 */
void Weapon::draw_bullet(const libfhi::Vector2 &view)
{
  // It might be even bullet weapons have no meshes.
  if(this->mesh == NULL)
  {
    return;
  }

  // Get target mesh.
  libfhi::Mesh *tgtmesh = this->model->get_mesh();

  // Get count to draw.
  size_t drawcnt = stdmin(this->bullets.size(), this->visible_count);

  // Get source and destination vertex arrays.
  libfhi::Vector3 *dst = tgtmesh->get_array_vertex(),
    *src = this->mesh->get_array_vertex();

  // Get vertex number.
  size_t vnum = this->mesh->get_num_vertex();

  // Initialize the number of actually drawn bullets.
  size_t actually_drawn = 0;

  // Iterate the list, only draw up to drawcnt 
  for(std::list<Munition*>::iterator iter = this->bullets.begin();
      (drawcnt--);
      ++iter)
  {
    if((*iter)->inside(view, this->maxdist))
    {
      (*iter)->transform(src, dst, vnum);
      dst += vnum;
      ++actually_drawn;
    }
  }

  // Set counts for drawing.
  tgtmesh->set_num_vertex(actually_drawn * vnum);

  // Set element counts.
  for(int i = 0; (i < static_cast<int>(libfhi::MESH_ELEMENT_COUNT)); ++i)
  {
    libfhi::ElementType etype = static_cast<libfhi::ElementType>(i);
    MESH_TYPEDEF_INDEX *selem = this->mesh->get_array_elem(etype);

    if(selem)
    {
      tgtmesh->set_num_elem(etype,
	  static_cast<MESH_TYPEDEF_INDEX>((*selem) * actually_drawn));
    }
  }

  // Draw the model.
  libfhi::Surface::draw_model(this->model);
}

/** Specialized tick method.
 */
void Weapon::tick_bullet()
{
  Terrain *terrain = Terrain::instance;

  bool homing_mode = false,
       absorbable = !this->has_unabsorbable();
  float max_velocity = 1.0f;

  // Initialize max velocity here every frame. It is a miniscule cost, and
  // using here allows for initializing base weapon classes with no reagard
  // to current terrain parameters.
  if(this->has_homing())
  {
    homing_mode = true;
    max_velocity = Entity::calculate_max_velocity(this->drag_v,
	this->drag_h, this->mass);
  }

  // I have no idea, but cast errors unless I use void. Not using prefetched
  // end, since the list may be altered.
  for(std::list<Munition*>::iterator iter = this->bullets.begin();
      (iter != this->bullets.end());)
  {
    bool has_to_explode = false;

    // Get current iterator and advance.
    std::list<Munition*>::iterator curr = iter;
    ++iter;

    // Advance, delete if lifetime out.
    bool has_life = (homing_mode) ?
      (*curr)->tick_homing(this->thurst, this->drag_v, this->drag_h,
	  this->mass, max_velocity, this->rotspeed) :
      (*curr)->tick_bullet();

    if(!has_life)
    {
      // Play fade sample if exists.
      if(this->sample_fade)
      {
	this->sample_fade->Play((*curr)->get_pos());
      }

      Effect *efu = (homing_mode) ? this->hit : this->fade;
      efu->execute(&((*curr)->get_pos()), &((*curr)->get_dir()));

      delete (*curr);
      this->bullets.erase(curr);

      continue;
    }
    // If has a trail, leave it here.
    else if(this->has_trail())
    {
      this->fade->execute(&((*curr)->get_pos()), &((*curr)->get_dir()));
    }

    // Write the bullet's faction to the collision map.
    this->cmap->set_faction((*curr)->get_faction());

    // Get potential collision.
    this->cmap->transform_single_dot((*curr)->get_pos());
    CollisionSquare *sq = terrain->get_collision_square(this->cmap);
    CollisionMap *cm = sq->collides_single_dot(this->cmap);

    // If happened, explode and remove.
    if(cm)
    {
      // Get object we hit.
      EntityBase *host = cm->get_host();
      Entity *ent = host->get_entity();
      int fac1 = (*curr)->get_faction(),
	  fac2 = host->get_faction();

      if(absorbable && (ent->get_absorb_state() > 0))
      {
	ent->absorb_bullet(this->id);
      }
      else if(!is_faction_same(fac1, fac2))
      {
	//std::cout << "Taking damage " << (*curr)->get_faction() << ", " <<
	//  host->get_faction() << "\n";
	//std::cout << "op1 " << fac1 << ", op2 " << fac2 << "Masked: " <<
	//  (fac1 & FACTION_MASK) << " - " << (fac2 & FACTION_MASK) << "\n";
  	host->take_damage((*curr)->get_damage());
      }

      has_to_explode = true;
    }
    // If miss, try to collide with terrain.
    else if(sq->collides_terrain_single_dot(this->cmap))
    {
      has_to_explode = true;
    }

    // If scheduled to explode.
    if(has_to_explode)
    {
      this->explode((*curr)->get_pos(), (*curr)->get_dir(),
	  (*curr)->get_damage());

      delete (*curr);
      this->bullets.erase(curr);
    }
  }
}

//############################################################################
// Instant-specific ##########################################################
//############################################################################

/** Specialized tick method.
 */
void Weapon::tick_instant()
{
  Terrain *terrain = Terrain::instance;

  for(std::list<Munition*>::iterator iter = this->bullets.begin();
      (iter != this->bullets.end());)
  {
    // Get current iterator and advance.
    std::list<Munition*>::iterator curr = iter;
    ++iter;

    // Delete if lifetime out. With instant weapons, almost all the time.
    if(!(*curr)->tick_instant())
    {
      delete (*curr);
      this->bullets.erase(curr);

      continue;
    }

    // Write the bullet's faction to the collision map.
    this->cmap->set_faction((*curr)->get_faction());

    // Initialize the collision variables.
    CollisionMap::CollisionLine *line = this->cmap->get_single_line();
    float beamlen = 0.0f,
	  beam_steplen = 0.5f - 2.0f * line->r;
    libfhi::Vector2
      beam = (*curr)->get_pos(),
      beam_step = (*curr)->get_dir() * beam_steplen;

    // This is needed externally.
    CollisionSquare *sq = NULL;

    this->cmap->set_longest_radius(0.25f);

    // Loop until a hit is encountered or length runs out.
    while(!sq && (beamlen < this->length))
    {
      libfhi::Vector2 beamend = beam + beam_step;

      // Call the beam drawing.
      this->fade->execute(&beam, &beam_step);

      line->tp1 = beam;
      line->tp2 = beamend;
      this->cmap->set_center((beam + beamend) * 0.5);

      // Get collision square.
      sq = terrain->get_collision_square(this->cmap);
      CollisionMap *cm = sq->collides_single_line(this->cmap);

      // If happened, explode and remove.
      if(cm)
      {
	// Get object we hit.
	EntityBase *host = cm->get_host();
	Entity *ent = host->get_entity();

	if(ent->get_absorb_state() > 0)
	{
	  ent->absorb_bullet(this->id);
	}
	else if(!is_faction_same((*curr)->get_faction(), host->get_faction()))
	{
	  host->take_damage((*curr)->get_damage());
	}

	// sq left non-null, will explode.
	break;
      }
      // If miss, try to collide with terrain.
      else if(sq->collides_terrain_single_line(this->cmap))
      {
	// sq left non-null, will explode.
	break;
      }

      sq = NULL;
      beamlen += beam_steplen;
      beam = beamend;
    }
	
    // If scheduled to explode.
    if(sq)
    {
      libfhi::Vector2 hit_spot = 
	CollisionMap::get_collision_position() + sq->get_trans();

      this->explode(CollisionMap::get_collision_position() + sq->get_trans(),
	  (*curr)->get_dir(), (*curr)->get_damage());
    }
  }
}

//############################################################################
// End #######################################################################
//############################################################################

