#include "levelselector.h"

#include "commands.h"
#include "levelitem.h"
#include "menu.h"
#include "options.h"

#include "SDL.h"

#include <sstream>

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor for levelselector.
 * @param filename File to load the definitions from.
 */
LevelSelector::LevelSelector(const char *filename) :
  current_item(0)
{
  libfhi::ConfigFile filu(filename);

  if(!filu.is_ok())
  {
    std::cerr << "Could not load level list data from \"" << filename <<
      "\"\n";
    return;
  }

  // These are consistently filled. Note that the order is important, first,
  // the name and texture are specified, after they are done, the 
  std::string
    nextdesc,
    nextname,
    nexttex;

  while(filu.advance())
  {
    // Specify name for next item.
    if(filu.has_id("name"))
    {
      nextname = filu.get_combined_string();
    }
    // Description for next item.
    else if(filu.has_id("description"))
    {
      nextdesc = filu.get_combined_string();
    }
    // Specify texture for next item.
    else if(filu.has_id_arg("texture", 1))
    {
      nexttex = filu.get_string(0);
    }
    // Invoke all the following under this filename.
    else if(filu.has_id_arg("file", 1))
    {
      // If not found, add to the map and nullify score file.
      if(this->item_map.find(filu.get_string(0)) == this->item_map.end())
      {
	LevelItem *it = new LevelItem(filu.get_string(0).c_str(),
	    nextname.c_str(),
	    nextdesc.c_str(),
	    nexttex.c_str());

  	this->items.push_back(it);
	this->item_map[filu.get_string(0)] = it;
	it->set_best_score(0, "Anonymous");
	it->set_best_time(INT_MAX, "Anonymous");
      }
    }
  }
}

/** Default destructor.
 */
LevelSelector::~LevelSelector()
{
  // Iterate through all the levels and delete each and every one of them.
  for(std::vector<LevelItem*>::iterator i = this->items.begin(),
      e = this->items.end(); (i != e); ++i)
  {
    delete (*i);
  }
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Read scores from a file and save the score filename for further score
 * updates.
 * @param filename Score file name.
 */
void LevelSelector::read_scores(const char *filename)
{
  libfhi::ConfigFile filu(filename);

  // Set the score file now that we see it has been opened.
  this->score_file = filename;

  if(!filu.is_ok())
  {
    std::cerr << "Warning: could not open score file \"" << filename <<
      "\"\n";
    return;
  }

  // Needed for reading.
  std::string
    name,
    author_score,
    author_time;
  int best_score = 0,
      best_time = INT_MAX;

  while(filu.advance())
  {
    if(filu.has_id("name"))
    {
      name = filu.get_combined_string();
    }
    else if(filu.has_id_arg("score", 1))
    {
      best_score = filu.get_int(0);
      author_score = name;
    }
    else if(filu.has_id_arg("time", 1))
    {
      best_time = filu.get_int(0);
      author_time = name;
    }
    else if(filu.has_id_arg("file", 1))
    {
      std::map<std::string, LevelItem*>::iterator iter =
	this->item_map.find(filu.get_string(0));

      // If found, update the scores that have been specified.
      if(iter != this->item_map.end())
      {
	if(best_score != 0)
	{
	  (*iter).second->set_best_score(best_score, author_score);
	}
	if(best_time != INT_MAX)
	{
  	  (*iter).second->set_best_time(best_time, author_time);
	}
      }

      // Null these again.
      best_score = 0;
      best_time = INT_MAX;
    }
  }
}

/** Saves the scores into a score file.
 */
void LevelSelector::save_scores()
{
  if(this->score_file.length() <= 0)
  {
    return;
  }

  FILE *filu = fopen(this->score_file.c_str(), "wt");

  if(filu == NULL)
  {
    std::cerr << "Error: could not open score file \"" << this->score_file <<
      "\" for writing.\n";
  }

  // Seek to beginning.
  fseek(filu, 0, SEEK_SET);

  for(std::vector<LevelItem*>::iterator i = this->items.begin(),
      e = this->items.end(); (i != e); ++i)
  {
    bool needwrite = false;

    if((*i)->get_best_time() != INT_MAX)
    {
      fprintf(filu, "name %s\ntime %i\n", (*i)->get_author_time().c_str(),
	  (*i)->get_best_time());
      needwrite = true;
    }

    if((*i)->get_best_score() != 0)
    {
      fprintf(filu, "name %s\nscore %i\n", (*i)->get_author_score().c_str(),
	  (*i)->get_best_score());
      needwrite = true;
    }

    if(needwrite)
    {
      fprintf(filu, "file %s\n", (*i)->get_file());
    }
  }

  // Colse the file before exit.
  fclose(filu);
}

//############################################################################
// Virtual methods ###########################################################
//############################################################################

/** Draw this into view area.
 * @param screen Screen to draw into.
 */
void LevelSelector::draw(libfhi::Surface *screen)
{
  int x1 = 0,
      y1 = 0,
      x2 = screen->get_w() - 1,
      y2 = screen->get_h() - 1,
      hx = x2 / 2,
      color_text = libfhi::Surface::makecol3(186, 186, 186),
      color_score = libfhi::Surface::makecol3(190, 190, 255);

  // Use the whole screen.
  screen->clear();
  screen->set_boundary(x1, y1, x2, y2);
  screen->select_2d();

  libfhi::Surface::nc_rect(x1, y1, x2, y1 + 1, Menu::get_color_border());
  libfhi::Surface::nc_rect(x1, y2 - 1, x2, y2, Menu::get_color_border());
  libfhi::Surface::nc_rect(x1, y1 + 2, x1 + 1, y2 - 2,
      Menu::get_color_border());
  libfhi::Surface::nc_rect(x2 - 1, y1 + 2, x2, y2 - 2,
      Menu::get_color_border());
  libfhi::Surface::nc_rect(hx - 1, y1 + 2, hx, y2 - 2,
      Menu::get_color_border());

  LevelItem *it = this->items[this->current_item];

  std::stringstream left;
  left << "Level " << static_cast<uint32_t>(this->current_item + 1) <<
    "\n\n" << it->get_name();

  std::stringstream score;
  score << "Best score:\n\n";
  if(it->get_best_score() > 0)
  {
    score << it->get_best_score() << " by " << it->get_author_score();
  }
  else
  {
    score << "No record";
  }
  score << "\n\nBest time:\n\n";
  if(it->get_best_time() < INT_MAX)
  {
    score << it->get_best_time() << " by " << it->get_author_time();
  }
  else
  {
    score << "No record";
  }

  std::stringstream desc;
  desc << "Description:\n\n" << it->get_description();

  Menu::draw_text_box(x1 + 10, y1 + 10, hx - 12, y2 - 10, left.str().c_str(),
      Menu::get_color_title());

  Menu::draw_text_box(x1 + 10, (y1 + y2) / 5, hx - 12, y2 - 10,
      score.str().c_str(), color_score);

  Menu::draw_text_box(x1 + 10, (y1 + y2) / 2, hx - 12, y2 - 10,
      desc.str().c_str(), color_text);

  // Draw the texture to the right side of the screen.
  libfhi::Texture *tex = it->get_texture();
  if(tex)
  {
    libfhi::Surface::draw_texture(hx + 1, y1 + 2, x2 - 2, y2 - 2, tex);
  }
}

/** Implementation of keydown.
 * @param key Key pressed.
 */
void LevelSelector::keydown(int key)
{
  int curritem = static_cast<int>(this->current_item),
      numitems = static_cast<int>(this->items.size());
      
  // Down.
  if((key == SDLK_DOWN) || (key == SDLK_RIGHT) ||
      (key == Options::key_down) || (key == Options::key_right))
  {
    LevelItem *it = this->items[curritem];

    if((it->get_best_time() != INT_MAX) || (it->get_best_score() != 0))
    {
      this->current_item = stdmin(curritem + 1, numitems - 1);
    }
  }
  // Up.
  else if((key == SDLK_UP) || (key == SDLK_LEFT) ||
      (key == Options::key_up) || (key == Options::key_left))
  {
    int nextitem = stdmax(curritem - 1, 0);

    LevelItem *it = this->items[nextitem];

    if((it->get_best_time() != INT_MAX) || (it->get_best_score() != 0))
    {
      this->current_item = nextitem;
    }
  }
  // Escape.
  else if((key == SDLK_ESCAPE) || (key == Options::key_absorb) ||
      (key == SDLK_BACKSPACE))
  {
    command_end_game();
  }
}

//############################################################################
// End #######################################################################
//############################################################################

