#ifndef entityarchtype_h
#define entityarchtype_h

//############################################################################
// Include ###################################################################
//############################################################################

#include "enums.h"

#include "libfhi.h"

//############################################################################
// Define ####################################################################
//############################################################################

// Forward declaration.
class CollisionMap;
class Effect;
class Sample;
class Thurster;
class Turret;
class WeaponSlot;

//############################################################################
// Class #####################################################################
//############################################################################

/** Entity represents one physical object that has properties that affect it's
 * mobility, and one game object that has model (or models), collision maps
 * and weapons.
 */
class EntityArchtype :
  public libfhi::Attributable,
  public libfhi::CanonicalDB<EntityArchtype>
{
  public:
    /** Movement toggle, if true, moves like in classic shooters. */
    static const uint8_t ATTR_DANMAKU = 0x01;

    /** Enables absorb command and alters a whole lot of stuff happening. */
    static const uint8_t ATTR_ABSORB = 0x02;

    /** Enables slowdown in survival mode (for easier gameplay). */
    static const uint8_t ATTR_SLOWDOWN = 0x04;

    /** Enables enemies that try to hit the player and explode. */
    static const uint8_t ATTR_SUICIDE = 0x08;

    /** Default attributes. */
    static const uint8_t ATTR_DEFAULT = 0x00;

  protected:
    /** Type of this entity. */
    EntityEnum type;

    /** Name of this entity. */
    std::string name;

    /** Mass of this (kg). */
    float mass;

    /** Thurst strength (newton). */
    float thurst[THURSTER_COUNT];

    /** Vertical drag coefficient. */
    float drag_vertical;

    /** Horizontal drag coefficient. */
    float drag_horizontal;

    /** Rotation speed (over z axis) of this. */
    int rotspeed;

    /** Forward thurst. */
    std::vector<Thurster*> thursters[THURSTER_COUNT];

    /** List of meshes. */
    std::vector<libfhi::Mesh*> meshes;

    /** List of distinct collision maps. */
    std::vector<CollisionMap*> maps;

    /** List of distinct weapon slots. */
    std::vector<WeaponSlot*> weaponslots;

    /** List of turrets. */
    std::list<Turret*> turrets;

    /** The longest radius this will be drawn in. */
    float longest_radius;

    /** Maximum life of this. */
    int life_maximum;

    /** AI aiming proximity. */
    float ai_aim_proximity;

    /** Bleeding sparks. */
    Effect *bleed;

    /** Dying sparks. */
    Effect *die;

    /** Number of die spark thingies to generate. */
    int die_multitude;

    /** Sample file for explosion. */
    Sample *die_sample;

    /** Object lower left corner (for bleeding and dying). */
    libfhi::Vector2 corner_ll;

    /** Object upper right corner (for bleeding and dying). */
    libfhi::Vector2 corner_ur;

  public:
    EntityArchtype();
    EntityArchtype(const char*);
    ~EntityArchtype();

  public:
    void fire_thursters(ThursterEnum, const libfhi::Vector2&,
	const libfhi::Vector2&, uint16_t);
    void invoke_sparks(const Effect*, const libfhi::Vector2&, uint16_t) const;

  public:
    inline float get_ai_aim_proximity() const;
    inline Effect *get_bleed();
    inline std::vector<CollisionMap*>* get_collisionmaps();
    inline const libfhi::Vector2& get_corner_ll() const;
    inline const libfhi::Vector2& get_corner_ur() const;
    inline Effect *get_die();
    inline int get_die_multitude() const;
    inline Sample* get_die_sample();
    inline float get_drag_horizontal() const;
    inline float get_drag_vertical() const;
    inline int get_life_maximum() const;
    inline float get_longest_radius() const;
    inline float get_mass() const;
    inline std::vector<libfhi::Mesh*>* get_meshes();
    inline const std::string& get_name() const;
    inline int get_rotspeed() const;
    inline float get_thurst(ThursterEnum) const;
    inline EntityEnum get_type() const;
    inline std::list<Turret*>* get_turrets();
    inline std::vector<WeaponSlot*>* get_weaponslots();
    inline bool has_absorb() const;
    inline bool has_danmaku() const;
    inline bool has_slowdown() const;
    inline void set_type(EntityEnum);
};

//############################################################################
// Inline methods ############################################################
//############################################################################

/** Get the aim proximity of this.
 * @return Aim proximity as float.
 */
float EntityArchtype::get_ai_aim_proximity() const
{
  return this->ai_aim_proximity;
}

/** Get the bleed effect.
 * @return pointer to bleed effect.
 */
Effect* EntityArchtype::get_bleed()
{
  return this->bleed;
}

/** Get all the collision maps of this.
 * @return Pointer to collision map vector.
 */
std::vector<CollisionMap*>* EntityArchtype::get_collisionmaps()
{
  return &(this->maps);
}

/** Get the lower left corner of all the meshes of this.
 * @return Vector reference.
 */
const libfhi::Vector2& EntityArchtype::get_corner_ll() const
{
  return this->corner_ll;
}

/** Get the upper right corner of all the meshes of this.
 * @return Vector reference.
 */
const libfhi::Vector2& EntityArchtype::get_corner_ur() const
{
  return this->corner_ur;
}

/** Get the die effect.
 * @return pointer to die effect.
 */
Effect* EntityArchtype::get_die()
{
  return this->die;
}

/** Get multitude of die effect.
 * @return Count as int.
 */
int EntityArchtype::get_die_multitude() const
{
  return this->die_multitude;
}

/** Get sound effect to play upon death.
 * @return Sound effect pointer.
 */
Sample* EntityArchtype::get_die_sample()
{
  return this->die_sample;
}

/** Get the horizontal drag of this.
 * @return Drag as float.
 */
float EntityArchtype::get_drag_horizontal() const
{
  return this->drag_horizontal;
}

/** Get the vertical drag of this.
 * @return Drag as float.
 */
float EntityArchtype::get_drag_vertical() const
{
  return this->drag_vertical;
}

/** Get the maximum life of this.
 * @return Lifemax as int.
 */
int EntityArchtype::get_life_maximum() const
{
  return this->life_maximum;
}

/** Get the farthest point this from the center of this to any vertex.
 * @return Distance as float.
 */
float EntityArchtype::get_longest_radius() const
{
  return this->longest_radius;
}

/** Get the mass of this model.
 * @return Mass as float.
 */
float EntityArchtype::get_mass() const
{
  return this->mass;
}

/** Get vector containing all the meshes.
 * @return Pointer to the mesh vector.
 */
std::vector<libfhi::Mesh*>* EntityArchtype::get_meshes()
{
  return &(this->meshes);
}

/** Get the name of this.
 * @return Reference to the name.
 */
const std::string& EntityArchtype::get_name() const
{
  return this->name;
}

/** Get the rotation speed of this.
 * @return Rotation speed as integer.
 */
int EntityArchtype::get_rotspeed() const
{
  return this->rotspeed;
}

/** Get the thurst.
 * @param op Thurster id.
 * @return Thurst as float.
 */
float EntityArchtype::get_thurst(ThursterEnum op) const
{
  return this->thurst[op];
}

/** Get the type of this entity.
 * @return Entity enum.
 */
EntityEnum EntityArchtype::get_type() const
{
  return this->type;
}

/** Get the turrets of this.
 * @return Pointer to turret vector.
 */
std::list<Turret*>* EntityArchtype::get_turrets()
{
  return &(this->turrets);
}

/** Get the list of weapon slots in this model.
 * @return Pointer to weapon slot vector.
 */
std::vector<WeaponSlot*>* EntityArchtype::get_weaponslots()
{
  return &(this->weaponslots);
}

/** Tell if this has an absorb mode.
 * @return True if yes, false if no.
 */
bool EntityArchtype::has_absorb() const
{
  return this->has_attr(ATTR_ABSORB);
}

/** Tell if this has a danmaku mode.
 * @return True if yes, false if no.
 */
bool EntityArchtype::has_danmaku() const
{
  return this->has_attr(ATTR_DANMAKU);
}

/** Tell if this has slowdown if so said.
 * @return True if yes, false if no.
 */
bool EntityArchtype::has_slowdown() const
{
  return this->has_attr(ATTR_SLOWDOWN);
}

/** Set the type of this entity. Only called by Game.
 * @param op New Type.
 */
void EntityArchtype::set_type(EntityEnum op)
{
  this->type = op;
}

//############################################################################
// End #######################################################################
//############################################################################

#endif

