#include "entity.h"

#include "collisionmap.h"
#include "data.h"
#include "effect.h"
#include "entityarchtype.h"
#include "game.h"
#include "sound.h"
#include "turret.h"
#include "weaponslot.h"

#include <float.h>

//############################################################################
// Constants #################################################################
//############################################################################

#ifdef DEBUG
libfhi::PostModel *pathmodel = NULL;
#endif

//############################################################################
// Construction ##############################################################
//############################################################################

/** Initialization from target. This is used to initialize the actual entities
 * after reference entities have been loaded from a file. The method performs
 * a deep copy of all relevant data.
 * @param op Source entity archtype.
 * @param f Faction to create under.
 */
Entity::Entity(EntityArchtype *op, int f) :
  EntityBase(f, op->get_life_maximum()), base_class(op), position(0.0f, 0.0f),
  movement(0.0f, 0.0f), accel(0.0f, 0.0f), absorb_fire(false),
  absorb_state(0), absorb_wait(0), charge_now(0), charge_last(0),
  ai_last_path_search(-AI_PATH_SEARCH_DELAY), ai_target(NULL)
{
  // Danmaku thingies point up!
  this->angle = op->has_danmaku() ? 16383 : 0;
  this->tilt = 0;

  // Copy values.
  this->mass = op->get_mass();
  for(int i = 0; (i < static_cast<int>(THURSTER_COUNT)); ++i)
  {
    this->thurst[i] = op->get_thurst(static_cast<ThursterEnum>(i));
  }
  this->drag_vertical = op->get_drag_vertical();
  this->drag_horizontal = op->get_drag_horizontal();
  this->rotspeed = op->get_rotspeed();

  // Add collision map copies.
  std::vector<CollisionMap*> *srcmaps = op->get_collisionmaps();
  for(std::vector<CollisionMap*>::iterator i = srcmaps->begin(),
      e = srcmaps->end(); (i != e); ++i)
  {
    this->add(*i);
  }

  // Add models generated on meshes there.
  std::vector<libfhi::Mesh*> *srcmeshes = op->get_meshes();
  for(std::vector<libfhi::Mesh*>::iterator i = srcmeshes->begin(),
      e = srcmeshes->end(); (i != e); ++i)
  {
    this->add(*i);
  }

  // Add weapon slot copies.
  std::vector<WeaponSlot*> *srcweaponslots = op->get_weaponslots();
  for(std::vector<WeaponSlot*>::iterator i = srcweaponslots->begin(),
      e = srcweaponslots->end(); (i != e); ++i)
  {
    this->add(*i);
  }

  // Deep copy turrets.
  std::list<Turret*> *srcturrets = op->get_turrets();
  for(std::list<Turret*>::iterator i = srcturrets->begin(),
      e = srcturrets->end(); (i != e); ++i)
  {
    this->add(*i);
  }

  // Clear absorb ammo.
  memset(this->absorb_ammo, 0, sizeof(int) * WEAPON_COUNT);

  // Entity-specific switch here.
  switch(op->get_type())
  {
      // Light bydo needs the different rotators here.
    case ENTITY_LIGHTBYDO:
      for(int i = 0; (i < 4); ++i)
      {
    	for(int j = 0; (j < 3); ++j)
    	{
    	  this->lightbydo.ang_diff[i][j] =
	    (rand() % (LIGHTBYDO_REVOLVE_SPEED + 1)) -
	    LIGHTBYDO_REVOLVE_SPEED / 2;
    	  this->lightbydo.ang_act[i][j] = rand() & 0x0000FFFF;
    	}
      }
      break;

    default:
      break;
  }

  // Now, if we have slowdown we need to also slow some entities down to give
  // the light bydo a fighting chanse in the survival mode. The slowdown is
  // directly proportional to the weapon slowdown.
  if(op->has_slowdown())
  {
    float slowdown = Weapon::get_slowdown(),
	  slowdown_sqr = slowdown * slowdown;

    for(int i = 0; (i < static_cast<int>(THURSTER_COUNT)); ++i)
    {
      this->thurst[i] *= slowdown_sqr;
    }
    this->mass *= slowdown_sqr;
    this->drag_vertical /= slowdown;
    this->drag_horizontal /= slowdown;
  }

  // Calculate the maximum velocity for this entity. Must be done here,
  // since dependant on the terrain.
  this->max_velocity = this->calculate_max_velocity(this->drag_vertical,
      this->drag_horizontal, this->mass);
}

/** Default destructor.
 */
Entity::~Entity()
{
  for(std::vector<libfhi::PostModel*>::iterator i = this->models.begin(),
      e = this->models.end(); (i != e); ++i)
  {
    delete *i;
  }
  for(std::vector<CollisionMap*>::iterator i = this->maps.begin(),
      e = this->maps.end(); (i != e); ++i)
  {
    delete *i;
  }
  for(std::list<Turret*>::iterator i = this->turrets.begin(),
      e = this->turrets.end(); (i != e); ++i)
  {
    delete *i;
  }
  for(std::vector<WeaponSlot*>::iterator i = this->weaponslots.begin(),
      e = this->weaponslots.end(); (i != e); ++i)
  {
    delete *i;
  }
}

//############################################################################
// Initialization ############################################################
//############################################################################

/** Append a new collision map to this a model that is a copy of given map.
 * @param op Mesh to use.
 */
void Entity::add(CollisionMap *op)
{
  // Warn if open has been unsuccessful.
  if(op == NULL)
  {
    std::cerr << "Warning: Trying to copy nonextint collision map.\n";
  }

  this->maps.push_back(new CollisionMap(op, this));
}

/** Append a new model using given mesh as a base to this model.
 * @param op Mesh to use.
 */
void Entity::add(libfhi::Mesh *op)
{
  // Warn if open has been unsuccessful.
  if(op == NULL)
  {
    std::cerr << "Warning: Trying to initialize from nonextint mesh.\n";
  }

  libfhi::PostModel *mdl = new libfhi::PostModel(op);

  // Copy user attributes from mesh.
  if(op->has_attr(libfhi::Mesh::ATTR_USER_1))
  {
    mdl->attr_or(libfhi::PostModel::ATTR_USER_1);
  }
  if(op->has_attr(libfhi::Mesh::ATTR_USER_2))
  {
    mdl->attr_or(libfhi::PostModel::ATTR_USER_2);
  }

  // Set rotation and push back the object.
  mdl->set_rotation(libfhi::Orientation::ROTATION_ABSORB);
  this->models.push_back(mdl);
}

/** Copy one turret to a new turret in this.
 * @param op Turret to copy.
 */
void Entity::add(Turret *op)
{
  if(op == NULL)
  {
    std::cerr << "Warning: Trying to copy nonextint turret.\n";
  }

  this->turrets.push_back(new Turret(op, this));
}

/** Copy one weapon mount to a new slot in this.
 * @param op Mount to use
 */
void Entity::add(WeaponSlot *op)
{
  // Warn if open has been unsuccessful.
  if(op == NULL)
  {
    std::cerr << "Warning: Trying to copy nonextint weapon slot.\n";
  }

  this->weaponslots.push_back(new WeaponSlot(op));
}

//############################################################################
// Static methods ############################################################
//############################################################################

/** Calculate the maximum safe velocity this can go before bad things happen,
 * all movement will be clamped to this velocity.
 */
float Entity::calculate_max_velocity(float drag_v, float drag_h, float mass)
{
  // Terrain is needed, because certain physics values depend on it
  Terrain *terrain = Terrain::instance;

  // Save to the max_velocity variable.
  return PHYSICS_DRAG_DIVERGENCE_SAFETY_LIMIT * 4.0f * mass /
    (stdmax(drag_v, drag_h) * terrain->get_rho() * PHYSICS_STEP_TIME);
}

//############################################################################
// Virtual methods ###########################################################
//############################################################################

/** Return the entity of this, in this case, just return me.
 */
Entity* Entity::get_entity()
{
  return this;
}

/** Take given amount of damage.
 * @param amount Amount of damage.
 */
void Entity::take_damage(int amount)
{
  this->life -= amount;
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Absorb one bullet of given id.
 */
void Entity::absorb_bullet(WeaponEnum id)
{
  int ammo_max = Game::instance->get_weapon(id)->get_absorb();

  if(++this->absorb_ammo[id] > ammo_max)
  {
    this->take_damage(1);
    this->absorb_ammo[id] = ammo_max;
  }
}

/** Clears this model from terrain. This is done upon death, since leaving
 * references to collision maps of this could be hazardous considering the AI
 * methods of other entities. No real reason to set the square to NULL, as the
 * entity is soon dead anyway.
 */
void Entity::clear_from_terrain()
{
  for(std::vector<CollisionMap*>::iterator i = this->maps.begin(),
      e = this->maps.end(); (i != e); ++i)
  {
    CollisionSquare *sq = (*i)->get_square();

    if(sq)
    {
      sq->clear_maps();
    }
  }
}

/** Collide with a collision line (in other words, terrain).
 * @param op Line the collision happened with.
 */
void Entity::collide_with(CollisionMap::CollisionLine *op)
{
  float dot = op->tnor.product_dot(this->movement);

  this->movement -= op->tnor * 2.0f * dot;
  this->accel.set(0.0f, 0.0f);

  // Move the entity outwards along the line normal.
  this->position += op->tnor * op->r;
}

/** Collide with does a physical collision between two objects.
 * Tries to preserve momentum.
 * @param op The other Physical model this collided with.
 * @param line The line collision happened in.
 */
void Entity::collide_with(Entity *op)
{
  float mass_sum = this->mass + op->get_mass();
  libfhi::Vector2 v1, v2;

  v1 = (this->movement * (this->mass - op->get_mass()) +
      op->get_mov() * (2.0f * op->get_mass())) / mass_sum;
  v2 = (op->get_mov() * (op->get_mass() - this->mass) +
      this->movement * (2.0f * this->mass)) / mass_sum;

  this->set_mov(v1);
  op->set_mov(v2);

  // Remove acceleration since there may be problems.
  this->accel.set(0.0f, 0.0f);
  op->accel.set(0.0f, 0.0f);

  // Push smaller object outwards along the line normal.
  if(this->mass > op->get_mass())
  {
    CollisionMap::CollisionLine *line = CollisionMap::get_line_lhs();

    if(line)
    {
      op->position += line->tnor * line->r;
    }
    else if((line = CollisionMap::get_line_rhs()) != NULL)
    {
      op->position -= line->tnor * line->r;
    }
    else // Push the objects apart a bit.
    {
      libfhi::Vector2 diff = CollisionMap::get_dot_difference();
      float len = diff.length();

      if(len > 0.0f)
      {
	diff /= 20.0f * len;
      }

      op->position += diff;
    }
  }
  else
  {
    CollisionMap::CollisionLine *line = CollisionMap::get_line_rhs();

    if(line)
    {
      this->position += line->tnor * line->r;
    }
    else if((line = CollisionMap::get_line_lhs()) != NULL)
    {
      op->position -= line->tnor * line->r;
    }
    else
    {
      libfhi::Vector2 diff = CollisionMap::get_dot_difference();
      float len = diff.length();

      if(len > 0.0f)
      {
	diff /= 20.0f * len;
      }

      this->position -= diff;
    }
  }
}

/** Wrapper for desire for allowing calls with points instead of angles.
 * @param tpos Target position.
 * @param target Target position this tries to look at.
 */
void Entity::desire(const libfhi::Vector2 &tpos,
    const libfhi::Vector2 &target)
{
  // Get destination orientation.
  libfhi::Vector2 ddir = Terrain::instance->get_relative_bounded_position(
      this->position, target);

  // Get lenght and potentially rotate.
  float len = ddir.length();

  if(len > 0.0f)
  {
    ddir /= len;

    this->desire(tpos, libfhi::pi2int(atan2f(ddir.yf, ddir.xf)));
  }
  else
  {
    this->desire(tpos, this->angle);
  }
}

/** Desire calculates the the new movement vector for this object that wishes
 * to be in halt at a position - angle pair given to it. This does not use
 * the same methods as normal command input, so artificial intelligence can
 * cheat here (if need).
 * @param tpos Target position.
 * @param dang Destination angle this entity wishes to be looking at.
 */
void Entity::desire(const libfhi::Vector2& tpos, int dang)
{
  // Then try to rotate to correct direction.
  int angle_before = angle;
  this->angle = Entity::ai_try_rotate(this->angle, dang, this->rotspeed);

  // Tilt according to difference angle.
  int diffang = this->angle - angle_before;
  if(diffang > 32768)
  {
    this->tilt += this->rotspeed;
  }
  else if(diffang < -32768)
  {
    this->tilt -= this->rotspeed;
  }
  else if(diffang < 0)
  {
    this->tilt += this->rotspeed;
  }
  else if(diffang > 0)
  {
    this->tilt -= this->rotspeed;
  }

  // Now get orientation again.
  float pi = libfhi::uint2pi(this->angle),
	cr = cosf(pi),
	sr = sinf(pi);

  // Create thurst as empty.
  libfhi::Vector2 firethurst(0.0f, 0.0f);

  // Get difference and clamp it.
  libfhi::Vector2 unrotated_diff = Terrain::instance->
    get_relative_bounded_position(this->position, tpos);

  // Only apply any thurst if we are not in our destination.
  if(unrotated_diff.length() > 0.0f)
  {
    // Rotate diff with dir.
    float ang = -atan2f(sr, cr);
    libfhi::Vector2 diff;
    diff.rotate(unrotated_diff, cosf(ang), sinf(ang));

    libfhi::Vector2 mov = this->movement * PHYSICS_STEP_TIME;

    // Right thurster.
    if(diff.yf < 0.0f)
    {
      this->fire_thurst(THURST_RT, cr, sr, mov, firethurst);
    }
    // Left thurster.
    else if(diff.yf > 0.0f)
    {
      this->fire_thurst(THURST_LT, cr, sr, mov, firethurst);
    }

    // Front thurster
    if(diff.xf > 0.0f)
    {
      this->fire_thurst(THURST_FW, cr, sr, mov, firethurst);
    }
    // Back thurster.
    else if(diff.xf < 0.0f)
    {
      this->fire_thurst(THURST_BK, cr, sr, mov, firethurst);
    }

    this->normalize_thurst(firethurst);
  }

  //getc(stdin);

  // Now that we know our thurst, call the static physics method.
  // actual changes to movement.
  Entity::apply_physics(
      this->position,
      this->movement,
      //this->accel,
      firethurst,
      cr,
      sr,
      this->drag_vertical,
      this->drag_horizontal,
      this->mass,
      this->max_velocity);

  // Return the tilt.
  this->tilt_return();
}

/** Fire all weapons of type given by enum. Most of the time, the weapons will
 * just return while doing nothing.
 * @param op Weapon types to fire.
 */
void Entity::fire()
{
  // Absorbing objects fire in different manner, and only on the keypress,
  // not by holding down.
  if(this->base_class->has_absorb() &&
      ((this->charge_now == 1) || this->absorb_fire))
  {
    this->absorb_fire = false;

    std::vector<std::pair<float, Entity*> > *targets =
      Entity::ai_get_proximity_targets(this->faction, ABSORB_FIRE_TARGETS,
	  this->position, ABSORB_FIRE_MAX_RANGE_SQR);

    // Cease fire if no targets.
    if(targets->size() <= 0)
    {
      delete targets;
      return;
    }

    // Waiting time left
    int wait = 0;

    // Get game struct.
    Game *game = Game::instance;

    // Iterate through all weapons, fire everywhere if need be.
    for(int i = 0; (i < static_cast<int>(WEAPON_COUNT)); ++i)
    {
      // Only one shot per frame.
      if(this->absorb_ammo[i] > 0)
      {
	// Get the weapon used.
	Weapon *weap = game->get_weapon(static_cast<WeaponEnum>(i));

	// Mark we are still firing.
	this->absorb_fire = true;

	// Update waiting time.
	wait = stdmax(wait, --this->absorb_ammo[i]);

	// Pick random target off our list.
	Entity *tgt = (*targets)[rand() % targets->size()].second;

	// Calculate lookahead required.
	float lookahead = Entity::ai_aim_lookahead(this->position,
	    this->movement * PHYSICS_STEP_TIME,
	    tgt->get_pos(),
	    weap->get_speed());

	// Only fire on legal input.
	if(lookahead != FLT_MAX)
	{
	  weap->add(this->faction,
	      this->position,
	      this->movement * PHYSICS_STEP_TIME,
	      cosf(lookahead),
	      sinf(lookahead),
	      1,
	      weap->get_damage());
	}
      }
    }

    // Update our absorb time quarantine.
    this->absorb_wait = stdmax(wait, this->absorb_wait);
    delete targets;
  }
  // If normal object, check if there is a way we can actually fire, if yes,
  // iterate through weapon slots and fire.
  else
  {
    libfhi::Vector2 mov = this->movement * PHYSICS_STEP_TIME;

    for(std::vector<WeaponSlot*>::iterator i = this->weaponslots.begin(),
	e = this->weaponslots.end(); (i != e); ++i)
    {
      (*i)->fire(this->faction, this->position, mov, this->angle,
	  this->charge_now, this->charge_last);
      (*i)->tick();
    }
  }

  // Update charge.
  this->charge_last = this->charge_now;
}

/** Fire thurst to some direction. Also modifies tilting.
 * @param thid Thurster id.
 * @param cr Cosine of current direction.
 * @param sr Sine of current direction.
 * @param mov Tick-multiplied movement vector.
 * @param thvec Vector to increment.
 */
void Entity::fire_thurst(ThursterEnum thid, float cr, float sr,
    const libfhi::Vector2 &mov, libfhi::Vector2 &thvec)
{
  float magnitude = thurst[thid];

  // Calculate thurst vector. For reminder on how this should go, vectors are
  // as follows:
  // fw(cr, sr)
  // bk(-cr, -sr)
  // rt(sr, -cr)
  // lt(-sr, cr)
  switch(thid)
  {
    case THURST_FW:
      thvec.xf += cr * magnitude;
      thvec.yf += sr * magnitude;
      break;
    case THURST_BK:
      thvec.xf -= cr * magnitude;
      thvec.yf -= sr * magnitude;
      break;
    case THURST_RT:
      thvec.xf += sr * magnitude;
      thvec.yf -= cr * magnitude;
      this->tilt += this->rotspeed >> 1;
      break;
    default: // THURST_LT
      thvec.xf -= sr * magnitude;
      thvec.yf += cr * magnitude;
      this->tilt -= this->rotspeed >> 1;
      break;
  }

  // Call thruster display.
  this->base_class->fire_thursters(thid, this->position, mov, this->angle);
}

/** Fire turrets of this entity. Also check their AI.
 */
void Entity::fire_turrets()
{
  // Abort if not turrets.
  if(this->turrets.empty())
  {
    return;
  }

  // Get actual per-frame movement.
  libfhi::Vector2 mov = this->movement * PHYSICS_STEP_TIME;

  // Iterate all turrets and call their AI.
  for(std::list<Turret*>::iterator i = this->turrets.begin(),
      e = this->turrets.end(); (i != e); ++i)
  {
    (*i)->ai_turret(this->position, mov, this->angle);
  }
}

/** Inputs commands here.
 * @param array Command array that contains all the command enumerations.
 */
void Entity::input_commands(int *array)
{
  // Copy charge.
  this->charge_now = array[COMMAND_FIRE];

  // Rotate before thurst.
  if(!this->base_class->has_danmaku())
  {
    // Normal commands also include rotation.
    int
      left = stdmin(array[COMMAND_LEFT],
	  ROTATION_ACCELERATION_TIME),
      right = stdmin(array[COMMAND_RIGHT],
	  ROTATION_ACCELERATION_TIME);

    if(left > 0)
    {
      this->tilt -= this->rotspeed >> 1;
    }
    if(right > 0)
    {
      this->tilt += this->rotspeed >> 1;
    }

    this->angle += static_cast<int>(
	static_cast<float>(left - right) /
	static_cast<float>(ROTATION_ACCELERATION_TIME) *
  	static_cast<float>(this->rotspeed));
  }

  // Get orientation.
  float pi = libfhi::uint2pi(this->angle),
	cr = cos(pi),
	sr = sin(pi);

  libfhi::Vector2
    firethurst(0.0f, 0.0f),
    mov = this->movement * PHYSICS_STEP_TIME;

  // Forward and back.
  if(array[COMMAND_FORWARD])
  {
    this->fire_thurst(THURST_FW, cr, sr, mov, firethurst);
  }
  if(array[COMMAND_BACKWARD])
  {
    this->fire_thurst(THURST_BK, cr, sr, mov, firethurst);
  }

  // Danmaku maps directional pad to movement.
  if(this->base_class->has_danmaku())
  {
    if(array[COMMAND_RIGHT])
    {
      this->fire_thurst(THURST_RT, cr, sr, mov, firethurst);
    }
    if(array[COMMAND_LEFT])
    {
      this->fire_thurst(THURST_LT, cr, sr, mov, firethurst);
    }

    // Danmaku has no tilt, no matter which way we turn.
    this->tilt = 0;
  }
  // With normal commands absorb / target are strafe.
  else
  {
    if(array[COMMAND_ABSORB])
    {
      this->fire_thurst(THURST_RT, cr, sr, mov, firethurst);
    }
    if(array[COMMAND_TARGET])
    {
      this->fire_thurst(THURST_LT, cr, sr, mov, firethurst);
    }

    this->normalize_thurst(firethurst);
  }

  // Now that we know our thurst, call the static physics method to calculate
  // actual changes to movement.
  Entity::apply_physics(this->position,
      this->movement,
      //this->accel,
      firethurst,
      cr,
      sr,
      this->drag_vertical,
      this->drag_horizontal,
      this->mass,
      this->max_velocity);

  // Physics have been applied. Fire, the player ALWAYS fires at normal speed
  // regardless of the slowdown in game.
  float slowdown = Weapon::get_slowdown();
  Weapon::set_slowdown(1.0f);
  this->fire();
  this->fire_turrets();
  Weapon::set_slowdown(slowdown);

  // Perform absorb mode checks.
  if(this->base_class->has_absorb())
  {
    // If button is pressed, turn absorb state on.
    if(array[COMMAND_ABSORB] > 0)
    {
      if(this->absorb_state >= 0)
      {
	if(this->absorb_state <= ABSORB_TIME_MAX)
	{
	  ++this->absorb_state;
	}
	else
	{
	  this->absorb_state = -this->absorb_state;
	}
      }
      // If absorb state is negative (recharging), it is possible to flip
      // over to real absorb state supposing that we are not over the wait
      // threshold.
      else if(this->absorb_state < 0)
      {
	if(this->absorb_state > -ABSORB_WAIT_THRESHOLD)
	{
	  this->absorb_state = 1 - this->absorb_state;
	}
	else
	{
	  ++this->absorb_state;
	}
      }
    }
    else
    {
      // Otherwise decrease waiting time of absorb.
      if(this->absorb_state < 0)
      {
	++this->absorb_state;
      }
      // Or turn from absorb mode to wait mode.
      else if(this->absorb_state > 0)
      {
	this->absorb_state = -this->absorb_state;
      }
    }
  }

  // At this point, tilt has been modified, return it back a bit.
  this->tilt_return();
}

/** Insert his object into the terrain. Simply add the entity to the terrain
 * collision data structures and do not care for any possible underlying
 * colliding maps. To perform insert with checks, use try_insert instead.
 */
void Entity::insert()
{
  Terrain *terrain = Terrain::instance;

  float pi = libfhi::uint2pi(this->angle),
	cr = cosf(pi),
	sr = sinf(pi);

  for(std::vector<CollisionMap*>::iterator i = this->maps.begin(),
      e = this->maps.end(); (i != e); ++i)
  {
    // Transform the map.
    (*i)->transform(this->position, cr, sr);

    // J-J-JAM IT IN!
    terrain->get_collision_square(*i)->insert(*i);
  }
}

/** Invokes death sparks (the amount is within the entity.
 */
void Entity::invoke_death_sparks()
{
  int cnt = this->base_class->get_die_multitude();
  Effect *die = this->base_class->get_die();

  while(cnt--)
  {
    this->base_class->invoke_sparks(die, this->position, this->angle);
  }

  // Loop through turrets and invoke their death too.
  for(std::list<Turret*>::iterator i = this->turrets.begin(),
      e = this->turrets.end(); (i != e); ++i)
  {
    (*i)->invoke_death();
  }

  // Play the death sound while at it.
  Sample *death_sample = this->base_class->get_die_sample();
  if(death_sample)
  {
    death_sample->Play(this->position);
  }
}

/** Normalizes the thurst vector given as parameter.
 * @param thvec Reference parameter thurst vector.
 */
void Entity::normalize_thurst(libfhi::Vector2 &thvec)
{
  // Prevent straferunning by normalizing thurst vector size to thurst_fw
  // (if greater).
  float len = thvec.length_sqr(),
	magnitude = this->thurst[THURST_FW];

  if(len > magnitude * magnitude)
  {
    thvec *= sqrt(len) / magnitude;
  }
}

/** Return the tilt torwards zero state.
 */
void Entity::tilt_return()
{
  int return_speed = this->rotspeed >> 2,
      rotmax = this->rotspeed << 3;

  if(tilt < -return_speed)
  {
    tilt += return_speed;

    if(tilt < -rotmax)
    {
      tilt = -rotmax;
    }
  }
  else if(tilt > return_speed)
  {
    tilt -= return_speed;

    if(tilt > rotmax)
    {
      tilt = rotmax;
    }
  }
  else
  {
    tilt = 0;
  }
}

/** Try to insert this into the playfield. Return false if there is something
 * in the squares it would be going to. Otherwise insert to those squares and
 * return true.
 */
bool Entity::try_insert()
{
  Terrain *terrain = Terrain::instance;

  float pi = libfhi::uint2pi(this->angle),
	cr = cosf(pi),
	sr = sinf(pi);

  // Iterate through all collision maps.
  for(std::vector<CollisionMap*>::iterator i = this->maps.begin(),
      e = this->maps.end(); (i != e); ++i)
  {
    // Transform the collision map.
    (*i)->transform(this->position, cr, sr);

    // Get the square and possible first colliding object.
    CollisionSquare *sq = terrain->get_collision_square(*i);
    CollisionMap *hit = sq->collides_freeform(*i);

    // Check for hits, if hits detected, return false.
    if((hit) && (hit->get_host() != this))
    {
      return false;
    }
    else
    {
      CollisionMap::CollisionLine *line =
	sq->collides_terrain_freeform(*i);
      if(line)
      {
	return false;
      }
    }
  }

  // Insert and return.
  this->insert();
  return true;
}

//############################################################################
// Virtual methods ###########################################################
//############################################################################

/** AI method.
 */
void Entity::ai()
{
  switch(this->base_class->get_type())
  {
    case ENTITY_R_11A:
    case ENTITY_R_11S2:
    case ENTITY_R_13A:
    case ENTITY_R_13B:
    case ENTITY_R_9A:
    case ENTITY_R_9B2:
    case ENTITY_R_9D2:
    case ENTITY_R_9D:
    case ENTITY_R_9DH2:
    case ENTITY_R_9DV2:
    case ENTITY_R_9K:
    case ENTITY_R_9O:
    case ENTITY_RX_10:
    case ENTITY_TX_T:
      // Note others here anyways, since they needs to home.
    case ENTITY_GENON:
    case ENTITY_NULAYRAGOM:
    case ENTITY_PSYVARIAR_BOSS:
    case ENTITY_RVA_818:
      this->ai_rtype();
      break;

    case ENTITY_INTERDIMENSIONAL_BATTLESHIP:
      this->ai_battleship();
      break;

    default:
      this->ai_dumbwalker();
      break;
  }
}

/** Draw exports this Entity onto a libfhi surface in 3d. Draw has a
 * default implementation that may be overridden.
 * @param view Viewport.
 * @param screen Surface to draw onto.
 */
void Entity::draw(const libfhi::Vector2 &view)
{
  Terrain *terrain = Terrain::instance;

  libfhi::Vector2 tpos = terrain->get_relative_bounded_position(view,
      this->position);

  if(!Game::inside_view_area(tpos, this->base_class->get_longest_radius()))
  {
    return;
  }

  // Were inside. Safe.
  for(size_t i = 0; (i < this->models.size()); ++i)
  {
    libfhi::PostModel *mdl = this->models[i];

    if((mdl->has_attr(libfhi::PostModel::ATTR_USER_1) &&
	  (this->absorb_state <= 0)) ||
	(mdl->has_attr(libfhi::PostModel::ATTR_USER_2) &&
	 (this->absorb_state > 0)))
    {
      // There is some entity-specific code here.
      switch(this->base_class->get_type())
      {
	case ENTITY_LIGHTBYDO:
	  mdl->set_angles(this->lightbydo.ang_act[i][0],
	      this->lightbydo.ang_act[i][1],
	      this->lightbydo.ang_act[i][2]);
	  break;
	default:
  	  mdl->set_angles(this->tilt, 0, this->angle);
	  break;
      }

      mdl->set_pos(tpos.xf, tpos.yf, 0.0f);
      mdl->tick();
      libfhi::Surface::draw_model(mdl);
    }
  }

  // Now, loop the turrets and draw them.
  for(std::list<Turret*>::iterator i = this->turrets.begin(),
      e = this->turrets.end(); (i != e); ++i)
  {
    (*i)->draw(tpos, this->angle);
  }
}

/** Tick contains all the functionality Entitys need to be ran each tick.
 * This includes such things a AI, animation, collisions, etc. Note that since
 * the damage is checked upon entering tick, in some cases objects may receive
 * lethal damage and still live to see the next frame. Basically this doesn't
 * pretty much do anything, since one frame is heck of a small time.
 * @return True if the object still exists, false if no.
 */
bool Entity::tick()
{
  // Check that this has not received lethal damage. If yes, abort.
  if(this->life <= 0)
  {
    return false;
  }

  // Get terrain.
  Terrain *terrain = Terrain::instance;

  float pi = libfhi::uint2pi(this->angle),
	cr = cosf(pi),
	sr = sinf(pi);

  // Iterate through all collision maps.
  for(std::vector<CollisionMap*>::iterator i = this->maps.begin(),
      e = this->maps.end(); (i != e); ++i)
  {
    if(((*i)->has_attr(CollisionMap::ATTR_NORMAL) &&
	  (this->absorb_state <= 0)) ||
	((*i)->has_attr(CollisionMap::ATTR_ABSORB) &&
	 (this->absorb_state > 0)))
    {

      // Transform the collision map.
      (*i)->transform(this->position, cr, sr);

      // Get the square and possible first colliding object.
      CollisionSquare *sq = terrain->get_collision_square(*i);
      CollisionMap *hit = sq->collides_freeform(*i);

      // If a hit happened
      if((hit) && (hit->get_host() != this))
      {
	this->collide_with(hit->get_host()->get_entity());
      }
      else
      {
	CollisionMap::CollisionLine *line =
	  sq->collides_terrain_freeform(*i);
	if(line)
	{
	  this->collide_with(line);
	}
      }

      // Insert into the square found.
      sq->insert(*i);
    }
  }

  // Iterate all turrets and plant their collision maps onto the world.
  for(std::list<Turret*>::iterator i = this->turrets.begin(),
      e = this->turrets.end(); (i != e); ++i)
  {
    (*i)->insert_collision_maps(this->position, cr, sr, this->angle);
  }

  // Entity-specific tickings.
  switch(this->base_class->get_type())
  {
    case ENTITY_LIGHTBYDO:
      for(int i = 0; (i < 4); ++i)
      {
    	for(int j = 0; (j < 3); ++j)
    	{
    	  this->lightbydo.ang_act[i][j] += this->lightbydo.ang_diff[i][j];
    	}
      }
      // Lightbydo regenerates.
      this->life = stdmin(this->life + LIGHTBYDO_REGEN_SPEED,
	  this->base_class->get_life_maximum());
      break;

    default:
      break;
  }

  // Check if we're in bad shape. If yes, let's bleed.
  if(this->life < this->base_class->get_life_maximum() / ENTITY_BLEED_BORDER)
  {
    if((rand() & ENTITY_BLEED_PROBABILITY) == 0)
    {
      this->base_class->invoke_sparks(this->base_class->get_bleed(),
	  this->position, this->angle);
    }
  }

  return true;
}

//############################################################################
// AiSearch helper class #####################################################
//############################################################################

/** Helper class to store in the priority queue.
 */
class AiSearch
{
  private:
    /** Heuristic of this square. */
    float heuristic;

    /** Amount of heuristic this will add if traversed. */
    float add;

    /** Referenced square. */
    CollisionSquare *square;

  public:
    AiSearch(float, float, CollisionSquare*);
    ~AiSearch();

    inline float get_add() const;
    inline float get_heuristic() const;
    inline CollisionSquare* get_square() const;
    inline void operator=(const AiSearch&);
    inline bool operator<(const AiSearch&) const;
};

/** Default constructor.
 * @param h Heuristic to use.
 * @param a Addition.
 * @param s Referenced square.
 */
AiSearch::AiSearch(float h, float a, CollisionSquare *s)
  : heuristic(h), add(a), square(s)
{
  // Do nothing.
}

/** Default destructor.
 */
AiSearch::~AiSearch()
{
  // Do nothing.
}

/** Get the addition.
 * @return The number this will increment the cumulative heuristic with.
 */
float AiSearch::get_add() const
{
  return this->add;
}

/** Get the search heuristic value of this.
 * @return Value as float.
 */
float AiSearch::get_heuristic() const
{
  return this->heuristic;
}

/** Get the square pointer.
 * @return CollisionSquare pointer.
 */
CollisionSquare* AiSearch::get_square() const
{
  return this->square;
}

/** Assignment operator.
 * @param rhs Right-hand side operand.
 */
void AiSearch::operator=(const AiSearch &rhs)
{
  this->heuristic = rhs.heuristic;
  this->square = rhs.square;
}

/** Comparison operator for priority queue.
 * @param rhs Right-hand side operand.
 */
bool AiSearch::operator<(const AiSearch &rhs) const
{
  // Inverted since we are minimizing, not maximizing.
  return (this->heuristic > rhs.heuristic);
}

//############################################################################
// AI functions ##############################################################
//############################################################################

/** Method to make ai follow the path it has been given.
 * @return True if there is still stuff to follow, false if req. rethink.
 */
bool Entity::ai_follow_path()
{
  static const float SKIPPROX = AI_REACH_PROXIMITY * AI_REACH_PROXIMITY;
  int j = 1, skip = 0;

  // Check that we're not near any other entities.
  CollisionSquare *sq = Terrain::instance->get_collision_square(
      static_cast<int>(this->position.xf),
      static_cast<int>(this->position.yf));
  if(sq->has_nearby_entities(this))
  {
    return false;
  }

  for(std::list<libfhi::Vector2>::iterator i = this->ai_path.begin();
      ((i != this->ai_path.end()) && (j <= AI_NODE_SKIP));
      ++j)
  {
    // Advance iterator.
    std::list<libfhi::Vector2>::iterator curr = i;
    ++i;

    // Calculate difference to position.
    libfhi::Vector2 diff = (*curr) - this->position;

    if(((diff.xf * diff.xf) + (diff.yf * diff.yf)) <= SKIPPROX)
    {
      skip = j;
      break;
    }
  }

  // Pop all the skipped nodes.
  if(skip)
  {
    do
    {
      ai_path.pop_front();
    } while(--skip);
  }

  // Return and tell if we have an empty path.
  return !this->ai_path.empty();
}

/** Get a point near the given point that is allowed for AI travel.
 * @param pos Position to look from.
 * @param sqdist Square distance to do do the selection in.
 * @return Vector2 containing a target point.
 */
libfhi::Vector2 Entity::ai_get_allowed_vincinity_point(
    const libfhi::Vector2 &pos, float hoverdist)
{
  Terrain *terrain = Terrain::instance;

  int x = static_cast<int>(pos.xf) - 1,
      y = static_cast<int>(pos.yf) - 1;

  // Get our current difference vector.
  libfhi::Vector2 diff = this->position - pos;
  float difflen = diff.length();
  if(difflen > 0.0f)
  {
    diff /= difflen;
  }
  else
  {
    diff.set(1.0f, 0.0f);
  }

  // Try at most some number of times, then give up.
  int retry_count = AI_VINCINITY_RETRY_COUNT;
  do {
    // We want to turn less than maximum to avoid running onto the player.
    float angdiff = ((rand() % 2001) - 1000) * 0.001f;
    angdiff = angdiff * fabsf(angdiff) * static_cast<float>(M_PI);

    // Same with the distance, rather far than very close.
    float radius = (rand() % 1001) * 0.001f;
    radius = hoverdist * (1.0f - radius * radius);

    libfhi::Vector2 newhover;
    newhover.rotate_translate(diff,
	radius * cosf(angdiff),
	radius * sinf(angdiff),
	pos);

    // Get actual position.
    int sx = static_cast<int>(newhover.xf) - 1,
	sy = static_cast<int>(newhover.yf) - 1;

    if((abs(x - sx) <= 1) || (abs(y - sy) <= 1))
    {
      continue;
    }

    // Get the randomized square from terrain.
    CollisionSquare *sq = terrain->get_collision_square(sx, sy);

    // Return if square is allowed.
    if(sq->is_safe_to_travel(this))
    {
      return sq->get_trans() + libfhi::Vector2(1.0f, 1.0f);
    }
  } while(--retry_count);

  // Nothing sensible found. Just return our current position and do not move.
  return this->position;
}

/** Get the nearest position eglible to travel from the front of the path.
 */
const libfhi::Vector2& Entity::ai_get_path_front() const
{
  return this->ai_path.empty() ? this->position : this->ai_path.front();
}

/** Get a point fixed number of squares ahead in our path.
 * @param cnt How many nodes to look ahead (mut be > 0)?
 * @return Vector position of lookahead node.
 */
const libfhi::Vector2& Entity::ai_path_lookahead(int cnt)
{
  // If empty path, just return our position.
  if(this->ai_path.empty())
  {
    return this->position;
  }

  std::list<libfhi::Vector2>::iterator iter = this->ai_path.begin();

  while(cnt-- > 0)
  {
    std::list<libfhi::Vector2>::iterator next = iter;
    ++next;

    if(next == this->ai_path.end())
    {
      return (*iter);
    }

    ++iter;
  }

  return (*iter);
}

/** Erases the current path of ai and replaces it with a new search path to
 * given position. Do NOT make closed spaces, AI will go nuts.
 * @param tgt Target position.
 */
void Entity::ai_search_path(libfhi::Vector2 tgt)
{
  static const libfhi::Vector2 SQCENTER_ADD(1.0f, 1.0f);

  // Keep in same order as with the collision squares.
  static const float NEIGHBOR_DIST[8] =
  {
    sqrtf(2.0f) * sqrtf(2.0f),
    1.0f,
    sqrtf(2.0f) * sqrtf(2.0f),
    1.0f,
    1.0f,
    sqrtf(2.0f) * sqrtf(2.0f),
    1.0f,
    sqrtf(2.0f) * sqrtf(2.0f)
  };

  // Takes care of the search id.
  static int search_id = 0;
  ++search_id;

  // Get terrain.
  Terrain *terrain = Terrain::instance;

  // Take source and destination squares (according to positions).
  CollisionSquare
    *start = terrain->get_collision_square(
	static_cast<int>(this->position.xf) - 1,
	static_cast<int>(this->position.yf) - 1),
    *end = terrain->get_collision_square(
	static_cast<int>(tgt.xf) - 1,
	static_cast<int>(tgt.yf) - 1),
    *current = start,
    *previous = NULL;

  // If destination square is illegal, abort. Otherwise we are going to
  // search, so clear the current path.
  if(!end->is_safe_to_travel(this))
  {
    //std::cout << "Destination not allowed for AI, forgetting.\n";
    return;
  }
  this->ai_path.clear();

  // To avoid heuristics errors, set the target to exactly in the square
  // center.
  tgt = end->get_trans() + SQCENTER_ADD;

  // We need square path and priority queue.
  std::priority_queue<AiSearch> pqueue;

  // Get heuristics.
  float cumulative_heuristic = 0.0f;

  // Some precautions.
  end->set_ai_previous(NULL);
  start->set_ai_previous(NULL);

  //std::cout << "Aiming to: " << end->get_trans() << " from " <<
  //  start->get_trans() << "\n";

  // This WILL loop forever if there is no route.
  while(current)
  {
    // If there is something in the queue, replace current with the topmost
    // item.
    if(!pqueue.empty())
    {
      current = pqueue.top().get_square();
      cumulative_heuristic += pqueue.top().get_add();
      pqueue.pop();

      //std::cout << "Popped square: " << current->get_trans() << "\n";
    }
    else if(current == previous)
    {
      //std::cout << "Error: nowhere to go!\n";
      return;
    }

    // Abort if reached end.
    if(current == end)
    {
      break;
    }

    previous = current;

    // Push all nearby squares to priority queue (hopefully this is unrolled).
    for(int i = 0; (i < 8); ++i)
    {
      CollisionSquare *neighbor = current->get_neighbor(i);

      //std::cout << "Checking neighbor " << neighbor->get_trans();

      // AI evades squares with collision stuff.
      if(neighbor->is_safe_to_travel(this))
      {
	float new_heuristic;

	// Otherwise calculate stuff.
	new_heuristic = terrain->heuristic_distance(tgt,
	    neighbor->get_trans() + SQCENTER_ADD) +
	  NEIGHBOR_DIST[i] + cumulative_heuristic;

	//std::cout << " - allowed, heuristic: " << new_heuristic << "\n";

	// If neighbor already is in search and it's heuristic is better than
	// the new one, skip.
	if((!neighbor->include_in_search(search_id)) &&
	    (neighbor->get_ai_heuristic() <= new_heuristic))
	{
	  //std::cout << "Existed with better or equal heuristic, aborting.\n";
	  continue;
	}

	// Was not better or was not in search at all, set values.
	neighbor->set_ai_heuristic(new_heuristic);
	neighbor->set_ai_previous(current);

	//std::cout << "Setting previous of " << neighbor->get_trans() << " to "
	//  << current->get_trans() << "\n";

	// Push a new one into the fray.
	pqueue.push(AiSearch(new_heuristic, NEIGHBOR_DIST[i], neighbor));
      }
      else
      {
	//std::cout << " - not allowed for AI.\n";
      }
    }
  }

#ifdef DEBUG
  libfhi::PreModel *pre = new libfhi::PreModel();
#endif

  // Starting from end, we may now give the path as coordinates by following
  // the previous references until we reach start. Precautions will make
  // sure the whole thing is cancelled if we encounter NULL:s.
  for(current = end; ((current != NULL) && (current != start));
      current = current->get_ai_previous())
  {
    //std::cout << "Previous node for current is " <<
    //  current->get_ai_previous()->get_trans() << "\n";

#ifdef DEBUG
    libfhi::ComponentVertex *ver1, *ver2; 

    if(this->ai_path.size() > 0)
    {
      ver1 = pre->add_vertex(
	  libfhi::Vector3(
	    this->ai_path.front().xf,
	    this->ai_path.front().yf,
	    0.0f),
	  libfhi::Color4(1.0f, 0.0f, 0.0f, 1.0f),
	  libfhi::Vector2(0.0f, 0.0f));
    }
#endif

    libfhi::Vector2 dst = current->get_trans() + SQCENTER_ADD;
    dst.xf = libfhi::congr(dst.xf, terrain->get_size_x());
    dst.yf = libfhi::congr(dst.yf, terrain->get_size_y());

    this->ai_path.push_front(dst);

#ifdef DEBUG
    if(this->ai_path.size() > 1)
    {
      ver2 = pre->add_vertex(
	  libfhi::Vector3(
	    this->ai_path.front().xf,
	    this->ai_path.front().yf,
	    0.0f),
	  libfhi::Color4(1.0f, 0.0f, 0.0f, 1.0f),
	  libfhi::Vector2(0.0f, 0.0f)),

      pre->add_face(ver1->get_idx(), ver2->get_idx());
    }
#endif
  }

#ifdef DEBUG
  if(pathmodel)
  {
    delete pathmodel->get_mesh();
    delete pathmodel;
  }
  pathmodel = new libfhi::PostModel(pre->compile(NULL));
  delete pre;
#endif
}

/** Selects a new target for the ai.
 */
libfhi::Vector2 Entity::ai_select_target(bool allow_homing)
{
  // Get suitable target list.
  std::vector<std::pair<float, Entity*> > *targets =
    this->ai_get_proximity_targets(this->faction, AI_FEASIBLE_TARGETS,
	this->position, 0.0f);
  size_t target_cnt = targets->size();

  // If there was legal targets...
  if(target_cnt > 0)
  {
    // Now we have target_cnt targets in order closest - farthest. Select
    // in random (closest is most probable).
    size_t i = target_cnt;
    for(std::vector<std::pair<float, Entity*> >::iterator iter =
	targets->begin(); (true); ++iter, --i)
    {
      if((i <= 1) || (rand() % i))
      {
	this->ai_target = (*iter).second;
	break;
      }
    }
  }

  // We either have or do not have the target now, get rid of target list in
  // any case.
  delete targets;

  // Only get a point from near the target if we have a target and we are
  // allowed to home on it.
  if(this->ai_target && allow_homing)
  {
    return ai_get_allowed_vincinity_point(this->ai_target->get_pos(),
	AI_VINCINITY_SELECTION);
  }

  // Otherwise just get a completely random point in map.
  Terrain *terrain = Terrain::instance;
  return libfhi::Vector2(
      static_cast<float>(rand() %
	static_cast<int>(terrain->get_size_x() * 1000.0f)) * 0.001f,
      static_cast<float>(rand() %
	static_cast<int>(terrain->get_size_y() * 1000.0f)) * 0.001f);
}

/** Try to aim at a target.
 * @return True if was close enough (thus aimed) and moved.
 */
bool Entity::ai_try_aim()
{
  if(!Game::instance->entity_exists(this->ai_target))
  {
    this->ai_target = NULL;
    return false;
  }

  // If no weapons, do not aim (abort).
  if(this->weaponslots.empty())
  {
    return false;
  }

  // Calculate distance.
  const float AIM_SQR = this->base_class->get_ai_aim_proximity();

  libfhi::Vector2 diff = Terrain::instance->get_relative_bounded_position(
      this->ai_target->get_pos(), this->position);

  // If too far away, do not aim (abort).
  if(((diff.xf * diff.xf) + (diff.yf * diff.yf)) > AIM_SQR)
  {
    return false;
  }

  // Get the speed of our main weapon or segfault if none.
  float wspeed = this->weaponslots.front()->get_speed(),
	dang = Entity::ai_aim_lookahead(this->position,
	    this->movement * PHYSICS_STEP_TIME,
	    this->ai_target->get_pos(),
	    wspeed);

  // If it is not possible to hit, just try to look at target.
  if(dang == FLT_MAX)
  {
    this->desire(this->ai_get_path_front(),
	this->ai_target->get_pos());
    this->fire();
    this->fire_turrets();

    return true;
  }

  // Otherwise try to look at where we could hit the target.
  int iang = libfhi::pi2int(dang);
  this->desire(this->ai_get_path_front(), iang);

  // Check if we have permission to pull or release the trigger.
  bool arc_close_enough = (abs((iang - this->angle) % 65536) <
      AI_FIRE_ARC);

  // Get the next charge and fire (possibly).
  this->charge_now = Entity::ai_fire_decision(arc_close_enough,
      charge_now, &(this->weaponslots));
  this->fire();
  this->fire_turrets();

  return true;
}

//############################################################################
// Static AI #################################################################
//############################################################################

/** Approximator for aiming look-ahead. This method is NOT optimized, since
 * I could not read it if I did. It's constant-time anyway, and computer-
 * optimizable. No need to get worked about it.
 * @param srcpos Shooter's position.
 * @param mov Shooter's movement (per tick).
 * @param tpos Target position to aim at.
 * @param wspeed Speed of the weapon we are going to fire.
 * @return Angle to look at or negative FLT_MAX if unable to determine.
 */
float Entity::ai_aim_lookahead(const libfhi::Vector2 &srcpos,
    const libfhi::Vector2 &mov, const libfhi::Vector2 &tpos, float wspeed)
{
  // Get desire vector, then adjust it to unit length.
  libfhi::Vector2 diff = Terrain::instance->get_relative_bounded_position(
      srcpos, tpos);
  float dlen = diff.length();
  if(dlen == 0.0f)
  {
    return FLT_MAX;
  }
  diff /= dlen;

  // Project movement onto desire vector.
  float movproject = mov.product_dot(diff);

  // Get vector that is from the intersection point, to the end of mov.
  float catet = (mov - (diff * movproject)).length();

  // We know the hypotenuse and other catet, solve other catet unless
  // wspeed is less than catet (in which case there is no solution).
  if(wspeed <= catet)
  {
    return FLT_MAX;
  }

  float ilen = sqrtf(wspeed * wspeed - catet * catet) + movproject;

  // Get the intersection point.
  libfhi::Vector2 ipoint = diff * ilen;

  // Get the difference vector.
  libfhi::Vector2 req = ipoint - mov;

  // Unit the requirements vector.
  float reqlen = req.length();
  if(reqlen == 0.0f)
  {
    return FLT_MAX;
  }
  req /= reqlen;

  // Return the angle.
  return atan2f(req.yf, req.xf);
}

/** Get fire decision from artificial intelligence. Basically, if there is
 * ammo in reload cannons, try to fire, otherwise begin charging and do not
 * release until ready to fire.
 * @param permission Permission to release the trigger.
 * @param charge Current charge.
 * @param slots Weapon slots.
 * @return New charge.
 */
int Entity::ai_fire_decision(bool permission, int charge,
    std::vector<WeaponSlot*> *slots)
{
  // If we are charging AND have permission to release the trigger, check
  // for release.
  if(charge > 1)
  {
    if(permission)
    {
      bool had_charge_weapons = false;

      for(std::vector<WeaponSlot*>::iterator i = slots->begin(),
	  e = slots->end(); (i != e); ++i)
      {
	had_charge_weapons |= (*i)->get_mount()->has_charge();

	if((*i)->has_charge(charge))
	{
	  return 0;
	}
      }

      // If we were charging but did not have charge weapons, just fire for
      // fuck's sake.
      if(!had_charge_weapons)
      {
	return 1;
      }
    }

    // Didn't fire, had charge weapons. Return incremented charge.
    return charge + 1;
  }
  else
  {
    if(permission)
    {
      for(std::vector<WeaponSlot*>::iterator i = slots->begin(),
	  e = slots->end(); (i != e); ++i)
      {
	if((*i)->has_ammo())
	{
	  return 1;
	}
      }

      // Had permission but had nothing to fire, start charging.
      return charge + 1;
    }

    // No permission to fire, do nothing.
    return 0;
  }
}

/** Get a list of targets close to this. If the max_dist is smaller or equal
 * to zero, will not consider the distance at all. Note that the list
 * returned must be deleted since it is reserved in this method.
 * @param max_cnt Maximum count of targets.
 * @param max_dist Do not consider targets farther than this (squared).
 * @return List containig pairs of these.
 */
std::vector< std::pair<float, Entity*> >*
Entity::ai_get_proximity_targets(int faction, size_t max_cnt,
    const libfhi::Vector2 &srcpos, float max_dist)
{
  // Our return type.
  std::vector< std::pair<float, Entity*> > *ret =
    new std::vector< std::pair<float, Entity*> >();
  ret->reserve(max_cnt + 1);

  // Check if need to check against length.
  bool checklength = (max_dist > 0.0f);

  // The shortest thing we have in the list now.
  float shortest = FLT_MAX;

  // Get terrain and null our target.
  Terrain *terrain = Terrain::instance;

  // Get entity list from game singleton.
  std::list<Entity*> *entities = Game::instance->get_entity_list();

  // Iterate through the list and save a number of closest ones to the
  // feasible target list.
  for(std::list<Entity*>::iterator i = entities->begin(),
      e = entities->end(); (i != e); ++i)
  {
    // No targets of our faction.
    if(is_faction_same((*i)->get_faction(), faction))
    {
      continue;
    }

    // Get difference.
    libfhi::Vector2 diff = terrain->get_relative_bounded_position(
	srcpos, (*i)->get_pos());

    // Get length and abort if too great.
    float len = diff.length_sqr();
    if(checklength && (len > max_dist))
    {
      continue;
    }

    // If nothing in array, insert at beginning.
    if(ret->empty())
    {
      ret->push_back(std::pair<float, Entity*>(len, (*i)));
      shortest = len;
    }
    // Otherwise insertion sort into the array if we either have room or the
    // lenght specified is shorter than the shortest now here.
    else if((ret->size() < max_cnt) || (len < shortest))
    {
      bool inserted = false;

      for(std::vector<std::pair<float, Entity*> >::iterator iter =
	  ret->begin(), enditer = ret->end(); (iter != enditer); ++iter)
      {
	if((*iter).first > len)
	{
	  ret->insert(iter, std::pair<float, Entity*>(len, (*i)));

	  // If too much, pop the last and update shortest.
	  if(ret->size() > max_cnt)
	  {
	    ret->pop_back();
	    shortest = ret->back().first;
	  }

	  inserted = true;
	  break;
	}
      }

      if(!inserted)
      {
	ret->push_back(std::pair<float, Entity*>(len, (*i)));
	shortest = len;
      }
    }
  }

  return ret;
}

/** Rotate an angle towards target angle at given maximum speed.
 * @param dir The angle to rotate.
 * @param tgt Target angle.
 * @param rotspeed Rotation speed.
 * @return New angle.
 */
int Entity::ai_try_rotate(uint16_t dir, int tgt, int rotspeed)
{
  int dang = tgt - dir;

  // Get angle difference.
  if(dang > 32768)
  {
    dang -= 65536;
  }
  else if(dang < -32768)
  {
    dang += 65536;
  }
  else if(dang == 0)
  {
    return dir;
  }

  // Get absolute angle difference.
  int absang = abs(dang);

  // Rotate towards.
  if(absang < rotspeed)
  {
    return tgt;
  }
  return (dang >= 0) ? (dir + rotspeed) : (dir - rotspeed);
}

//############################################################################
// Specialized AI ############################################################
//############################################################################

/** The basic ai of ships tat do not try to aim at anything (i.e.) ones that
 * just pick random sports, move there, then let the turrets shoot.
 */
void Entity::ai_battleship()
{
  int frame_number = Game::get_frame_number();

  // Check if need to reselect path.
  if((!ai_follow_path() || (rand() % AI_RETHINK_BATTLESHIP == 0)) &&
      (frame_number - this->ai_last_path_search > AI_PATH_SEARCH_DELAY))
  {
    this->ai_last_path_search = frame_number;
    this->ai_search_path(this->ai_select_target(false));
  }

  // Just look forward and go. Then fire turrets.
  this->desire(this->ai_get_path_front(),
      this->ai_path_lookahead(AI_NODE_LOOKAHEAD));
  this->fire_turrets();
}

/** Ai for 'dumb walkers' i.e. entities that traverse randomly around the
 * screen doing nothing and fire at the player if he gets too close.
 */
void Entity::ai_dumbwalker()
{
  int frame_number = Game::get_frame_number();

  // Check if need to reselect path.
  if((!ai_follow_path() || (rand() % AI_RETHINK_DUMBWALKER == 0)) &&
      (frame_number - this->ai_last_path_search > AI_PATH_SEARCH_DELAY))
  {
    this->ai_last_path_search = frame_number;
    this->ai_search_path(this->ai_select_target(
	  (rand() % AI_DUMBWALKER_FORGET == 0) ? true : false));
  }

  // Try to look at target, return if succeeded in looking.
  if(this->ai_try_aim())
  {
    return;
  }

  // Just look forward and go. Then fire turrets.
  this->desire(this->ai_get_path_front(),
      this->ai_path_lookahead(AI_NODE_LOOKAHEAD));
  this->fire_turrets();
}

/** The basic ai of r-type series fighters. Stupid, yet hopefully efficient
 * enough.
 */
void Entity::ai_rtype()
{
  int frame_number = Game::get_frame_number();

  // Check if need to reselect path.
  if((!ai_follow_path() || (rand() % AI_RETHINK_RTYPE == 0)) &&
      (frame_number - this->ai_last_path_search > AI_PATH_SEARCH_DELAY))
  {
    this->ai_last_path_search = frame_number;
    ai_search_path(ai_select_target(true));
  }

  // Try to look at target, return if succeeded in looking.
  if(this->ai_try_aim())
  {
    return;
  }

  // Otherwise just look forward.
  this->desire(this->ai_get_path_front(),
      this->ai_path_lookahead(AI_NODE_LOOKAHEAD));
  this->fire_turrets();
}

//############################################################################
// End #######################################################################
//############################################################################

