#include "effect.h"

#include "data.h"
#include "effectdb.h"
#include "particlesystem.h"

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor.
 * @param op ConfigFile pointer to fetch data from.
 */
Effect::Effect()
  : Attributable(DEFAULT_ATTR), type(EFFECT_DEFAULT), system(NULL)
{
  // Do nothing.
}

/** Load constructor.
 * @param filename File to load from.
 */
Effect::Effect(const char *filename)
  : Attributable(DEFAULT_ATTR), system(NULL)
{
  libfhi::ConfigFile filu(filename);

  if(!filu.is_ok())
  {
    std::cerr << "Error: could not open effect: \"" << filename << "\"\n";
  }

  while(filu.advance())
  {
    if(filu.has_id_arg("system", 1))
    {
      this->system = Data::load_part(filu.get_string(0).c_str());
    }
    else if(filu.has_id_arg("type", 1))
    {
      this->type = static_cast<EffectEnum>(filu.get_int(0));
    }
    else if(filu.has_id_arg("attr", 1))
    {
      this->set_attr(filu.get_int(0));
    }
    else if(filu.has_id_arg("count", 2))
    {
      this->count_min = filu.get_int(0);
      this->count_max = filu.get_int(1);
    }
    else if(filu.has_id_arg("speed", 2))
    {
      this->speed_min = filu.get_float(0);
      this->speed_max = filu.get_float(1);
    }
    else if(filu.has_id_arg("lifetime", 1))
    {
      this->lifetime = filu.get_int(0);
    }
    else if(filu.has_id_arg("color", 4))
    {
      this->color.rf = filu.get_float(0);
      this->color.gf = filu.get_float(1);
      this->color.bf = filu.get_float(2);
      this->color.af = filu.get_float(3);
    }
    else
    {
      filu.warn_empty();
    }
  }

  std::cout << "Loaded effect \"" << filename << "\"\n";
}

/** Default destructor.
 */
Effect::~Effect()
{
  // Do nothing.
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Executes this effect.
 * @param op1 First vector parameter pointer.
 * @param op2 Second vector parameter pointer.
 */
void Effect::execute(const libfhi::Vector2 *op1,
    const libfhi::Vector2 *op2) const
{
  // If temporal, add this to the effect database and cease execution (this
  // will be executed via the effect database later).
  if(this->has_temporal())
  {
    EffectDB::instance->add(new EffectInstance(op1, op2, this,
	  this->lifetime));
  }
  // Otherwise every calculation neccessary can be done now at this point,
  // in which case execute them immediately.
  else
  {
    this->execute_now(op1, op2);
  }
}

/** Executes this effect on given parameters.
 * @param op1 First vector parameter pointer.
 * @param op2 Second vector parameter pointer.
 */
void Effect::execute_now(const libfhi::Vector2 *op1,
    const libfhi::Vector2 *op2) const
{
  switch(this->type)
  {
    case EFFECT_BEAM:
      {
	// op1 = position
	// op2 = end - position
	const float smul = 0.001f * (this->speed_max - this->speed_min);
	int cnt = this->get_cnt();

	while(cnt-- > 0)
	{
	  this->system->add(this->color,
	      (*op1) + (*op2) * static_cast<float>(rand() % 1001) * 0.001f,
	      static_cast<float>(rand() % 1001) * smul + this->speed_min,
	      rand() % 65535,
	      this->lifetime);
	}
      }
      break;

    case EFFECT_THURSTER:
      {
	// op1 = position
	// op2 = movement
	int cnt = this->get_cnt();

	while(cnt-- > 0)
	{
	  this->system->add(this->color,
	      (*op1),
	      (*op2),
	      this->lifetime);
	}
      }
      break;

    case EFFECT_CHARGE:
      {
	// op1 = position
	// op2 = movement
	const float
	  dist = this->speed_max *
 	      static_cast<float>(this->lifetime),
	  smul = 0.001f * (this->speed_max - this->speed_min);
	int cnt = this->get_cnt();

	while(cnt-- > 0)
	{
	  float pi = libfhi::uint2pi(static_cast<uint16_t>(rand() % 65536)),
		realspeed = static_cast<float>(rand() % 1001) * smul +
		  this->speed_min;
	  libfhi::Vector2 gendir(cosf(pi), sinf(pi));

	  this->system->add(this->color,
	      (*op1) - gendir * dist,
	      gendir * realspeed + (*op2),
	      this->lifetime);
	}
      }
      break;

    default: // Equals typical puff / explosion effect.
      {
	// op1 = position
	const float smul = 0.001f * (this->speed_max - this->speed_min);
	int cnt = this->get_cnt();

	while(cnt-- > 0)
	{
	  this->system->add(this->color,
	      (*op1),
	      static_cast<float>(rand() % 1001) * smul + this->speed_min,
	      rand() % 65536,
	      this->lifetime);
	}
      }
      break;
  }
}

//############################################################################
// End #######################################################################
//############################################################################

