#include "libfhi_texture.h"
#include "libfhi_misc.h"
#include "libfhi_glsurface.h"

#ifdef LIBFHI_OPENGL
#include LIBFHI_GLU_H
#endif

namespace libfhi {

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor.
 */
Texture::Texture()
#ifdef LIBFHI_OPENGL
  : texture_name(0)
#endif
{
  // Do nothing.
}

/** Load constructor.
 * @param filename Image file to open the texture from.
 */
Texture::Texture(const char *filename)
#ifdef LIBFHI_OPENGL
  : texture_name(0)
#endif
{
  this->load(filename);
}

/** Default destructor.
 */
Texture::~Texture()
{
#ifdef LIBFHI_OPENGL
  if(this->texture_name)
  {
    glDeleteTextures(1, &(this->texture_name));
  }
#endif
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Loader.
 * @param filename File to to load the texture from.
 * @return True on success, false on failure.
 */
bool Texture::load(const char *filename)
{
#ifdef LIBFHI_OPENGL
  if(!glSurface::is_opengl_initialized())
  {
    std::cerr << "Warning: Not in OpenGL mode, skipping texture \"" <<
      filename << "\"\n";
    return false;
  }

  // Open image file.
  sdlSurface *tex = sdlsurface_new(filename);

  if(tex)
  {
    GLenum format;
    switch(tex->get_bpp())
    {
      case 24:
	format = GL_RGB;
	break;
      case 32:
	format = GL_RGBA;
	break;
      default:
	std::cerr << "Error: Illegal texture bit depth: " << tex->get_bpp() <<
	  " (allowed: 24, 32).\n";
	delete tex;
	return false;
    }

    // Generate new texture and bind it.
    glEnable(GL_TEXTURE_2D);
    glGenTextures(1, &(this->texture_name));
    glBindTexture(GL_TEXTURE_2D, this->texture_name);

    glPixelStorei(GL_PACK_ALIGNMENT, 1);
    glPixelStorei(GL_PACK_ROW_LENGTH, tex->get_w());
    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
    glPixelStorei(GL_UNPACK_ROW_LENGTH, tex->get_w());

    // Set mipmapping.
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
	GL_LINEAR_MIPMAP_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,
	GL_LINEAR_MIPMAP_LINEAR);

    // Create texture from the data.
    gluBuild2DMipmaps(GL_TEXTURE_2D, 4, tex->get_w(), tex->get_h(), format,
	GL_UNSIGNED_BYTE, tex->get_cbuf());

    std::cout << "Loaded texture \"" << filename << "\" with D(" <<
      tex->get_w() << "x" << tex->get_h() << ")\n";

    // Disable textures again.
    glDisable(GL_TEXTURE_2D);

    return true;
  }

  // Report inability and fallback.
  std::cerr << "Error: Could not generate texture from \"" << filename <<
    "\"\n";
#endif

  // Will fallback to returning.
  return false;
}

//############################################################################
// Functions #################################################################
//############################################################################

/** Load a new texture from a file.
 * @param filename File to to load the texture from.
 * @return New texture object (in the canonical database) or NULL on error.
 */
Texture* texture_new(const char *filename)
{
#ifdef LIBFHI_OPENGL
  Texture *ret = Texture::canon_get(filename);

  if(!ret)
  {
    ret = Texture::canon_load(filename);
  }

  // If it got a legal texture name it is ok, return it.
  if(ret->get_texture_name())
  {
    return ret;
  }

  // Delete the reserved texture. Then fallback to returning NULL.
  Texture::canon_delete(filename);
#endif

  return NULL;
}

//############################################################################
// End #######################################################################
//############################################################################

}

