#include "libfhi_orientation.h"

using namespace libfhi;

//############################################################################
// Init ######################################################################
//############################################################################

/** Default constructor.
 */
Orientation::Orientation()
{
  null();
}

/** Default destructor.
 */
Orientation::~Orientation()
{
  // Do nothing.
}

/** Clear the orientation.
 */
void Orientation::null()
{
  om.identity();
  mv.null();
  attach.null();
  target = NULL;
  normalizer = 0;
  angles[0] = angles[1] = angles[2] = da[0] = da[1] = da[2] = 0;
  rotation = DEFAULT_ROTATION;
}

/** Make this orientation a copy of target orientation.
 * @param src [in] Orientation object.
 */
void Orientation::copy(Orientation *src)
{
  om = src->om;
  mv = src->mv;
  attach = src->attach;
  target = src->target;
  normalizer = src->normalizer;
  angles[0] = src->angles[0];
  angles[1] = src->angles[1];
  angles[2] = src->angles[2];
  da[0] = src->da[0];
  da[1] = src->da[1];
  da[2] = src->da[2];
  rotation = src->rotation;
}

//############################################################################
// Luokan funktiot ###########################################################
//############################################################################

/** Decrement the normalization variable and perform matrix normalization if
 * neccessary.
 */
void Orientation::normalize_check()
{
  if(--normalizer <= 0)
  {
    this->om.normalize();
    this->normalizer = NORMALIZE_DELAY;
  }
}

/** A method that is calculated each frame to determine new orientation.
 * Works 'in-place' for the orientation data.
 */
void Orientation::tick()
{
  Matrix44 rm, tm;
  Vector3 dv;

  // Increment angles on the difference set.
  this->angles[0] += this->da[0];
  this->angles[1] += this->da[1];
  this->angles[2] += this->da[2];

  // Increment position on the difference set.
  this->pv += this->mv;

  // Perform approperiate rotation
  switch(rotation)
  {
    case ROTATION_NO:
      this->om.translation(this->pv);
    case ROTATION_QUAKE:
      this->om.rotation_zxy_translation(this->angles, this->pv);
      break;
    case ROTATION_ABSORB:
      this->om.rotation_zyx_translation(this->angles, this->pv);
      break;
    case ROTATION_CUMULATIVE:
      rm.rotation_zxy(this->angles[0], this->angles[1], this->angles[2]);
      this->om = this->om * rm;
      this->normalize_check();
      break;
    case ROTATION_RELATIVE:
      rm.rotation_rel(this->om, this->angles[0], this->angles[1],
	  this->angles[2]);
      this->om = this->om * rm;
      this->normalize_check();
      break;
    case ROTATION_TARGET:
      this->om.lookat(this->target->get_pv(), this->get_pv(),
	  Vector3(0.0f, 1.0f, 0.0f));
      break;
    case ROTATION_ATTACH:
      rm.rotation_zxy(this->angles);
      tm.translation(this->target->get_pv() +
	  rm.get_rt() * attach.xf +
	  rm.get_up() * attach.yf +
	  rm.get_fw() * attach.zf);
      this->om = tm * rm;
      break;
    default:
      break;
  }
}

//############################################################################
// Debug #####################################################################
//############################################################################

#ifdef LIBFHI_DEBUG

/** Print this orientation.
 * @param s [in] The sream to print to.
 */
std::ostream& Orientation::print(std::ostream &s) const
{
  return s << "Attach: " << attach << "\nNormalizer: " << normalizer <<
    "\nRotation: " << rotation << "\nTarget: " << target << "\nAngles: <" <<
    angles[0] << ", " << angles[1] << ", " << angles[2] << ">\nPv: " <<
    this->pv << "\nOm:\n" << om;
}

#endif

//############################################################################
// Loppu #####################################################################
//############################################################################

