#include "libfhi_mesh.h"
#include "libfhi_premodel.h"

#include <iostream>
#include HASHMAP_INCLUDE

namespace libfhi {

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor. */
Mesh::Mesh()
  : Attributable(DEFAULT_ATTR)
#ifdef LIBFHI_OPENGL
    ,display_list(0)
#endif
{
  null();
}

/** Copy constructor.
 * @param src Source.
 */
Mesh::Mesh(Mesh *src)
  : Attributable(src->get_attr())
#ifdef LIBFHI_OPENGL
    ,display_list(0)
#endif
{
  // First of all, null everything.
  null();

  // Copy texture info BEFORE reserving vertex arrays.
  this->set_texture(src->get_texture());

  // Now, reserve vertex arrays.
  this->reserve_vertex(src->num_vertex);

  // Copy vertex data.
  memcpy(this->color, src->color, this->num_vertex * sizeof(Color4));
  memcpy(this->vertex, src->vertex, this->num_vertex * sizeof(Vector3));
  memcpy(this->vnormal, src->vnormal, this->num_vertex * sizeof(Vector3));

  // Copy texcoord data if there is a point.
  if(this->has_texture())
  {
    memcpy(this->texcoord, src->texcoord, this->num_vertex * sizeof(Vector2));
  }

  // Copy all skins.
  for(int i = 0; (i < MESH_ELEMENT_COUNT); ++i)
  {
    if(src->elem[i] != NULL)
    {
      int cnt = src->elem[i][0];
      this->elem[i] = new MESH_TYPEDEF_INDEX[cnt];
      memcpy(this->elem[i], src->elem[i],
	  sizeof(MESH_TYPEDEF_INDEX) * (cnt + 1));
    }
  }
}

/** Default destructor. */
Mesh::~Mesh()
{
  // Empty the attributes.
  this->set_attr(DEFAULT_ATTR);

  // Delete all skins.
  for(int i = 0; (i < MESH_ELEMENT_COUNT); ++i)
  {
    delete[] this->elem[i];
  }

  delete[] color;
  delete[] vertex;
  delete[] vnormal;
  delete[] texcoord;

#ifdef LIBFHI_OPENGL
  if(this->display_list)
  {
    glDeleteLists(this->display_list, 1);
  }
#endif
}

/** Empty this mesh, to be ran in constructors and nowhere else.
 */
void Mesh::null()
{
  this->color = NULL;
  this->vertex = this->vnormal = NULL;
  this->texcoord = NULL;
  this->skel = NULL;
  this->texture = NULL;
  this->num_vertex = 0;

  for(int i = 0; (i < MESH_ELEMENT_COUNT); ++i)
  {
    this->elem[i] = NULL;
  }
}

//############################################################################
// Utility ###################################################################
//############################################################################

/** Get the maximum distance from the center of this to a vertex.
 * @return maximum distance as float.
 */
float Mesh::get_maxdist() const
{
  float ret = 0.0f;

  for(size_t i = 0; (i < this->num_vertex); ++i)
  {
    ret = stdmax(this->vertex[i].length(), ret);
  }

  return ret;
}

/** Get the maximum value in an axis.
 * @param axis Integer describing the axis [0, 2].
 * @return Maximum coordinate value in axis.
 */
float Mesh::get_maximum_in_axis(int axis) const
{
  float ret = -FLT_MAX;

  for(size_t i = 0; (i < this->num_vertex); ++i)
  {
    ret = stdmax(ret, this->vertex[i].f[axis]);
  }

  return ret;
}

/** Get the minimum value in an axis.
 * @param axis Integer describing the axis [0, 2].
 * @return Maximum coordinate value in axis.
 */
float Mesh::get_minimum_in_axis(int axis) const
{
  float ret = FLT_MAX;

  for(size_t i = 0; (i < this->num_vertex); ++i)
  {
    ret = stdmin(ret, this->vertex[i].f[axis]);
  }

  return ret;
}

/** Set new reservation size.
 * @param num The new element counts in an array.
 */
void Mesh::reserve_face(MESH_TYPEDEF_INDEX num[MESH_ELEMENT_COUNT])
{
  for(int i = 0; (i < static_cast<int>(MESH_ELEMENT_COUNT)); ++i)
  {
    zdel_array(this->elem[i]);

    if(num[i] > 0)
    {
      this->elem[i] = new MESH_TYPEDEF_INDEX[num[i] + 1];
      *(this->elem[i]) = num[i];
    }
  }
}

#ifdef LIBFHI_OPENGL
/** Reserve new display list and save the index here.
 */
GLuint Mesh::reserve_display_list()
{
  this->display_list = glGenLists(1);
  return this->display_list;
}
#endif

/** Reserve this to be of a size multiple of given mesh, then fill all the
 * face slots assuming the first mesh indexes are endlessly repeated.
 * @param src Source.
 * @param cnt Multiplicity count.
 */
void Mesh::reserve_multiple(Mesh *src, size_t cnt)
{
  MESH_TYPEDEF_INDEX num[MESH_ELEMENT_COUNT];
  size_t svert = src->get_num_vertex();

  // Again, copy texture first.
  this->set_texture(src->get_texture());

  // Reserve new vertices now that we know if we need texcoords.
  this->reserve_vertex(svert * cnt);

  // Get face counts.
  for(int i = 0; (i < static_cast<int>(MESH_ELEMENT_COUNT)); ++i)
  {
    MESH_TYPEDEF_INDEX *el = src->get_array_elem(static_cast<ElementType>(i));

    if(el)
    {
      num[i] = *el * cnt;
    }
    else
    {
      num[i] = 0;
    }
  }

  // Reserve new faces.
  this->reserve_face(num);

  // Fill all mesh data with clones.
  for(size_t j = 0; (j < cnt); ++j)
  {
    for(size_t i = 0; (i < svert); ++i)
    {
      this->color[j * svert + i] = src->color[i];
      this->vertex[j * svert + i] = src->vertex[i];
      this->vnormal[j * svert + i] = src->vnormal[i];

      // Only copy if there is a point.
      if(this->has_texture())
      {
	this->texcoord[j * svert + i] = src->texcoord[i];
      }
    }
  }

  // Fill all face data with clones plus the index.
  for(unsigned k = 0; (k < static_cast<int>(MESH_ELEMENT_COUNT)); ++k)
  {
    ElementType etype = static_cast<ElementType>(k);
    MESH_TYPEDEF_INDEX *sel = src->get_array_elem(etype),
	  	       *del = this->get_array_elem(etype);

    if(sel)
    {
      for(size_t j = 0; (j < cnt); ++j)
      {
  	for(MESH_TYPEDEF_INDEX i = 0; (i < *sel); ++i)
	{
	  del[j * (*sel) + i + 1] =
	    sel[i + 1] + static_cast<MESH_TYPEDEF_INDEX>(j * svert);
	}
      }
    }
  }
}

/** Set new reservation size.
 * @param op The new vertex count.
 */
void Mesh::reserve_vertex(size_t op)
{
  // Delete possible previous stuff
  zdel_array(this->color);
  zdel_array(this->vertex);
  zdel_array(this->vnormal);
  zdel_array(this->texcoord);

  // Set count and exit if less than zero.
  this->num_vertex = op;
  if(this->num_vertex <= 0)
  {
    this->num_vertex = 0;
    return;
  }

  // Reserve.
  this->color = new Color4[num_vertex];
  this->vertex = new Vector3[num_vertex];
  this->vnormal = new Vector3[num_vertex];

  // Reserve texture coordinates if there is a point.
  if(this->has_texture())
  {
    this->texcoord = new Vector2[num_vertex];
  }
}

/** Setting point size of a mesh has multiple functions. 
 * 1) It turns on the point sprite attribute, that tells this mesh has to be
 * rendered by using point sprites (software will still just spawn pixels).
 * 2) It will delete all face arrays not of type point (they should be empty
 * anyway, but it will be ensured). Point array is kept unmodified (someone
 * might want to issue same vertex multiple times, etc.).
 * 3) It will delete any texture coordinates potentially stored in the mesh.
 * When using point sprites, texture coordinates are generated on the gly.
 * 4) The specified point size is stored to the mesh.
 * Note: only supports linear distance attenuation of 1 until I decide
 * implement other methods.
 * @param op New point size.
 */
void Mesh::set_point_size(float op)
{
  // Set point size and point sprite attribute.
  this->point_size = op;
  this->attr_or(ATTR_POINT_SPRITES);

  // Destroy all other face types. 
  for(int i = static_cast<int>(MESH_POINTS + 1);
      (i < static_cast<int>(MESH_ELEMENT_COUNT));
      ++i)
  {
    zdel_array(this->elem[i]);
  }

  // Destroy possible texture coordinate array.
  zdel_array(this->texcoord);
}

//############################################################################
// Debug #####################################################################
//############################################################################

#ifdef LIBFHI_DEBUG

std::ostream& Mesh::print(std::ostream &s) const
{
  s << "Vertices: " << static_cast<unsigned>(num_vertex);

  for(size_t i = 0; (i < this->num_vertex); i++)
  {
    s << "\n" << static_cast<unsigned>(i) << ": C: " << color[i] << " V: "
      << vertex[i] << " N: " << vnormal[i];

    // Only print if there is a point.
    if(this->has_texture())
    {
      s <<  " T: " << texcoord[i];
    }
  }

  for(int i = 0; (i < MESH_ELEMENT_COUNT); ++i)
  {
    MESH_TYPEDEF_INDEX *ptr = elem[i];

    if(ptr == NULL)
    {
      continue;
    }

    s << "\nElement type ";

    switch(i)
    {
      case MESH_POINTS:
	s << "POINTS";
	break;
      case MESH_FLAT_LINES:
	s << "FLAT_";
      case MESH_LINES:
	s << "LINES";
	break;
      case MESH_FLAT_TRIANGLES:
	s << "FLAT_";
      case MESH_TRIANGLES:
	s << "TRIANGLES";
	break;
      case MESH_FLAT_QUADS:
	s << "FLAT_";
      case MESH_QUADS:
	s << "QUADS";
	break;
      default:
	s << "ILLEGAL_TYPE";
	break;
    }   
   
    s << " (" << *ptr << "):";

    for(int j = *ptr++; (j--); ++ptr)
    {
      s << " " << *ptr;
    }
  }

  return s;
}

#endif

//############################################################################
// Loppu #####################################################################
//############################################################################

}

