#include "libfhi_matvec.h"

using namespace libfhi;

//############################################################################
// Matrix44 OpenGL matrices ##################################################
//############################################################################

/** Normalization of a matrix to be used on degradation.
 */
void Matrix44::normalize()
{
  // FIXME: implement
}

/** Create a lookat matrix identical to the OpenGl gluLookaAt.
 * @param pos Eye position.
 * @param look Eye target position.
 * @param up Up vector.
 */
void Matrix44::lookat(const Vector3&, const Vector3&,
    const Vector3&)
{
  // FIXME: implement
}

/* The rotation matrices call for some explanation. Basically, the rotation
 * matrices consist of three distinct rotation matrices applied over the
 * euclidian coordinate axises.
 *
 * X:
 *   1     0     0     0
 *   0    ca   -sa     0
 *   0    sa    ca     0
 *   0     0     0     1
 *
 * Y:
 *  ca     0    sa     0
 *   0     1     0     0
 * -sa     0    ca     0
 *   0     0     0     1
 *
 * Z:
 *  ca   -sa     0     0
 *  sa    ca     0     0
 *   0     0     1     0
 *   0     0     0     1
 *
 * When precalculated into a simultaneous three-axis transformation, they
 * produce the following matrix.
 *
 * Absorb:
 * cy*cz - sx*sy*sz             -cx*sz   sy*cz + sx*cy*sz                  0
 * cy*sz + sx*sy*cz              cx*cz   sy*sz - sx*cy*cz                  0
 *           -cx*sy                 sx              cx*cy                  0
 *                0                  0                  0                  1
 *
 * This schema is the rotation mostly used in first-person shooters (order of
 * multiplification is z * x * y), so we call it quake-rotation:
 * 
 * Quake:
 * cy*cz + sx*sy*sz   sx*sy*cz - sz*cy              sy*cx                  0
 *            sz*cx              cx*cz                -sx                  0
 * sx*sz*cy - sy*cz   sy*sz + sx*cy*cz              cx*cy                  0
 *                0                  0                  0                  1
 */

/** Calculates the sin and cos values into given variables.
*/
inline void count_csxyz(uint16_t x, uint16_t y, uint16_t z, float &cx,
  float &sx, float &cy, float &sy, float &cz, float &sz)
{
  float fx, fy, fz;
  fx = uint2pi(x);
  fy = uint2pi(y);
  fz = uint2pi(z);
  cx = cos(fx);
  sx = sin(fx);
  cy = cos(fy);
  sy = sin(fy);
  cz = cos(fz);
  sz = sin(fz);
}

/** Relative rotation.
* Same as normal rotation, but performs the rotation in accordance to the
* axises of the current matrix. In other words, rotates the object in it's
* own coordinate system.
* @param op Matrix to base rotation to.
* @param x X rotation.
* @param y Y rotation.
* @param z Z rotation.
*/
void Matrix44::rotation_rel(const Matrix44&, uint16_t x, uint16_t y,
  uint16_t z)
{
  float cx, sx, cy, sy, cz, sz;

  count_csxyz(x, y, z, cx, sx, cy, sy, cz, sz);

  // FIXME: implement
}

/** Quake rotation of z * x * y.
* Restricted version only touches the 3x3 part.
* @param x [in] Rotation over x axis.
* @param y [in] Rotation over y axis.
* @param z [in] Rotation over z axis.
*/
void Matrix44::rotation_zxy33(uint16_t x, uint16_t y, uint16_t z)
{
  float cx, sx, cy, sy, cz, sz;

  count_csxyz(x, y, z, cx, sx, cy, sy, cz, sz);

  // The transformation data
  this->f[0] = cy * cz + sx * sy * sz;
  this->f[4] = sx * sy * cz - sz * cy;
  this->f[8] = sy * cx;
  this->f[1] = sz * cx;
  this->f[5] = cx * cz;
  this->f[9] = -sx;
  this->f[2] = sx * sz * cy - sy * cz;
  this->f[6] = sy * sz + sx * cy * cz;
  this->f[10] = cx * cy;
}

/** Absorb rotation of z * y * x.
* Restricted version only touches the 3x3 part.
* @param x [in] Rotation over x axis.
* @param y [in] Rotation over y axis.
* @param z [in] Rotation over z axis.
*/
void Matrix44::rotation_zyx33(uint16_t x, uint16_t y, uint16_t z)
{
  float cx, sx, cy, sy, cz, sz;

  count_csxyz(x, y, z, cx, sx, cy, sy, cz, sz);

  // The transformation data
  this->f[0] = cz * cy;
  this->f[4] = cz * sy * sx - sz * cx;
  this->f[8] = sz * sx + cz * sy * cx;
  this->f[1] = sz * cy;
  this->f[5] = cz * cx + sz * sy * sx;
  this->f[9] = sz * sy * cx - cz * sx;
  this->f[2] = -sy;
  this->f[6] = cy * sx;
  this->f[10] = cy * cx;
}

/** Quake rotation of z * x * y.
* Utilizes the restricted version.
* @param x [in] Rotation over x axis.
* @param y [in] Rotation over y axis.
* @param z [in] Rotation over z axis.
*/
void Matrix44::rotation_zxy(uint16_t x, uint16_t y, uint16_t z)
{
  // Call base method
  this->rotation_zxy33(x, y, z);

  // Other values
  this->f[3] = this->f[7] = this->f[11] = this->f[12] = this->f[13] =
    this->f[14] = 0.0f;
  this->f[15] = 1.0f;
}

/** Quake rotation of z * x * y.
* @param ang [in] Pointer to array of rotation angles.
*/
void Matrix44::rotation_zxy(const uint16_t *ang)
{
  this->rotation_zxy(ang[0], ang[1], ang[2]);
}

/** Standard translation matrix.
* Only sets the 1x3 part.
* @param x [in] Translation over x axis.
* @param y [in] Translation over y axis.
* @param z [in] Translation over z axis.
*/
void Matrix44::translation_13(float x, float y, float z)
{
  this->f[12] = x;
  this->f[13] = y;
  this->f[14] = z;
}

/** Standard translation matrix.
* Utilizes the restricted version.
* @param x [in] Translation over x axis.
* @param y [in] Translation over y axis.
* @param z [in] Translation over z axis.
*/
void Matrix44::translation(float x, float y, float z)
{
  // Call base method.
  this->translation_13(x, y, z);

  // Set remaining values.
  this->f[0] = this->f[5] = this->f[10] = this->f[15] = 1.0f;
  this->f[1] = this->f[2] = this->f[3] = this->f[4] = this->f[6] =
    this->f[7] = this->f[8] = this->f[9] = this->f[11] = 0.0;
}

/** Standard translation matrix.
* @param tv [in] Translation vector.
*/
void Matrix44::translation(const Vector3& tv)
{
  this->translation(tv.xf, tv.yf, tv.zf);
}

//############################################################################
// Combinatory transformation matrices #######################################
//############################################################################

/** Rotation (over own axis) and translation in one operation.
* @param ang [in] Pointer to array of rotation angles.
* @param tv [in] Translation vector.
*/
void Matrix44::rotation_zxy_translation(const uint16_t *ang,
  const Vector3& tv)
{
  // Set 3x3 rotation.
  this->rotation_zxy33(ang[0], ang[1], ang[2]);

  // Set 1x3 translation.
  this->translation_13(tv.xf, tv.yf, tv.zf);

  // Remaining values.
  this->f[3] = this->f[7] = this->f[11] = 0.0f;
  this->f[15] = 1.0f;
}

/** Rotation (over own axis) and translation in one operation.
* @param ang [in] Pointer to array of rotation angles.
* @param tv [in] Translation vector.
*/
void Matrix44::rotation_zyx_translation(const uint16_t *ang,
  const Vector3& tv)
{
  // Set 3x3 rotation.
  this->rotation_zyx33(ang[0], ang[1], ang[2]);

  // Set 1x3 translation.
  this->translation_13(tv.xf, tv.yf, tv.zf);

  // Remaining values.
  this->f[3] = this->f[7] = this->f[11] = 0.0f;
  this->f[15] = 1.0f;
}

//############################################################################
// Miscellaneous transformation ##############################################
//############################################################################

/** This matrix is a transpose of the source.
* @param op [in] The matrix to transponate.
*/
void Matrix44::transpose(const Matrix44& op)
{
this->f[0] = op.f[0];
this->f[1] = op.f[4];
this->f[2] = op.f[8];
this->f[3] = op.f[12];
this->f[4] = op.f[1];
this->f[5] = op.f[5];
this->f[6] = op.f[9];
this->f[7] = op.f[13];
this->f[8] = op.f[2];
this->f[9] = op.f[6];
this->f[10] = op.f[10];
this->f[11] = op.f[14];
this->f[12] = op.f[3];
this->f[13] = op.f[7];
this->f[14] = op.f[11];
this->f[15] = op.f[15];
}

/** Reads a normal 3x3 - 1x3 -transformation matrix and makes a camera view
* matrix out of it.
* @param op [in] The matrix to modify.
*/
void Matrix44::view_matrix(const Matrix44& op)
{
// First, transpose the 3x3 part.
this->f[0] = op.f[0];
this->f[1] = op.f[4];
this->f[2] = op.f[8];
this->f[4] = op.f[1];
this->f[5] = op.f[5];
this->f[6] = op.f[9];
this->f[8] = op.f[2];
this->f[9] = op.f[6];
this->f[10] = op.f[10];

// Then transform the inverse of 1x3 part.
this->f[12] = -this->f[0] * op.f[12] - this->f[4] * op.f[13] -
  this->f[8] * op.f[14];
this->f[13] = -this->f[1] * op.f[12] - this->f[5] * op.f[13] -
  this->f[9] * op.f[14];
this->f[14] = -this->f[2] * op.f[12] - this->f[6] * op.f[13] -
  this->f[10] * op.f[14];

// Default the rest.
this->f[3] = this->f[7] = this->f[11] = 0.0f;
this->f[15] = 1.0f;
}

//############################################################################
// Debug #####################################################################
//############################################################################

#ifdef LIBFHI_DEBUG

/** Print this into a given stream.
 * @param s Stream to print to.
 */
std::ostream& Vector2::print(std::ostream &s) const
{
  return s << "<" << this->xf << ", " << this->yf << ">";
}

/** Print this into a given stream.
 * @param s Stream to print to.
 */
std::ostream& Vector3::print(std::ostream &s) const
{
  return s << "<" << this->xf << ", " << this->yf << ", " << this->zf << ">";
}

/** Print this into a given stream.
 * @param s Stream to print to.
 */
std::ostream& Matrix44::print(std::ostream &s) const
{
return s << "[" << this->f[0] << ", " << this->f[4] << ", " <<
  this->f[8] << ", " << this->f[12] << "]\n[" << this->f[1] << ", " <<
  this->f[5] << ", " << this->f[9] << ", " << this->f[13] << "]\n[" <<
  this->f[2] << ", " << this->f[6] << ", " << this->f[10] << ", " <<
  this->f[14] << "]\n[" << this->f[3] << ", " << this->f[7] << ", " <<
  this->f[11] << ", " << this->f[15] << "]";
}

#endif

//############################################################################
// Loppu #####################################################################
//############################################################################

