#ifndef libfhi_premodel_include
#define libfhi_premodel_include

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi_canonicaldb.h"
#include "libfhi_component.h"

#include <list>
#include <queue>
#include HASHMAP_INCLUDE

namespace libfhi {

//############################################################################
// Definet ###################################################################
//############################################################################

// Forward declaration.
class Mesh;

//############################################################################
// Luokka ####################################################################
//############################################################################

/** Pre-version of a model. Used to do assembling and precalculating, then
 * deleted in favor of more memory-friendly (but fixed) PostModel.
 */
class PreModel :
  public Attributable,
  public CanonicalDB<PreModel>
{
  public:
    /** Edge map type. */
    typedef HASHMULTIMAP<size_t, ComponentEdge*> EDGEMAP;

    /** Wireframe flag for compile. */
    static const uint8_t ATTR_WIREFRAME = 0x01;

    /** Disallow flat shading in compile. */
    static const uint8_t ATTR_DENYFLAT = 0x02;

    /** If this is on, forget inputted faces from final product. */
    static const uint8_t ATTR_FORGET = 0x04;

    /** Defaults. */
    static const uint8_t DEFAULT_ATTR = 0x00;

  private:
    void cleanup();

  protected:
    /** Vertex array. */
    std::vector<ComponentVertex*> vertex_array;

    /** Face array. */
    std::vector<ComponentFace*> face_array;

    /** Vertex queue. */
    std::priority_queue<size_t> vertex_queue;

    /** Face queue. */
    std::priority_queue<size_t> face_queue;

    /** Edges are stored in a multiassociative hashmap for fast access. */
    EDGEMAP edge_map;

    /** Forget list, do not include these faces in the compiled mesh. */
    HASHMAP<size_t, bool> forget;

    /** String containing the texture name. */
    std::string texture_file;

#ifdef LIBFHI_DEBUG
  public:
    DEBUG_PRINT_PROTOTYPE(PreModel)
#endif

  public:
    PreModel();
    PreModel(const char*);
    ~PreModel();

    ComponentVertex* add_vertex(const Vector3&, const Color4&,
	const Vector2&);
    ComponentEdge* add_edge(size_t, size_t);
    ComponentFace* add_face(size_t);
    ComponentFace* add_face(size_t, size_t);
    ComponentFace* add_face(size_t, size_t, size_t);
    ComponentFace* add_face(size_t, size_t, size_t, size_t);
    Mesh* compile(const char*);
    EDGEMAP::iterator find_edge(size_t, size_t);
    void remove_edge(size_t, size_t);
    void remove_face(size_t);
    void remove_vertex(size_t);
    void scale_to(int, float);
    void strip(Mesh*);

    // Inline.
    inline ComponentEdge* get_edge(size_t, size_t);
    inline ComponentFace* get_face(size_t);
    inline ComponentVertex* get_vertex(size_t);
    inline bool has_wireframe() const;
    inline bool has_denyflat() const;
    inline bool has_forget() const;
    inline size_t get_num_edge() const;
    inline size_t get_num_face() const;
    inline size_t get_num_vertex() const;
    inline void set_texture_file(const std::string&);
    inline void set_texture_file(const char*);
};

//############################################################################
// Inline-metodit ############################################################
//############################################################################

/** Get an edge by two face indices.
 * @param idx1 Vertex index 1.
 * @param idx2 Vertex index 2.
 * @return Pointer to the edge or NULL if not found.
 */
ComponentEdge* PreModel::get_edge(size_t idx1, size_t idx2)
{
  EDGEMAP::iterator iter = this->find_edge(idx1, idx2);

  if(iter == edge_map.end())
  {
    return NULL;
  }
  return (*iter).second;
}

/** Get face by index.
 * @param Index of the face.
 * @return Pointer to the face or NULL if not found.
 */
ComponentFace* PreModel::get_face(size_t idx)
{
  if(idx < face_array.size())
  {
    return face_array[idx];
  }
  return NULL;
}

/** Get vertex by index.
 * @param Index of the vertex.
 * @return Pointer to the vertex or NULL if not found.
 */
ComponentVertex* PreModel::get_vertex(size_t idx)
{
  if(idx < vertex_array.size())
  {
    return vertex_array[idx];
  }
  return NULL;
}

/** Tell if this exports in wireframe.
 * @return True if yes, false if no.
 */
bool PreModel::has_wireframe() const
{
  return this->has_attr(ATTR_WIREFRAME);
}

/** Tell if flat shading is denied.
 * @return True if yes, false if no.
 */
bool PreModel::has_denyflat() const
{
  return this->has_attr(ATTR_DENYFLAT);
}

/** Tell if we are currently forgetting added faces.
 * @return True if yes, false if no.
 */
bool PreModel::has_forget() const
{
  return this->has_attr(ATTR_FORGET);
}

/** Get the number of edges.
 * @return The number as an integer.
 */
size_t PreModel::get_num_edge() const
{
  return edge_map.size();
}

/** Get the number of faces.
 * Note that this may or may not be the size of the face array.
 * @return The number as an integer.
 */
size_t PreModel::get_num_face() const
{
  return face_array.size() - face_queue.size();
}

/** Get the number of vertices.
 * Note that this may or may not be the size of the vertex array.
 * @return The number as an integer.
 */
size_t PreModel::get_num_vertex() const
{
  return vertex_array.size() - vertex_queue.size();
}

/** Set the texture file of this.
 * @param op New texture file.
 */
void PreModel::set_texture_file(const std::string &op)
{
  this->texture_file = op;
}

/** Set the texture file of this character string version.
 * @param op New texture file.
 */
void PreModel::set_texture_file(const char *op)
{
  // Should be overloaded nicely.
  this->texture_file = op;
}

//############################################################################
// Loppu #####################################################################
//############################################################################

}
#endif

