#ifndef libfhi_point_include
#define libfhi_point_include

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi_color.h"
#include "libfhi_matvec.h"

namespace libfhi {

//############################################################################
// Definet ###################################################################
//############################################################################

// Forward declaration
class Boundary;
class Surface;

//############################################################################
// Luokka ####################################################################
//############################################################################

/** Point class is never used in it's base form, and most of the functions
 * dealing with it are actually external. The reasoning for this is that the
 * points are used in clipping algorithms. Thay need function pointers that
 * will deal with an array of base class -type Points while polymorphing the
 * clipping algorithms in ways unable to master with mere inheritance.
 * 
 * Thus, it is easier to call an istantiated version of a template function
 * than to really use template classes.
 * 
 * There is probably an infinitely more beautiful solution, but it is unclear
 * if there is a noticeably more efficient solution.
 */
class Point
{
  friend class Surface;

  public:
    /** Inside clipping mask. */
    static const uint8_t INSIDE = 0x01;

    /** Outside minimum boundary clipping mask. */
    static const uint8_t OUTSIDE_MIN = 0x02;

    /** Outside maximum boundary clipping mask. */
    static const uint8_t OUTSIDE_MAX = 0x04;

    /** Outside either boundary clipping mask. */
    static const uint8_t OUTSIDE_EITHER = OUTSIDE_MIN | OUTSIDE_MAX;

  protected:
    /** Position of this point. */
    Vector3 pos;

    /** Color of this point. */
    Color4 col;

    /** Previous point in sequence. */
    Point *prev;

    /** Next point in sequence. */
    Point *next;

    /** Mask containing the clipping rules for this point. */
    uint8_t mask;

  public:
    // Mask-altering methods
    void mask_x(const Boundary*);
    void mask_y(const Boundary*);
    void mask_z(const Boundary*);
    void project(const Boundary*);

    // Inline methods
    inline const Color4& get_col() const;
    inline Color4& get_col();
    inline uint8_t get_mask() const;
    inline Point* get_next();
    inline const Vector3& get_pos() const;
    inline Vector3& get_pos();
    inline bool inside() const;
    inline bool outside() const;
    inline bool outside(uint8_t) const;
    inline void set_col(float, float, float);
    inline void set_col(float, float, float, float);
    inline void set_mask(uint8_t op);
    inline void set_pos(const Vector3& op);

    // Inline state machine set methods
    inline void set_nc(int, int);
    inline void set_nc(int, int, int, int, int, int);
    inline void set_nc(int, int, int);
    inline void set_nc(int, int, int, int, int, int, int);
    inline void set(float, float);
    inline void set(float, float, const Color4&);
    inline void set(const Vector3&);
    inline void set(const Vector3&, float, float, float);
    inline void set(const Vector3&, const Color4&);
    
    // Friend templates
    template <class Type> friend void point_clip_x(Point*, const Point*,
	const Point*, float);
    template <class Type> friend void point_clip_y(Point*, const Point*,
	const Point*, float);
    template <class Type> friend void point_clip_z(Point*, const Point*,
	const Point*, float);
};

/** Variant of the point for two-dimensional flat forms.
 */
class Point2F : public Point
{
#ifdef LIBFHI_DEBUG
  public:
    DEBUG_PRINT_PROTOTYPE(Point2F)
#endif

  public:
    void setvi();
};

/** Variant of the point for two-dimensional gouraud forms.
 */
class Point2G : public Point
{
#ifdef LIBFHI_DEBUG
  public:
    DEBUG_PRINT_PROTOTYPE(Point2G)
#endif

  public:
    void setvi();
};

/** Variant of the point for three-dimensional flat forms.
 */
class Point3F : public Point
{
#ifdef LIBFHI_DEBUG
  public:
    DEBUG_PRINT_PROTOTYPE(Point3F)
#endif

  public:
    void setvi();
};

/** Variant of the point for three-dimensional gouraud forms.
 */
class Point3G : public Point
{
#ifdef LIBFHI_DEBUG
  public:
    DEBUG_PRINT_PROTOTYPE(Point3G)
#endif

  public:
    void setvi();
};

//############################################################################
// Set functions (for state machine) #########################################
//############################################################################

/** These functions perform the initialization of Point structures when feeding
 * the points to the state machine in Surface class. They are purposefully
 * vague in the naming principle, since they are called on the approperiate
 * points of the state machine.
 *
 * The mechanism uses heavy function name overloading, correct version is only
 * determinef from the number of parameters. No unspecified parameters will
 * ever be set or copied, using these functions will leave the other
 * components or coordinates in unspecified values.
 *
 * Points can be specified by x/y coordinates or vector structures, which
 * account for 3 component vectors. Colors can only be specified by component
 * triplets or quadruplets to prevent accidental overspecification when
 * unneccessary.
 *
 * Integer versions are just for the deaw operations. When floating point
 * values are specified, clipping methods (in Surface class) must be ran, upon
 * completion they either convert the floating point values into fixed point
 * format or return nontrue, in which case nothing needs to be drawn.
 *
 * @param x X coordinate.
 * @param y Y coordinate.
 */
void Point::set_nc(int x, int y)
{
  this->pos.set_roundable(x, y);
}

/** Set non-clipped 2d gouraud with alpha.
 * @param x X coordinate.
 * @param y Y coordinate.
 * @param r R color component.
 * @param g G color component.
 * @param b B color component.
 * @param a A color component.
 */
void Point::set_nc(int x, int y, int r, int g, int b, int a)
{
  this->pos.set_roundable(x, y);
  this->col.set_roundable(r, g, b, a);
}

/** Set non-clipped 3d.
 * @param x X coordinate.
 * @param y Y coordinate.
 * @param z Z coordinate.
 */
void Point::set_nc(int x, int y, int z)
{
  this->pos.set_roundable(x, y, z);
}

/** Set non-clipped 3d gouraud with alpha.
 * @param p Position in fixed point.
 * @param r R color component.
 * @param g G color component.
 * @param b B color component.
 * @param a A color component.
 */
void Point::set_nc(int x, int y, int z, int r, int g, int b, int a)
{
  this->pos.set_roundable(x, y, z);
  this->col.set_roundable(r, g, b, a);
}

/** Set clipped 2d.
 * @param x X coordinate.
 * @param y Y coordinate.
 */
void Point::set(float x, float y)
{
  this->pos.xf = x;
  this->pos.yf = y;
}

/** Set clipped 2d gouraud with alpha.
 * @param x X coordinate.
 * @param y Y coordinate.
 * @param c Color in floating point.
 */
void Point::set(float x, float y, const Color4& c)
{
  this->pos.xf = x;
  this->pos.yf = y;
  this->col = c;
}

/** Set clipped 3d.
 * @param p Position in floating point.
 */
void Point::set(const Vector3& p)
{
  this->pos = p;
}

/** Set clipped 3d gouraud. This is an extra method used internally by the
 * software renderer (via Surface::point_add), it cannot be accessed from
 * outside.
 * @param p Position in floating point.
 * @param r R color component.
 * @param g G color component.
 * @param b B color component.
 */
void Point::set(const Vector3& p, float r, float g, float b)
{
  this->pos = p;
  this->col.set(r, g, b);
}

/** Set clipped 3d gouraud.
 * @param p Position in floating point.
 * @param c Color in floating point.
 */
void Point::set(const Vector3& p, const Color4& c)
{
  this->pos = p;
  this->col = c;
}

//############################################################################
// Misc ######################################################################
//############################################################################

/** Get the color of this point.
 * @return Color vector.
 */
Color4& Point::get_col()
{
  return this->col;
}

/** Get the color of this point, const version.
 * @return Color vector.
 */
const Color4& Point::get_col() const
{
  return this->col;
}

/** Get the mask of this point, const version.
 * @return Mask integer.
 */
uint8_t Point::get_mask() const
{
  return this->mask;
}

/** Get the next point in sequence.
 * @return Pointer to the next point.
 */
Point* Point::get_next()
{
  return this->next;
}

/** Get the position of this point.
 * @return Position vector.
 */
Vector3& Point::get_pos()
{
  return this->pos;
}

/** Get the position of this point, const version.
 * @return Position vector.
 */
const Vector3& Point::get_pos() const
{
  return this->pos;
}

/** Tell if inside the boundary.
 * @return True if inside, false if outside.
 */
bool Point::inside() const
{
  return (mask & INSIDE);
}

/** Tell if outside the boundary.
 * @return True if outside, false if inside.
 */
bool Point::outside() const
{
  return ((mask & OUTSIDE_EITHER) != 0);
}

/** Tell if outside a certain boundary.
 * If INSIDE is passed as parameter, will return true erraneously.
 * @return True if yes, false if no.
 */
bool Point::outside(uint8_t op) const
{
  return ((mask & op) != 0);
}

/** Set the color of this point.
 * @param r R component.
 * @param g G component.
 * @param b B component.
 */
void Point::set_col(float r, float g, float b)
{
  this->col.set(r, g, b);
}

/** Set the color of this point.
 * @param r R component.
 * @param g G component.
 * @param b B component.
 * @param a A component.
 */
void Point::set_col(float r, float g, float b, float a)
{
  this->col.set(r, g, b, a);
}

/** Set the mask of this point.
 * @param op New mask.
 */
void Point::set_mask(uint8_t op)
{
  this->mask = op;
}

/** Set the position of this point.
 * @param op New position.
 */
void Point::set_pos(const Vector3& op)
{
  this->pos = op;
}

//############################################################################
// Loppu #####################################################################
//############################################################################

}
#endif

