#ifndef libfhi_color_include
#define libfhi_color_include

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi_fixed.h"

namespace libfhi {

//############################################################################
// Luokka ####################################################################
//############################################################################

/** Color represents one 4 component color as with OpenGL, red, green, blue and
 * alpha. Alpha is mostly forgotten in software mode, but exists for hardware
 * rendering.
 *
 * To circumvent users accidentally using ineffective operations that take the
 * a component into account when not neccessary, all operations in Color are
 * named with 3 or 4 suffix, which denotes the number of components included
 * into the operation.
 */
class Color4
{
  public:
    union
    {
      // Array representation.
      float f[4];

      struct
      {
	// Coordinates
	union
	{
	  float rf;
	  Fixed ri;
	};
	union
	{
	  float gf;
	  Fixed gi;
	};
	union
	{
	  float bf;
	  Fixed bi;
	};
	union
	{
	  float af;
	  Fixed ai;
	};
      };
    };

  public:
#ifdef LIBFHI_DEBUG
    DEBUG_PRINT_PROTOTYPE(Color4)
#endif

  public:
    // Constructors.
    inline Color4();
    inline Color4(float, float, float, float);
    inline Color4(int, int, int, int);

    // Methods.
    inline void average(const Color4&, const Color4&, const Color4&,
	const Color4&);
    inline void clamp4(float);
    inline void clamp3(float);
    inline void clip4(const Color4&, const Color4&, float);
    inline void clip3(const Color4&, const Color4&, float);
    inline void copy4(const Color4&);
    inline void copy3(const Color4&);
    inline float* get_fv();
    inline const float* get_fv() const;
    inline bool is_null3() const;
    inline void null4();
    inline void null3();
    inline void round4();
    inline void round3();
    inline void set(float, float, float);
    inline void set(float, float, float, float);
    inline void set(int, int, int);
    inline void set(int, int, int, int);
    inline void set_roundable(int, int, int);
    inline void set_roundable(int, int, int, int);

    // Operators.
    inline void operator=(const Color4&);
    inline bool operator==(const Color4&) const;
    inline bool operator!=(const Color4&) const;
};

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor.
 */
Color4::Color4()
{
  // Do nothing.
}

/** Initializing constructor. There is no 3 parameter constructor to prevent
 * users from accidentally initializing to unknown values.
 * @param r R component.
 * @param g G component.
 * @param b B component.
 * @param a A component.
 */
Color4::Color4(float r, float g, float b, float a)
{
  this->set(r, g, b, a);
}

/** Initializing constructor. There is no 3 parameter constructor to prevent
 * users from accidentally initializing to unknown values.
 * @param r R component.
 * @param g G component.
 * @param b B component.
 * @param a A component.
 */
Color4::Color4(int r, int g, int b, int a)
{
  this->set(r, g, b, a);
}

//############################################################################
// Get / set #################################################################
//############################################################################

/** Copy of source. 4 component version.
 * @param op Source color.
 */
void Color4::copy4(const Color4& op)
{
  this->ri = op.ri;
  this->gi = op.gi;
  this->bi = op.bi;
  this->ai = op.ai;
}

/** Copy of source. 3 component version.
 * @param op Source color.
 */
void Color4::copy3(const Color4& op)
{
  this->ri = op.ri;
  this->gi = op.gi;
  this->bi = op.bi;
}

/** Get color as a float vector.
 * @return Floating point array representation.
 */
float* Color4::get_fv()
{
  return this->f;
}

/** Get color as a float vector. Const version.
 * @return Floating point array representation.
 */
const float* Color4::get_fv() const
{
  return this->f;
}

/** Empties this color.
 */
void Color4::null4()
{
  this->ri.set_null();
  this->gi.set_null();
  this->bi.set_null(); 
  this->ai.set_null(); 
}

/** Empties this color. 3 component version.
 */
void Color4::null3()
{
  this->ri.set_null();
  this->gi.set_null();
  this->bi.set_null(); 
}

/** Set the values of this.
 * @param r R component.
 * @param g G component.
 * @param b B component.
 */
void Color4::set(float r, float g, float b)
{
  this->rf = r;
  this->gf = g;
  this->bf = b;
}

/** Set the values of this.
 * @param r R component.
 * @param g G component.
 * @param b B component.
 * @param a A component.
 */
void Color4::set(float r, float g, float b, float a)
{
  this->rf = r;
  this->gf = g;
  this->bf = b;
  this->af = a;
}

/** Set the values of this.
 * @param r R component.
 * @param g G component.
 * @param b B component.
 */
void Color4::set(int r, int g, int b)
{
  this->ri = Fixed::from(r);
  this->gi = Fixed::from(g);
  this->bi = Fixed::from(b);
}

/** Set the values of this.
 * @param r R component.
 * @param g G component.
 * @param b B component.
 * @param a A component.
 */
void Color4::set(int r, int g, int b, int a)
{
  this->ri = Fixed::from(r);
  this->gi = Fixed::from(g);
  this->bi = Fixed::from(b);
  this->ai = Fixed::from(a);
}

//############################################################################
// Testing ###################################################################
//############################################################################

/** Tell if this color equals zero. Note, this only touches the three first
 * components, alpha channel is irrelevant.
 * @return True if yes, false if no.
 */
bool Color4::is_null3() const
{
  return (fabs(this->rf) + fabs(this->gf) + fabs(this->bf) == 0.0f);
}

//############################################################################
// Math ######################################################################
//############################################################################

/** Calculate an average.
 * @param op1 Color 1.
 * @param op2 Color 2.
 * @param op3 Color 3.
 * @param op4 Color 4.
 */
void Color4::average(const Color4& op1, const Color4& op2, const Color4& op3,
    const Color4& op4)
{
  this->rf = (op1.rf + op2.rf + op3.rf + op4.rf) / 4.0f;
  this->gf = (op1.gf + op2.gf + op3.gf + op4.gf) / 4.0f;
  this->bf = (op1.bf + op2.bf + op3.bf + op4.bf) / 4.0f;
  this->af = (op1.af + op2.af + op3.af + op4.af) / 4.0f;
}

//############################################################################
// Clipping ##################################################################
//############################################################################

/** Colors do not have clip value calculation, all clippings are done with an
 * external percent.
 * @param op1 First color.
 * @param op2 Second color.
 * @param percent Value to clip (starting from 0.0f op1 to 1.0f op2).
 */
void Color4::clip4(const Color4& op1, const Color4& op2, float percent)
{
  float dr = op2.rf - op1.rf,
	dg = op2.gf - op1.gf,
	db = op2.bf - op1.bf,
	da = op2.af - op1.af;

  this->rf = percent * dr + op1.rf;
  this->gf = percent * dg + op1.gf;
  this->bf = percent * db + op1.bf;
  this->af = percent * da + op1.af;
}

/** Works like clip, but only clips the first 3 (r,g,b) colors.
 * @param op1 First color.
 * @param op2 Second color.
 * @param percent Value to clip (starting from 0.0f op1 to 1.0f op2).
 */
void Color4::clip3(const Color4& op1, const Color4& op2, float percent)
{
  float dr = op2.rf - op1.rf,
	dg = op2.gf - op1.gf,
	db = op2.bf - op1.bf;

  this->rf = percent * dr + op1.rf;
  this->gf = percent * dg + op1.gf;
  this->bf = percent * db + op1.bf;
}

//############################################################################
// Operators #################################################################
//############################################################################

/** Assignment operator.
 * @param rhs Right hand side operand.
 */
void Color4::operator=(const Color4& rhs)
{
  this->ri = rhs.ri;
  this->gi = rhs.gi;
  this->bi = rhs.bi;
  this->ai = rhs.ai;
}

/** Equality operator.
 * @param rhs Right hand side operand.
 */
bool Color4::operator==(const Color4& rhs) const
{
  return ((this->ri == rhs.ri) && (this->gi == rhs.gi) &&
      (this->bi == rhs.bi) && (this->ai == rhs.ai));
}

/** Inequality operator.
 * @param rhs Right hand side operand.
 */
bool Color4::operator!=(const Color4& rhs) const
{
  return ((this->ri != rhs.ri) || (this->gi != rhs.gi) ||
      (this->bi != rhs.bi) || (this->ai != rhs.ai));
}

//############################################################################
// Miscellaneous #############################################################
//############################################################################

/** Clamp all floating point values of this vector to the given value.
 * @param op Clamp value.
 */
void Color4::clamp4(float op)
{
  if(this->rf > op)
  {
    this->rf = op;
  }
  if(this->gf > op)
  {
    this->gf = op;
  }
  if(this->bf > op)
  {
    this->bf = op;
  }
  if(this->af > op)
  {
    this->af = op;
  }
}

/** 3 component version of clamp.
 * @param op Clamp value.
 */
void Color4::clamp3(float op)
{
  if(this->rf > op)
  {
    this->rf = op;
  }
  if(this->gf > op)
  {
    this->gf = op;
  }
  if(this->bf > op)
  {
    this->bf = op;
  }
}

/** Round vector from float to fixed.
 */
void Color4::round4()
{
  this->ri.set_roundable(rf);
  this->gi.set_roundable(gf);
  this->bi.set_roundable(bf);
  this->ai.set_roundable(af);
}

/** Round the first two components of vector from float to fixed.
 */
void Color4::round3()
{
  this->ri.set_roundable(rf);
  this->gi.set_roundable(gf);
  this->bi.set_roundable(bf);
}

/** Initialize with roundable values.
 * @param r R component.
 * @param g G component.
 * @param b B component.
 */
void Color4::set_roundable(int r, int g, int b)
{
  this->ri.set_roundable(r);
  this->gi.set_roundable(g);
  this->bi.set_roundable(b);
}

/** Initialize with roundable values.
 * @param r R component.
 * @param g G component.
 * @param b B component.
 * @param a A component.
 */
void Color4::set_roundable(int r, int g, int b, int a)
{
  this->ri.set_roundable(r);
  this->gi.set_roundable(g);
  this->bi.set_roundable(b);
  this->ai.set_roundable(a);
}

//############################################################################
// End #######################################################################
//############################################################################

}
#endif

