#ifndef libfhi_canonicaldb_include
#define libfhi_canonicaldb_include

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi_default.h"

#include <map>
#include <string>

namespace libfhi {

//############################################################################
// Class #####################################################################
//############################################################################

/** CanonicalContainer is a container that houses the string containing the
 * canonical name of the unit and a pointer to the unit.
 */
template <class Type>
class CanonicalContainer
{
  private:
    /** String containing the canonical representation of the name. */
    std::string name;

    /** The contained object. */
    Type *obj;

  public:
    CanonicalContainer(const char *);
    CanonicalContainer(const char *, Type*);
    ~CanonicalContainer();

    // Inline
    inline const char* get_name() const;
    inline Type* get_obj();
};

/** CanonicalDB is a template cotainer for different types of data. It is
 * used to open and save things based on canonization of their file name
 * (or a new name altogether if none specified). Note that the CanonicalDB has
 * no constructors and no non-static functions, it always exists.
 * TODO: Usage counters and working set / trash discard?
 */
template <class Type>
class CanonicalDB
{
  private:
    /** The hash map storing all entities. */
    static std::map<std::string, void*> canon_data;

  public:
    static void canon_clear();
    static bool canon_delete(const char*);
    static Type* canon_get(const char*);
    static Type* canon_load(const char*);
    static Type* canon_new(const char*);
    static Type* canon_save(const char*, Type*);
};

//############################################################################
// Globals ###################################################################
//############################################################################

template <class Type>
std::map<std::string, void*> CanonicalDB<Type>::canon_data;

//############################################################################
// Helper inline functions ###################################################
//############################################################################

/** Canonifies a character string into a c++ string.
 * @param str The input string.
 * @return C++ string object.
 */
static inline std::string canonify_string(const char *str)
{
  return std::string(str);
}

//############################################################################
// Inline CanonicalContainer #################################################
//############################################################################

/** Get the name of this container.
 * @return The name as const char.
 */
template <class Type>
const char* CanonicalContainer<Type>::get_name() const
{
  return this->name.c_str();
}

/** Get the type from this container.
 * @return The type pointer.
 */
template <class Type>
Type* CanonicalContainer<Type>::get_obj()
{
  return this->obj;
}

//############################################################################
// CanonicalContainer ########################################################
//############################################################################

/** Default constructor. Specify the name, then create a child object with
 * the given name (filename?), canonify it, and store.
 * @param str The name to use.
 */
template <class Type>
CanonicalContainer<Type>::CanonicalContainer(const char *str)
  : name(canonify_string(str))
{
  this->obj = new Type(str);
}

/** Alternate constructor. Specify the name and the object as separate
 * entities (the object is assumed already created).
 * @param str The name to use.
 * @param op The object pointer.
 */
template <class Type>
CanonicalContainer<Type>::CanonicalContainer(const char *str, Type *op)
  : name(canonify_string(str))
{
  this->obj = op;
}

/** Default destructor. */
template <class Type>
CanonicalContainer<Type>::~CanonicalContainer()
{
  delete this->obj;
}

//############################################################################
// CanonicalDB ###############################################################
//############################################################################

/** Clear the database freeing all memory. Should be called for all classes
 * implementing CanonicalDB on libfhi quit.
 */
template <class Type>
void CanonicalDB<Type>::canon_clear()
{
  while(!canon_data.empty())
  {
    std::map<std::string, void*>::iterator iter = canon_data.begin();
    CanonicalContainer<Type> *obj =
      static_cast<CanonicalContainer<Type>*>((*iter).second);
    
    delete obj;
    canon_data.erase(iter);
  }
}

/** Delete the object pointed by the given string.
 * @param str The string to look with.
 * @return False if not found, true if found and deleted.
 */
template <class Type>
bool CanonicalDB<Type>::canon_delete(const char *str)
{
  std::map<std::string, void*>::iterator iter =
    canon_data.find(canonify_string(str));

  if(iter == canon_data.end())
  {
    return false;
  }

  delete static_cast<CanonicalContainer<Type>*>((*iter).second);
  canon_data.erase(iter);
  return true;
}

/** Get an object pointed by given string.
 * @param str The string to look with.
 * @return Pointer to the object or NULL on error.
 */
template <class Type>
Type* CanonicalDB<Type>::canon_get(const char *str)
{
  std::map<std::string, void*>::iterator iter =
    canon_data.find(canonify_string(str));

  if(iter == canon_data.end())
  {
    return NULL;
  }
  return static_cast<CanonicalContainer<Type>*>((*iter).second)->get_obj();
}

/** Load up a new object.
 * @param str The string to use to initialize the object (filename?).
 */
template <class Type>
Type* CanonicalDB<Type>::canon_load(const char *str)
{
  CanonicalContainer<Type> *ccon = new CanonicalContainer<Type>(str);

  canon_data[std::string(ccon->get_name())] = ccon;

  return ccon->get_obj();
}

/** Create a new object in a database under a given name.
 * @param str The string to save the object under.
 */
template <class Type>
Type* CanonicalDB<Type>::canon_new(const char *str)
{
  return canon_save(str, new Type());
}

/** Save a new object in a database under a given name.
 * @param str The string to save the object under.
 * @param obj New object to save.
 * @return The object that was saved in the database.
 */
template <class Type>
Type* CanonicalDB<Type>::canon_save(const char *str, Type *obj)
{
  CanonicalContainer<Type> *ccon = new CanonicalContainer<Type>(str, obj);

  canon_data[ccon->get_name()] = ccon;
  return obj;
}

//############################################################################
// End #######################################################################
//############################################################################

}
#endif

