#ifndef libfhi_attributable_include
#define libfhi_attributable_include

//############################################################################
// Includes ##################################################################
//############################################################################

#include "libfhi_default.h"

namespace libfhi {

//############################################################################
// Class #####################################################################
//############################################################################

/** Attributable is the base class that allows for bit-flag attribute
 * behavior. Several classes appeared to use this, so I thought it was
 * better to make it's own class out of it.
 * Note that attributable defines no non-inline functions.
 */
class Attributable
{
  protected:
    /** The attribute mask. */
    uint8_t attributes;

  public:
    inline Attributable();
    inline Attributable(uint8_t);

    // Inline
    inline void attr_and(uint8_t);
    inline void attr_not(uint8_t);
    inline void attr_or(uint8_t);
    inline uint8_t get_attr() const;
    inline bool has_attr(uint8_t) const;
    inline void set_attr(uint8_t);
};

//############################################################################
// Inline ####################################################################
//############################################################################

/** Default constructor. */
Attributable::Attributable()
{
  this->attributes = 0x00;
}

/** Set contructor.
 * @param op The initial attributes.
 */
Attributable::Attributable(uint8_t op)
{
  this->attributes = op;
}

/** And the attributes with given attribute mask.
 * @param op [in] Preserve these attributes, destroy the rest.
 */
void Attributable::attr_and(uint8_t op)
{
  this->attributes &= op;
}

/** Remove certain attributes.
 * @param op [in] The attributes to remove as a uint8_t mask.
 */
void Attributable::attr_not(uint8_t op)
{
  this->attributes &= 0xFF - op;
}

/** Add certain attributes.
 * @param op [in] Include these attributes.
 */
void Attributable::attr_or(uint8_t op)
{
  this->attributes |= op;
}

/** Return the attributes.
 * @return The attribute integer.
 */
uint8_t Attributable::get_attr() const
{
  return this->attributes;
}

/** Tell if this has given attribute.
 * @return True if yes, false if no.
 */
bool Attributable::has_attr(uint8_t op) const
{
  return ((this->attributes & op) != 0);
}

/** Replace the attributes.
 * @param op [in] New attributes, omit for defaults.
 */
void Attributable::set_attr(uint8_t op)
{
  this->attributes = op;
}

//############################################################################
// End #######################################################################
//############################################################################

}
#endif

