/* Functions for building objects to put in scenes */

/* The components of an object depend on the lighting model it uses, but in general include:
 * model: a function from rendering.js which takes the object and a world transformation matrix,
 *        and does its rendery goodness
 * v: a list of vertices; each of which is a 3-vector
 * p: a list of polygons; each of which is a triplet of indexes into the v array
 * additionally, renderFlat expects n: a list of normals, one for each poly
 * renderGouraud expects n: a list of normals, one for each vertex
 */
function attachFaceNormals(obj) { /* one normal per poly - only meaningful for renderFlat! */
	obj.n = [];
	for (var i = 0; i < obj.p.length; i++) {
		var poly = [obj.v[obj.p[i][0]], obj.v[obj.p[i][1]], obj.v[obj.p[i][2]]];
		var a = [poly[2][0] - poly[1][0], poly[2][1] - poly[1][1], poly[2][2] - poly[1][2]];
		var b = [poly[1][0] - poly[0][0], poly[1][1] - poly[0][1], poly[1][2] - poly[0][2]];
		obj.n[i] = normalise([a[1]*b[2]-a[2]*b[1], a[2]*b[0]-a[0]*b[2], a[0]*b[1]-a[1]*b[0]]);
	}
	return obj;
}

function makeSphere(r, m, n, colour, actualM) {
	/* r = radius
	 * m = number of latitude subdivisions
	 * n = number of longitude subdivisions
	 * actualM = number of latitude subdivisions that we actually plot
	 */
	if (!actualM) actualM = m;
	var vert = [];
	var norm = [];
	var poly = [];
	/* place vertex at north pole */
	vert[0] = [0, r, 0]; norm[0] = [0, 1, 0];
	/* then fill in the rest */
	for (var i = 1; i < actualM; i++) {
		for (var j = 0; j < n; j++) {
			var sini = Math.sin(Math.PI * i / m);
			var currentNorm = [
				sini * Math.cos(2*Math.PI*j / n),
				Math.cos(Math.PI*i/m),
				sini * Math.sin(2*Math.PI*j / n)
			];
			norm[(i-1)*n + j + 1] = currentNorm;
			vert[(i-1)*n + j + 1] = [r * currentNorm[0], r * currentNorm[1], r * currentNorm[2]];
		}
	}
	if (m == actualM) {
		/* south pole */
		vert[(m-1)*n+1] = [0, -r, 0]; norm[(m-1)*n+1] = [0, -1, 0];
	}
	/* now do the polys. First n polys attach to the north pole to form a cap */
	for (var i = 0; i < n; i++) {
		poly.push([0, i+1, ((i+1)%n) + 1]);
	}
	/* next, for each latitude row from 1 to m-2, each vertex connects to the one below it forming quads,
	and then those quads are slashed */
	for (var i = 1; i < actualM-1; i++) {
		var firstInRow = 1 + (i-1)*n;
		for (var j = 0; j < n; j++) {
			poly.push([firstInRow + j, firstInRow + n + j, firstInRow + n + ((j+1)%n)]);
			poly.push([firstInRow + j, firstInRow + n + ((j+1)%n), firstInRow + ((j+1)%n)]);
		}
	}
	/* finally attach row m-1 to the south pole */
	if (actualM == m) {
		for (var i = 0; i < n; i++) {
			poly.push([1+(m-2)*n + i, 1+(m-1)*n, 1+(m-2)*n + ((i+1)%n)]);
		}
	}
//	alert(vert.length + ' vertices:');
//	for (i = 0; i < vert.length; i++) alert(vert[i]);
//	alert(poly.length + ' polys:');
//	for (i = 0; i < poly.length; i++) alert(poly[i]);
	
	return { model: renderGouraud, lighting: LIGHT_LAMBERTIAN, v: vert, p: poly, n: norm, colour: colour };
}

function makeFlower(r, k, n, m, outsideColour, insideColour) {
	/* r = radius when closed; k = radius scale factor (= how far opened);
	n = number of segments per petal; m = number of petals */
	var vert = [];
	var norm = [];
	var poly = [];
	vert[0] = [0, 0, 0];
	norm[0] = [0, -1, 0];
	var vcount = 1; /* vertices plotted prior to start of current row; index of start of current row */
	for (i = 1; i < n; i++) {
		var a = i * Math.PI / (k * n);
		var ynorm = Math.cos(a);
		var y = k*r*(1-ynorm);
		var rad = k * Math.sin(a);
		var range = Math.PI * Math.sin(i * Math.PI / n) / (m * rad);
		for (j = 0; j < m; j++) {
			var midpt = 2 * Math.PI * j / m;
			var xnorm = Math.cos(midpt - range);
			var znorm = Math.sin(midpt - range);
			norm.push([xnorm, ynorm, znorm]);
			vert.push([
				r * rad * xnorm,
				y,
				r * rad * znorm,
			]);
			xnorm = Math.cos(midpt + range);
			znorm = Math.sin(midpt + range);
			norm.push([xnorm, ynorm, znorm]);
			vert.push([
				r * rad * xnorm,
				y,
				r * rad * znorm,
			]);
			
			if (i == 1) {
				poly.push([0, vcount+j+j+1, vcount+j+j]); /* outside face */
			} else {
				poly.push([vcount-m-m+j+j, vcount-m-m+j+j+1, vcount+j+j]);
				poly.push([vcount-m-m+j+j+1, vcount+j+j+1, vcount+j+j]); /* outside face */
			}
		}
		vcount += m+m;
	}
	var ynorm = Math.cos(Math.PI / k);
	var y = k*r*(1 - ynorm);
	var rad = k * Math.sin(Math.PI / k);
	for (j = 0; j < m; j++) {
		var midpt = 2 * Math.PI * j / m;
		var xnorm = Math.cos(midpt);
		var znorm = Math.sin(midpt);
		norm.push([xnorm, ynorm, znorm]);
		vert.push([
			r * rad * xnorm,
			y,
			r * rad * znorm
		]);
		poly.push([vcount-m-m+j+j, vcount-m-m+j+j+1, vcount+j]);
	}
	return { model: renderGouraud, lighting: LIGHT_LAMBERTIAN, v: vert, n: norm, p:poly, colour: outsideColour, altColour: insideColour};
}

function makeTorus(rmaj, rmin, m, n, colour) {
	/* rmaj = major radius; rmin = minor radius; m = number of subdivisions of major circle;
	n = number of subdivisions of minor circle */
	vert = [];
	norm = [];
	poly = [];
	for (var i = 0; i < m; i++) {
		var a = 2 * Math.PI * i / m;
		var cos_a = Math.cos(a); var sin_a = Math.sin(a);
		for (var j = 0; j < n; j++) {
			var b = 2 * Math.PI * j / n;
			var cos_b = Math.cos(b); var sin_b = Math.sin(b);
			vert.push([rmaj*cos_a + rmin*cos_a*cos_b, rmin*sin_b, -rmaj*sin_a - rmin*sin_a*cos_b]);
			norm.push([cos_a*cos_b, sin_b, -sin_a*cos_b]);
//			poly.push([i*n+j, ((i+1)%m)*n + ((j+1)%n), i*n + ((j+1)%n)]);
//			poly.push([i*n+j, ((i+1)%m)*n + j, ((i+1)%m)*n + ((j+1)%n)]);
			poly.push([i*n+j, i*n + ((j+1)%n), ((i+1)%m)*n + ((j+1)%n)]);
			poly.push([i*n+j, ((i+1)%m)*n + ((j+1)%n), ((i+1)%m)*n + j]);
		}
	}
	return { model: renderGouraud, lighting: LIGHT_LAMBERTIAN, v: vert, n: norm, p: poly, colour: colour };
}
