package file_resampler;

import java.io.*;

/**
 * <p>Title: FIleRESAmpler</p>
 *
 * <p>Description: A Program to reduce or expand an file by a factor of 2.
 * This is done by averaging the bytes in the file.</p>
 *
 * <p>Copyright: Copyright (c) 2004 Jo-Herman Haugholt</p>
 *
 * <p>License: This Program is distributed under the terms of the GNU
 * General Public License</p>
 *
 * <p>This file is part of FIleRESAmpler. </p>
 *
 * <p> FIleRESAmpler is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version. </p>
 *
 * <p> FIleRESAmpler is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details. </p>
 *
 * <p> You should have received a copy of the GNU General Public License
 * along with FIleRESAmpler; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * </p>
 *
 * @author Jo-Herman Haugholt
 * @version 1.0
 */

public class firesa {

  private final static int EXPAND = 0;
  private final static int REDUCE = 1;
  private final static String VERSION = "1.0";

  public static void main(String[] args) {
    System.out.println("FIleRESAmpler " + VERSION);
    System.out.println("FIleRESAmpler is distributed under the GNU General Public License, see Readme.txt for details");
    if (args.length < 3) {
      System.out.println("Syntax: firasa [x][r] <inputfile> <outputfile>" +
                         "x\tExpand file" +
                         "r\tReduce file");
      System.exit( -1);
    }
    int operation = -1;
    if (args[0].compareToIgnoreCase("x") == 0) {
      operation = EXPAND;
    }
    else if (args[0].compareToIgnoreCase("r") == 0) {
      operation = REDUCE;
    }
    else {
      System.err.print("Invalid operation: " + args[0]);
      System.exit( -1);
    }
    try {
      FileInputStream file = new FileInputStream(args[1]);
      ByteArrayOutputStream bytes = new ByteArrayOutputStream();
      int i = 0;
      while (i != -1) {
        i = file.read();
        bytes.write(i);
      }
      file.close();
      byte outdata[] = null;
      switch (operation) {
        case EXPAND:
          outdata = SampleUp(bytes.toByteArray());
          break;
        case REDUCE:
          outdata = SampleDown(bytes.toByteArray());
          break;
      }

      FileOutputStream outfile = new FileOutputStream(args[2]);
      outfile.write(outdata);
      outfile.close();
    }
    catch (FileNotFoundException ex) {
      System.err.println("Error opening file: " + ex.getMessage());
    }
    catch (IOException ex) {
      System.err.println("IO error: " + ex.getMessage());
    }
    System.out.println("Operation Complete");
  }

  /**
   * This method takes an input array of bytes, and roughly doubles it.
   * (Actually to a size of (2*x)-1). New bytes are calculated by averaging two
   * neighbouring bytes.
   *
   * @param data byte[] The original byte array
   * @return byte[] The adjusted byte array
   */
  private static byte[] SampleUp(byte data[]) {
    byte newdata[] = new byte[ (int) (2 * data.length) - 1];
    int origcount = 0;
    int newcount = 0;
    newdata[0] = data[0];
    while (origcount + 1 < data.length && newcount + 1 < newdata.length) {
      newcount++;
      newdata[newcount] = (byte) ( (data[origcount] + data[origcount + 1]) / 2);
      newcount++;
      origcount++;
      newdata[newcount] = data[origcount];
    }
    return newdata;
  }

  /**
   * This method takes an input array of bytes, and halves it it. Two and two
   * bytes are averaged together to form the new byte..
   *
   * @param data byte[] The original byte array
   * @return byte[] The adjusted byte array
   */
  private static byte[] SampleDown(byte data[]) {
    byte newdata[] = new byte[ (int) (data.length / 2)];
    int origcount = 0;
    int newcount = 0;
    while (origcount < data.length && newcount < newdata.length) {
      newdata[newcount] = (byte) ( (data[origcount] + data[origcount + 1]) / 2);
      newcount++;
      origcount += 2;
    }
    return newdata;
  }
}
