///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>
#include <shared/ThString.h>

#include <Theresa/WGL/ThDialog.h>
#include <Theresa/WGL/resource.h>

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThDialog constructors --------------------------------------------------------------------------

ThDialog::ThDialog(WORD resource):
	m_resource(resource),
	m_window(NULL)
{
}

ThDialog::~ThDialog(void)
{
}

// ThDialog methods -------------------------------------------------------------------------------

bool ThDialog::create(HWND parent)
{
	if (!CreateDialogParam(GetModuleHandle(NULL), MAKEINTRESOURCE(m_resource), parent, (DLGPROC) dialogCallback, (LPARAM) this))
		return false;

	return true;
}

INT_PTR ThDialog::display(HWND parent)
{
	return DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(m_resource), parent, (DLGPROC) dialogCallback, (LPARAM) this);
}

void ThDialog::release(INT_PTR result)
{
	if (!m_window)
		return;

	EndDialog(m_window, result);
}

void ThDialog::show(void)
{
	if (!m_window)
		return;

	ShowWindow(m_window, SW_SHOW);
	UpdateWindow(m_window);
}

void ThDialog::hide(void)
{
	if (!m_window)
		return;

	ShowWindow(m_window, SW_HIDE);
}

// ThDialog message methods -----------------------------------------------------------------------

LRESULT ThDialog::sendMessage(WORD identifier, UINT message, WPARAM wParam, LPARAM lParam)
{
	return SendDlgItemMessage(m_window, identifier, message, wParam, lParam);
}

BOOL ThDialog::postMessage(WORD identifier, UINT message, WPARAM wParam, LPARAM lParam)
{
	return PostMessage(GetDlgItem(m_window, identifier), message, wParam, lParam);
}

// ThDialog attributes ----------------------------------------------------------------------------

bool ThDialog::isCreated(void) const
{
	return m_window != NULL;
}

bool ThDialog::isVisible(void) const
{
	if (!m_window)
		return false;

	return IsWindowVisible(m_window) ? true : false;
}

HWND ThDialog::getHandle(void) const
{
	return m_window;
}

HWND ThDialog::getControl(WORD identifier) const
{
	if (!m_window)
		return NULL;

	return GetDlgItem(m_window, identifier);
}

bool ThDialog::getTitle(ThString& title) const
{
	if (!m_window)
		return false;

	int length = GetWindowTextLength(m_window);

	title.allocate(length + 1);

	if (length)
	{
		if (!GetWindowText(m_window, title, title.getSize()))
			return false;
	}

	return true;
}

void ThDialog::setTitle(const char* title)
{
	if (!m_window)
		return;

	SetWindowText(m_window, title);
}

INT_PTR ThDialog::getResult(void) const
{
	return GetWindowLongPtr(m_window, DWL_MSGRESULT);
}

void ThDialog::setResult(INT_PTR result)
{
	SetWindowLongPtr(m_window, DWL_MSGRESULT, result);
}

// ThDialog callbacks -----------------------------------------------------------------------------

BOOL ThDialog::initialize(HWND control)
{
	return TRUE;
}

BOOL ThDialog::command(HWND control, WORD identifier, BOOL accel)
{
	switch (identifier)
	{
		case IDOK:
		case IDCANCEL:
		{
			release(identifier);
			break;
		}
	}

	return FALSE;
}

BOOL ThDialog::message(UINT message, WPARAM wParam, LPARAM lParam)
{
	return FALSE;
}

// ThDialog static methods ------------------------------------------------------------------------

INT_PTR CALLBACK ThDialog::dialogCallback(HWND window, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_INITDIALOG:
		{
			SetWindowLongPtr(window, DWLP_USER, lParam);

			HICON icon = LoadIcon(GetModuleHandle(NULL), MAKEINTRESOURCE(IDI_HYPERCUBE));

			SetClassLongPtr(window, GCL_HICON, (LONG) icon);
			SetClassLongPtr(window, GCL_HICONSM, (LONG) icon);

			if (ThDialog* dialog = reinterpret_cast<ThDialog*>(lParam))
			{
				dialog->m_window = window;

				return dialog->initialize((HWND) wParam);
			}

			return TRUE;
		}

		case WM_COMMAND:
		{
			if (ThDialog* dialog = reinterpret_cast<ThDialog*>(GetWindowLongPtr(window, DWLP_USER)))
			{
				return dialog->command(GET_WM_COMMAND_HWND(wParam, lParam), GET_WM_COMMAND_ID(wParam, lParam), GET_WM_COMMAND_CMD(wParam, lParam));
			}

			break;
		}

		default:
		{
			if (ThDialog* dialog = reinterpret_cast<ThDialog*>(GetWindowLongPtr(window, DWLP_USER)))
			{
				return dialog->message(message, wParam, lParam);
			}

			break;
		}
	}

	return FALSE;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
