///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>

#include <Theresa/WGL/ThCore.h>

#include <cstdio>
#include <cstdlib>
#include <cstring>

///////////////////////////////////////////////////////////////////////////////////////////////////

const char* getPlatformName(void)
{
	static char platformName[THERESA_BUFFER_SIZE];

	strcpy(platformName, "Windows");

	OSVERSIONINFO osvi;

	memset(&osvi, 0, sizeof(OSVERSIONINFO));
	osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);

	if (!GetVersionEx(&osvi))
		return platformName;

	char version[THERESA_BUFFER_SIZE];

	if (osvi.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS)
	{
		if (osvi.dwMinorVersion)
		{
			if (osvi.dwMinorVersion == 10)
			{
				strcat(platformName, " 98");

        if (osvi.szCSDVersion[1] == 'A')
          strcat(platformName, "SE");
				else if (osvi.szCSDVersion[0] != '\0')
				{
					strcat(platformName, " ");
					strcat(platformName, osvi.szCSDVersion);
				}
			}
			else if (osvi.dwMinorVersion == 90)
				strcat(platformName, " Me");
			else
				strcat(platformName, " 9x family");
		}
		else
		{
			strcat(platformName, " 95");

			if (osvi.szCSDVersion[1] == 'C')
				strcat(platformName, " OSR2");
			else if (osvi.szCSDVersion[0] != '\0')
			{
				strcat(platformName, " ");
				strcat(platformName, osvi.szCSDVersion);
			}
		}
	}
	else if (osvi.dwPlatformId == VER_PLATFORM_WIN32_NT)
	{
		if (osvi.dwMajorVersion == 4)
			strcat(platformName, " NT");
		else if (osvi.dwMajorVersion == 5)
		{
			if (osvi.dwMinorVersion)
			{
				if (osvi.dwMinorVersion == 1)
					strcat(platformName, " XP");
				else
					strcat(platformName, " XP family");
			}
			else
				strcat(platformName, " 2000");
		}
		else
			strcat(platformName, " NT family");

		OSVERSIONINFOEX osviex;
		memset(&osviex, 0, sizeof(OSVERSIONINFOEX));
		osviex.dwOSVersionInfoSize = sizeof(OSVERSIONINFOEX);

		if (GetVersionEx((OSVERSIONINFO*) &osviex))
    {
      if (osviex.wProductType == VER_NT_WORKSTATION)
      {
        if (osviex.wSuiteMask & VER_SUITE_PERSONAL)
          strcat(platformName, " Home Edition");
        else
          strcat(platformName, " Professional");
      }
      else if (osviex.wProductType == VER_NT_SERVER)
      {
        if (osviex.wSuiteMask & VER_SUITE_DATACENTER)
          strcat(platformName, " DataCenter Server");
        else if (osviex.wSuiteMask & VER_SUITE_ENTERPRISE)
          strcat(platformName, " Advanced Server");
        else
          strcat(platformName, " Server");
      }
    }

		if (osvi.dwMajorVersion == 4)
		{
			snprintf(version, sizeof(version), " version %u.%u", osvi.dwMajorVersion, osvi.dwMinorVersion);
			strcat(platformName, version);
		}

		if (osvi.szCSDVersion[0] != '\0')
		{
			strcat(platformName, " ");
			strcat(platformName, osvi.szCSDVersion);
		}

		snprintf(version, sizeof(version), " (Build %u)", osvi.dwBuildNumber);
		strcat(platformName, version);
	}

	return platformName;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
#ifdef THERESA_DEBUG_BUILD
///////////////////////////////////////////////////////////////////////////////////////////////////

//! \note This function may not use \b any system code.
void writeDebugMessage(const char* message)
{
	OutputDebugString(message);
	OutputDebugString(THERESA_NEWLINE);
}

//! \note This function may not use \b any system code.
bool assertExpression(const char* expression, const char* message, const char* fileName, unsigned int line)
{
	static char buffer[THERESA_BUFFER_SIZE];

	if (snprintf(buffer, sizeof(buffer), "Assertion failed!%s%sExpression: %s%sFile: %s%sLine: %u%s%s%s", THERESA_NEWLINE, THERESA_NEWLINE, expression, THERESA_NEWLINE, fileName, THERESA_NEWLINE, line, THERESA_NEWLINE, THERESA_NEWLINE, message) < 0)
		buffer[sizeof(buffer) - 1] = '\0';

	writeDebugMessage(buffer);

	int result = MessageBox(GetActiveWindow(), buffer, "Theresa", MB_ICONERROR | MB_ABORTRETRYIGNORE);

	if (result == IDABORT)
		exit(0);

	if (result == IDIGNORE)
		return false;

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
#endif /* THERESA_DEBUG_BUILD */
///////////////////////////////////////////////////////////////////////////////////////////////////

// ThCriticalSection constructors -----------------------------------------------------------------

ThCriticalSection::ThCriticalSection(void)
{
}

ThCriticalSection::~ThCriticalSection(void)
{
	close();
}

// ThCriticalSection methods ----------------------------------------------------------------------

bool ThCriticalSection::open(void)
{
	InitializeCriticalSection(&m_section);
		
	return true;
}

void ThCriticalSection::close(void)
{
	DeleteCriticalSection(&m_section);
}

// ThCriticalSection interface methods ------------------------------------------------------------

void ThCriticalSection::begin(void)
{
	EnterCriticalSection(&m_section);
}

void ThCriticalSection::end(void)
{
	LeaveCriticalSection(&m_section);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMutex constructors ---------------------------------------------------------------------------

ThMutex::ThMutex(void)
{
	m_mutex = NULL;
}

ThMutex::~ThMutex(void)
{
	close();
}

// ThMutex methods --------------------------------------------------------------------------------

bool ThMutex::open(void)
{
	m_mutex = CreateMutex(NULL, FALSE, NULL);
	if (!m_mutex)
		return false;

	return true;
}

void ThMutex::close(void)
{
	if (m_mutex)
	{
		CloseHandle(m_mutex);
		m_mutex = NULL;
	}
}

// ThMutex interface methods ----------------------------------------------------------------------

bool ThMutex::request(float timeout)
{
	DWORD milliseconds;

	if (!timeout)
		milliseconds = INFINITE;
	else
		milliseconds = (DWORD) (timeout * 1000.f);

	switch (WaitForSingleObject(m_mutex, milliseconds))
	{
		case WAIT_OBJECT_0:
			return true;

		case WAIT_ABANDONED:
		{
			if (WaitForSingleObject(m_mutex, 0) == WAIT_OBJECT_0)
				return true;

			return false;
		}

		case WAIT_TIMEOUT:
			return false;
	}

	return false;
}

void ThMutex::release(void)
{
	ReleaseMutex(m_mutex);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
