///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMath.h>
#include <shared/ThMemory.h>
#include <shared/ThString.h>
#include <shared/ThTimer.h>
#include <shared/ThError.h>
#include <shared/ThStream.h>
#include <shared/ThStorage.h>
#include <shared/ThServer.h>
#include <shared/ThDisplay.h>
#include <shared/ThMusic.h>
#include <shared/ThEngine.h>
#include <shared/ThEffect.h>
#include <shared/ThSystem.h>

#include <theresa/ThSystem.h>

#if HAVE_TIME_H
	#include <time.h>
#endif

#if HAVE_STDLIB_H
	#include <stdlib.h>
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////

ThSingleton<IThSystem> System;

///////////////////////////////////////////////////////////////////////////////////////////////////

extern bool initializePlatform(int argc, char* argv[]);
extern void shutdownPlatform(void);

///////////////////////////////////////////////////////////////////////////////////////////////////

bool initializeTheresa(int argc, char* argv[])
{
	srand(time(NULL));
	
	if (!initializePlatform(argc, argv))
		return false;

	if (!IThError::create())
	{
#if THERESA_DEBUG_BUILD
		writeDebugMessage("Setup was unable to create error object");
#endif
		return false;
	}

	if (!IThStorage::create("data.zip"))
	{
		if (!IThStorage::create())
		{
			Error->display("Setup", "Unable to create storage object.");
			return false;
		}
	}

	if (!IThServer::create())
	{
		Error->display("Setup", "Unable to create server object.");
		return false;
	}
	
	if (!IThSystem::create())
	{
		Error->display("Setup", "Unable to create system object.");
		return false;
	}

	return true;
}

void shutdownTheresa(void)
{
	if (System)
		System.release();
	
	if (Server)
		Server.release();

	if (Storage)
		Storage.release();

	if (Error)
		Error.release();
		
	shutdownPlatform();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThApplication constructors ---------------------------------------------------------------------

ThApplication::ThApplication(void)
{
}

ThApplication::~ThApplication(void)
{
}

//  ThApplication callbacks -----------------------------------------------------------------------

bool ThApplication::receive(const IThMessage* message)
{
	switch (message->getMessage())
	{
		case THMSG_LOCAL_UPDATE:
		{
			const float deltaTime = *reinterpret_cast<const float*>(message->getData());

			return update(deltaTime);
		}

		case THMSG_LOCAL_RENDER:
			return render(System->getDisplay()->getDefaultCanvas());

		case THMSG_REQUEST_EXIT:
		{
			if (System->isStarted())
				System->stop();

			break;
		}
	}

	return ThClient::receive(message);
}

bool ThApplication::update(float deltaTime)
{
	return true;
}

bool ThApplication::render(IThCanvas* target)
{
	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThSystem constructors -------------------------------------------------------------------------

IThSystem::~IThSystem(void)
{
}

// IThSystem static methods -----------------------------------------------------------------------

bool IThSystem::create(void)
{
	if (System)
		return true;

	ThPtr<ThSystem> system = new ThSystem();

	if (!system->open())
		return false;

	System.attach(system.detach());
	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThSystem constructors --------------------------------------------------------------------------

ThSystem::ThSystem(void):
	ThServerObject(THID_SYSTEM)
{
	m_prevTime = 0.f;
}

ThSystem::~ThSystem(void)
{
	close();
}

// ThSystem methods -------------------------------------------------------------------------------

bool ThSystem::open(void)
{
	close();

	return true;
}

void ThSystem::close(void)
{
	THASSERT(m_timer.isStarted() == false, "Cannot release a running system.");
	
	closeDisplay();
	closeMusic();
	
	m_prevTime = 0.f;
}

// ThSystem interface methods ---------------------------------------------------------------------

bool ThSystem::start(void)
{
	sendMessage(THMSG_SYSTEM_START, THID_ANNOUNCE);

	m_timer.start();

	startSystemMessageLoop(getID());
	
	m_timer.stop();

	sendMessage(THMSG_SYSTEM_STOP, THID_ANNOUNCE);

	return true;
}

void ThSystem::stop(void)
{
	stopSystemMessageLoop();
}

void ThSystem::pause(void)
{
	if (m_timer.isPaused())
		return;
		
	m_timer.pause();

	if (m_music)
		m_music->pause();
		
	sendMessage(THMSG_SYSTEM_PAUSE, THID_ANNOUNCE);
}

void ThSystem::resume(void)
{
	if (!m_timer.isPaused())
		return;
		
	m_timer.resume();

	if (m_music)
		m_music->resume();

	sendMessage(THMSG_SYSTEM_RESUME, THID_ANNOUNCE);
}

// ThSystem interface display methods -------------------------------------------------------------

bool ThSystem::openDisplay(const ThContextMode* mode)
{
	closeDisplay();

	m_display = IThDisplay::createInstance(mode);
	if (!m_display)
		return false;

	sendMessage(THMSG_DISPLAY_OPEN, THID_ANNOUNCE);
	return true;
}

void ThSystem::closeDisplay(void)
{
	closeEffect();
	closeEngine();

	sendMessage(THMSG_DISPLAY_CLOSE, THID_ANNOUNCE);

	m_display.release();
}

// ThSystem interface music methods ---------------------------------------------------------------

bool ThSystem::openMusic(const char* fileName)
{
	closeMusic();

	ThPtr<IThStream> file = Storage->openFile(fileName);
	if (!file)
		return false;

	m_music = IThMusic::createInstance(file);
	if (!m_music)
		return false;

	sendMessage(THMSG_MUSIC_OPEN, THID_ANNOUNCE);
	return true;
}

void ThSystem::closeMusic(void)
{
	sendMessage(THMSG_MUSIC_CLOSE, THID_ANNOUNCE);

	m_music.release();
}

// ThSystem interface effect methods --------------------------------------------------------------

bool ThSystem::openEffect(void)
{
	closeEngine();

	if (!m_display)
		return false;

	m_effect = IThEffect::createInstance(m_display);
	if (!m_effect)
		return false;

	sendMessage(THMSG_EFFECT_OPEN, THID_ANNOUNCE);
	return true;
}

void ThSystem::closeEffect(void)
{
	sendMessage(THMSG_EFFECT_CLOSE, THID_ANNOUNCE);

	m_effect.release();
}

// ThSystem interface engine methods --------------------------------------------------------------

bool ThSystem::openEngine(void)
{
	closeEngine();

	if (!m_display)
		return false;

	m_engine = IThEngine::createInstance(m_display);
	if (!m_engine)
		return false;

	sendMessage(THMSG_ENGINE_OPEN, THID_ANNOUNCE);
	return true;
}

void ThSystem::closeEngine(void)
{
	sendMessage(THMSG_ENGINE_CLOSE, THID_ANNOUNCE);

	m_engine.release();
}

// ThSystem interface attributes ------------------------------------------------------------------

bool ThSystem::isStarted(void)
{
	return m_timer.isStarted();
}

bool ThSystem::isPaused(void)
{
	return m_timer.isPaused();
}

float ThSystem::getTime(void)
{
	return m_timer.getTime();
}

void ThSystem::setTime(float time)
{
	m_timer.setTime(time);
	
	if (m_music)
		m_music->setTime(time);
}

IThDisplay* ThSystem::getDisplay(void)
{
	return m_display;
}

IThMusic* ThSystem::getMusic(void)
{
	return m_music;
}

IThEffect* ThSystem::getEffect(void)
{
	return m_effect;
}

IThEngine* ThSystem::getEngine(void)
{
	return m_engine;
}

// ThSystem callbacks -----------------------------------------------------------------------------

bool ThSystem::receive(const IThMessage* message)
{
	switch (message->getMessage())
	{
		case THMSG_SERVER_UPDATE:
		{
			const float currTime = m_timer.getTime();
			const float deltaTime = currTime - m_prevTime;
				
			sendMessage(THMSG_LOCAL_UPDATE, THID_ANNOUNCE, &deltaTime, sizeof(deltaTime));
			sendMessage(THMSG_LOCAL_RENDER, THID_ANNOUNCE);
			
			m_prevTime = currTime;
			break;
		}
	}

	return ThServerObject::receive(message);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
