///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>
#include <shared/ThString.h>
#include <shared/ThError.h>
#include <shared/ThScript.h>

#include <theresa/ThScript.h>

#if HAVE_STRING_H
	#include <string.h>
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThToken constructors ---------------------------------------------------------------------------

inline ThToken::ThToken(unsigned int symbol, unsigned int line):
	m_symbol(symbol),
	m_line(line)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThSymbolTable constructors --------------------------------------------------------------------

IThSymbolTable::~IThSymbolTable(void)
{
}

// IThSymbolTable static methods ------------------------------------------------------------------

IThSymbolTable* IThSymbolTable::createInstance(void)
{
	return new ThSymbolTable();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThTokenizer constructors ----------------------------------------------------------------------

IThTokenizer::~IThTokenizer(void)
{
}

// IThTokenizer static methods --------------------------------------------------------------------

IThTokenizer* IThTokenizer::createInstance(IThSymbolTable* symbolTable)
{
	ThPtr<ThTokenizer> tokenizer = new ThTokenizer();

	if (!tokenizer->open(symbolTable))
		return false;

	return tokenizer.detach();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThAnalyzer constructors -----------------------------------------------------------------------

IThAnalyzer::~IThAnalyzer(void)
{
}

// IThAnalyzer static methods ---------------------------------------------------------------------

IThAnalyzer* IThAnalyzer::createInstance(IThSymbolTable* symbolTable)
{
	ThPtr<ThAnalyzer> analyzer = new ThAnalyzer();

	if (!analyzer->open(symbolTable))
		return NULL;

	return analyzer.detach();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThSyntax constructors -------------------------------------------------------------------------

IThSyntax::~IThSyntax(void)
{
}

// IThSyntax static methods ---------------------------------------------------------------------

IThSyntax* IThSyntax::createInstance(IThSymbolTable* symbolTable, const char* script)
{
	ThPtr<ThSyntax> syntax = new ThSyntax();

	if (!syntax->open(symbolTable, script))
		return NULL;

	return syntax.detach();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThSymbolTable constructors ---------------------------------------------------------------------

ThSymbolTable::ThSymbolTable(void)
{
}

ThSymbolTable::~ThSymbolTable(void)
{
}

// ThSymbolTable methods --------------------------------------------------------------------------

bool ThSymbolTable::registerSymbol(unsigned int id, const char* name)
{
	if (id == THSYM_INVALID)
		return false;

	if (translateSymbol(id))
	{
		Error->write("SymbolTable", "Symbol '%s' has conflicting id %u with '%s'.", name, id, translateSymbol(id));
		return false;
	}

	if (translateSymbol(name) != THSYM_INVALID)
	{
		Error->write("SymbolTable", "Symbol %u has conflicting name '%s' with '%u'.", id, name, translateSymbol(translateSymbol(name)));
		return false;
	}

	m_symbols.attachFirst(new Symbol(id, name));
	return true;
}

const char* ThSymbolTable::translateSymbol(unsigned int id) const
{
	for (ThConstIterator<Symbol> symbol(m_symbols);  symbol;  symbol.next())
	{
		if (symbol->m_id == id)
			return symbol->m_name;
	}

	return NULL;
}

unsigned int ThSymbolTable::translateSymbol(const char* name) const
{
	const unsigned int hash = ThString::hash(name);

	for (ThConstIterator<Symbol> symbol(m_symbols);  symbol;  symbol.next())
	{
		if (symbol->m_hash == hash)
			return symbol->m_id;
	}

	return THSYM_INVALID;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThSymbolTable::Symbol constructors -------------------------------------------------------------

ThSymbolTable::Symbol::Symbol(unsigned int id, const char* name):
	m_id(id),
	m_name(name)
{
	m_hash = ThString::hash(name);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThTokenizer constructors -----------------------------------------------------------------------

ThTokenizer::ThTokenizer(void)
{
}

ThTokenizer::~ThTokenizer(void)
{
	close();
}

// ThTokenizer methods ----------------------------------------------------------------------------

bool ThTokenizer::open(IThSymbolTable* symbolTable)
{
	m_symbolTable = symbolTable;

	return true;
}

void ThTokenizer::close(void)
{
	m_symbolTable = NULL;

	m_patternLists.release();
}

// ThTokenizer interface methods ------------------------------------------------------------------

bool ThTokenizer::registerPattern(unsigned int list, unsigned int symbol, const char* data, const char* substitute, unsigned int branchList)
{
	if (!m_symbolTable->translateSymbol(list))
		return false;

	if (!m_symbolTable->translateSymbol(symbol))
		return false;

	ThPtr<Pattern> pattern = new Pattern(symbol, substitute, branchList);

	if (!compilePattern(pattern, data))
		return false;

	PatternList* patternList = findPatternList(list);
	if (!patternList)
	{
		patternList = new PatternList(list);
		m_patternLists.attachLast(patternList);
	}

	patternList->attachLast(pattern.detach());
	return true;
}

bool ThTokenizer::tokenize(ThTokenList& target, unsigned int root, const char* source)
{
	// this is here as an optimization
	ThString buffer;

	// this needs to persist across loop iterations
	ThToken* token = NULL;

	// begin on line one
	m_line = 1;

	// find initial pattern list
	PatternList* patternList = findPatternList(root);

	while (*source != '\0')
	{
		if (!patternList)
			return false;

		Pattern* matchedPattern = NULL;
		unsigned int matchLength = 0;

		// find the pattern with the longest match-length
		for (ThIterator<Pattern> pattern(*patternList);  pattern;  pattern.next())
		{
			const unsigned int length = validatePattern(pattern, source);

			if (length > matchLength)
			{
				matchedPattern = pattern;
				matchLength    = length;
			}
		}

		// no pattern matched data
		if (!matchedPattern)
		{
			Error->write("Tokenizer", "No pattern matched data in line %u.", m_line);
			return false;
		}

		if (matchedPattern->m_lineBreak)
			m_line++;

		if (matchedPattern->m_discardToken)
			token = NULL;
		else
		{
			if (token && matchedPattern->m_mergePrev && token->m_symbol == matchedPattern->m_symbol)
			{
				if (!matchedPattern->m_discardData)
				{
					if (matchedPattern->m_substitute)
						token->m_data.append(matchedPattern->m_substitute);
					else
					{
						buffer.copy(source, matchLength);
						token->m_data.append(buffer);
					}
				}
			}
			else
			{
				token = new ThToken(matchedPattern->m_symbol, m_line);

				if (!matchedPattern->m_discardData)
				{
					if (matchedPattern->m_substitute)
						token->m_data = matchedPattern->m_substitute;
					else
						token->m_data.copy(source, matchLength);
				}

				target.attachLast(token);
			}
		}

		if (matchedPattern->m_branchList != THSYM_INVALID)
			patternList = findPatternList(matchedPattern->m_branchList);

		// advance data past matched pattern
		source += matchLength;
	}

	if (patternList->m_symbol != root)
	{
		Error->write("Tokenizer", "Tokenization did not end in root pattern list '%s'.", m_symbolTable->translateSymbol(root));
		return false;
	}

	return true;
}

// ThTokenizer interface attributes ---------------------------------------------------------------

const IThSymbolTable* ThTokenizer::getSymbolTable(void) const
{
	return m_symbolTable;
}

// ThTokenizer methods ----------------------------------------------------------------------------

bool ThTokenizer::compilePattern(Pattern* pattern, const char* data)
{
	data = compilePatternFlags(pattern, data);
	if (!data)
		return false;

	while (*data != '\0')
	{
		ThPtr<PatternNode> patternNode = new PatternNode();

		patternNode->fillRange(0, 256, false);

		data = compilePatternNode(patternNode, data);
		if (!data)
			return false;

		pattern->m_nodes.attachLast(patternNode.detach());
	}

	return pattern->m_nodes.getCount() > 0;
}

const char* ThTokenizer::compilePatternFlags(Pattern* pattern, const char* data)
{
	if (*data != '|')
		return data;

	data++;

	for (;;)
	{
		switch (*data)
		{
			case 'm':
			{
				// merge this token with previous of same type

				pattern->m_mergePrev = true;
				data++;

				continue;
			}

			case 'n':
			{
				// token contains a line break

				pattern->m_lineBreak = true;
				data++;

				continue;
			}

			case 't':
			{
				// do not save matched data in this token

				pattern->m_discardData = true;
				data++;

				continue;
			}

			case 'd':
			{
				// do not generate a token

				pattern->m_discardToken = true;
				data++;

				continue;
			}

			case '|':
			{
				// end of pattern flags section

				data++;
				return data;
			}

			default:
			{
				Error->write("Tokenizer", "Invalid character '%c' in flag section for '%s'.", *data, m_symbolTable->translateSymbol(pattern->m_symbol));
				return NULL;
			}
		}
	}
}

const char* ThTokenizer::compilePatternNode(PatternNode* patternNode, const char* data)
{
	bool inverse = false;

	for (;;)
	{
		switch (*data)
		{
			case '!':
			{
				// match any character not specified

				patternNode->fillRange(1, 255, true);

				inverse = true;
				data++;

				if (*data == '\0')
				{
					Error->write("Tokenizer", "Unexpected end of pattern '%s' after inversion prefix.", m_symbolTable->translateSymbol(m_pattern->m_symbol));
					return NULL;
				}

				continue;
			}

			case '+':
			{
				// character may be repeated 0 - n times

				patternNode->m_repeated = true;
				data++;

				if (*data == '\0')
				{
					Error->write("Tokenizer", "Unexpected end of pattern '%s' after repeat prefix.", m_symbolTable->translateSymbol(m_pattern->m_symbol));
					return NULL;
				}

				continue;
			}
		}

		break;
	}

	switch (*data)
	{
		case '?':
		{
			// match any character

			if (inverse)
			{
				Error->write("Tokenizer", "Wildcard '%c' in pattern '%s' specified as repeated.", *data, m_symbolTable->translateSymbol(m_pattern->m_symbol));
				return NULL;
			}

			if (patternNode->m_repeated)
			{
				Error->write("Tokenizer", "Wildcard '%c' in pattern '%s' specified as inverted.", *data, m_symbolTable->translateSymbol(m_pattern->m_symbol));
				return NULL;
			}

			patternNode->fillRange(1, 255, true);
			data++;

			return data;
		}

		case '*':
		{
			// match any number of any character

			// disallow repeated wildcard
			if (patternNode->m_repeated)
			{
				Error->write("Tokenizer", "Wildcard '%c', in pattern '%s' specified as repeated.", *data, m_symbolTable->translateSymbol(m_pattern->m_symbol));
				return NULL;
			}

			// disallow inverted wildcard
			if (inverse)
			{
				Error->write("Tokenizer", "Wildcard '%c' in pattern '%s' specified as inverted.", *data, m_symbolTable->translateSymbol(m_pattern->m_symbol));
				return NULL;
			}

			patternNode->m_fallback = true;
			data++;

			if (*data == '\0')
			{
				Error->write("Tokenizer", "Wildcard '%c' at the end of pattern '%s'.", *(data - 1), m_symbolTable->translateSymbol(m_pattern->m_symbol));
				return NULL;
			}

			return data;
		}

		case '\\':
		{
			// match a single escape-sequence character

			data++;

			const char c = convertEscapeCharacter(*data);
			if (c == '\0')
			{
				Error->write("Tokenizer", "Null character specified in pattern '%s'.", m_symbolTable->translateSymbol(m_pattern->m_symbol));
				return NULL;
			}

			patternNode->fillRange(c, 1, inverse ? false : true);
			data++;

			return data;
		}

		case '[':
		{
			// match a set of characters

			data++;

			while (*data != ']')
			{
				char c;

				switch (*data)
				{
					case '\\':
					{
						data++;

						c = convertEscapeCharacter(*data);
						if (c == '\0')
						{
							Error->write("Tokenizer", "Null character specified in pattern '%s'.", m_symbolTable->translateSymbol(m_pattern->m_symbol));
							return NULL;
						}

						data++;
						break;
					}

					case '\0':
					{
						Error->write("Tokenizer", "Unexpected end of set specifier in pattern '%s'.", m_symbolTable->translateSymbol(m_pattern->m_symbol));
						return NULL;
					}

					default:
					{
						c = *data;

						data++;
						break;
					}
				}

				if (*data == '-')
				{
					// match a range of characters

					data++;

					char e;

					switch (*data)
					{
						case '\\':
						{
							data++;

							e = convertEscapeCharacter(*data);
							if (e == '\0')
							{
								Error->write("Tokenizer", "Null character specified in pattern '%s'.", m_symbolTable->translateSymbol(m_pattern->m_symbol));
								return NULL;
							}

							data++;
							break;
						}

						case ']':
						case '\0':
						{
							Error->write("Tokenizer", "Unexpected end of range specifier in pattern '%s'.", m_symbolTable->translateSymbol(m_pattern->m_symbol));
							return NULL;
						}

						default:
						{
							e = *data;

							data++;
							break;
						}
					}

					// invert range if needed
					if (c > e)
					{
						const char t = c;

						c = e;
						e = t;
					}

					patternNode->fillRange(c, e - c + 1, inverse ? false : true);
				}
				else
					patternNode->fillRange(c, 1, inverse ? false : true);
			}

			data++;
			return data;
		}

		case '\0':
		{
			return data;
		}

		default:
		{
			// match a single character

			patternNode->fillRange(*data, 1, inverse ? false : true);
			data++;

			return data;
		}
	}
}

char ThTokenizer::convertEscapeCharacter(char c)
{
	switch (c)
	{
		case 'n':
			return '\n';
		case 'r':
			return '\r';
		case 't':
			return '\t';
		case 'b':
			return '\b';
		case '\0':
			return '\0';
		default:
			return c;
	}
}

unsigned int ThTokenizer::validatePattern(Pattern* pattern, const char* data)
{
	PatternNode* patternNode  = NULL;
	PatternNode* fallbackNode = NULL;

	unsigned int length = 0;

	patternNode = pattern->m_nodes.getFirst();

	for (;;)
	{
		if (!patternNode)
			return length;

		// fail if data is too short
		if (data[length] == '\0')
		{
			if (!patternNode->m_repeated)
				return 0;

			if (patternNode = patternNode->getNext())
				return 0;

			continue;
		}

		// store fallback node
		if (patternNode->m_fallback)
			fallbackNode = patternNode->getNext();

		if (patternNode->validate(data[length]))
		{
			if (!patternNode->m_repeated)
				patternNode = patternNode->getNext();
		}
		else
		{
			if (fallbackNode)
			{
				// check if following node matches
				if (fallbackNode->validate(data[length]))
				{
					// advance to node following fallback node
					patternNode  = fallbackNode->getNext();
					fallbackNode = NULL;
				}
			}
			else
			{
				// pattern does not match
				if (!patternNode->m_repeated)
					return 0;

				patternNode = patternNode->getNext();
				continue;
			}
		}

		length++;
	}
}

ThTokenizer::PatternList* ThTokenizer::findPatternList(unsigned int symbol)
{
	for (ThIterator<PatternList> patternList(m_patternLists);  patternList;  patternList.next())
	{
		if (patternList->m_symbol == symbol)
			return patternList;
	}

	return NULL;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThTokenizer::PatternNode constructors ----------------------------------------------------------

ThTokenizer::PatternNode::PatternNode(void):
	m_repeated(false),
	m_fallback(false)
{
}

// ThTokenizer::PatternNode methods ---------------------------------------------------------------

void ThTokenizer::PatternNode::fillRange(unsigned int start, unsigned int length, bool value)
{
	memset(m_table + start, value, length);
}

bool ThTokenizer::PatternNode::validate(unsigned char c) const
{
	return m_table[c];
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThTokenizer::Pattern constructors --------------------------------------------------------------

ThTokenizer::Pattern::Pattern(unsigned int symbol, const char* substitute, unsigned int branchList):
	m_symbol(symbol),
	m_substitute(substitute),
	m_mergePrev(false),
	m_lineBreak(false),
	m_discardData(false),
	m_discardToken(false),
	m_branchList(branchList)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThTokenizer::PatternList constructors ----------------------------------------------------------

ThTokenizer::PatternList::PatternList(unsigned int symbol):
	m_symbol(symbol)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThAnalyzer constructors ------------------------------------------------------------------------

ThAnalyzer::ThAnalyzer(void):
	m_error(false)
{
}

ThAnalyzer::~ThAnalyzer(void)
{
	close();
}

// ThAnalyzer methods -----------------------------------------------------------------------------

bool ThAnalyzer::open(IThSymbolTable* symbolTable)
{
	m_externalSymbols = symbolTable;

	if (!m_internalSymbols)
	{
		// setup shared internal symbol table

		m_internalSymbols = IThSymbolTable::createInstance();

		m_internalSymbols->registerSymbol(SYM_ROOT, "root");
		m_internalSymbols->registerSymbol(SYM_DISCARD, "discard");
		m_internalSymbols->registerSymbol(SYM_STATIC, "static");
		m_internalSymbols->registerSymbol(SYM_ABSOLUTE, "absolute");
		m_internalSymbols->registerSymbol(SYM_RECURSIVE, "recursive");
		m_internalSymbols->registerSymbol(SYM_PRESCAN, "pre-scan");
		m_internalSymbols->registerSymbol(SYM_REQUIRED, "required");
		m_internalSymbols->registerSymbol(SYM_OPTIONAL, "optional");
	}

	if (!m_tokenizer)
	{
		// setup shared tokenizer

		m_tokenizer = IThTokenizer::createInstance(m_internalSymbols);

		// root list
		m_tokenizer->registerPattern(SYM_ROOT, SYM_DISCARD, "|d|\\[", NULL, SYM_STATIC);
		m_tokenizer->registerPattern(SYM_ROOT, SYM_DISCARD, "|d|{", NULL, SYM_ABSOLUTE);
		m_tokenizer->registerPattern(SYM_ROOT, SYM_DISCARD, "|d|<", NULL, SYM_RECURSIVE);
		m_tokenizer->registerPattern(SYM_ROOT, SYM_DISCARD, "|d|(", NULL, SYM_PRESCAN);
		m_tokenizer->registerPattern(SYM_ROOT, SYM_REQUIRED, "|t|&\\[", NULL, SYM_STATIC);
		m_tokenizer->registerPattern(SYM_ROOT, SYM_REQUIRED, "|t|&{", NULL, SYM_ABSOLUTE);
		m_tokenizer->registerPattern(SYM_ROOT, SYM_REQUIRED, "|t|&<", NULL, SYM_RECURSIVE);
		m_tokenizer->registerPattern(SYM_ROOT, SYM_REQUIRED, "|t|&(", NULL, SYM_PRESCAN);
		m_tokenizer->registerPattern(SYM_ROOT, SYM_OPTIONAL, "|t|\\+\\[", NULL, SYM_STATIC);
		m_tokenizer->registerPattern(SYM_ROOT, SYM_OPTIONAL, "|t|\\+{", NULL, SYM_ABSOLUTE);
		m_tokenizer->registerPattern(SYM_ROOT, SYM_OPTIONAL, "|t|\\+<", NULL, SYM_RECURSIVE);
		m_tokenizer->registerPattern(SYM_ROOT, SYM_DISCARD, "|d|+[ \\t]");

		// static rule list
		m_tokenizer->registerPattern(SYM_STATIC, SYM_STATIC, "|m|\\\\\\]", "]");
		m_tokenizer->registerPattern(SYM_STATIC, SYM_STATIC, "|m|!]");
		m_tokenizer->registerPattern(SYM_STATIC, SYM_DISCARD, "|d|]", NULL, SYM_ROOT);

		// absolute rule list
		m_tokenizer->registerPattern(SYM_ABSOLUTE, SYM_ABSOLUTE, "|m|\\\\}", "}");
		m_tokenizer->registerPattern(SYM_ABSOLUTE, SYM_ABSOLUTE, "|m|!}");
		m_tokenizer->registerPattern(SYM_ABSOLUTE, SYM_DISCARD, "|d|}", NULL, SYM_ROOT);

		// recursive rule list
		m_tokenizer->registerPattern(SYM_RECURSIVE, SYM_RECURSIVE, "|m|\\\\>", ">");
		m_tokenizer->registerPattern(SYM_RECURSIVE, SYM_RECURSIVE, "|m|!>");
		m_tokenizer->registerPattern(SYM_RECURSIVE, SYM_DISCARD, "|d|>", NULL, SYM_ROOT);

		// pre-scan rule list
		m_tokenizer->registerPattern(SYM_PRESCAN, SYM_PRESCAN, "|m|\\\\)", ")");
		m_tokenizer->registerPattern(SYM_PRESCAN, SYM_PRESCAN, "|m|!)");
		m_tokenizer->registerPattern(SYM_PRESCAN, SYM_DISCARD, "|d|)", NULL, SYM_ROOT);
	}

	return true;
}

void ThAnalyzer::close(void)
{
	m_externalSymbols = NULL;

	m_grammarLists.release();

	m_error = false;
}

// ThAnalyzer interface methods -------------------------------------------------------------------

bool ThAnalyzer::registerTerminator(unsigned int symbol)
{
	if (!m_externalSymbols->translateSymbol(symbol))
	{
		Error->write("Analyzer", "Symbol %u for terminator does not exist.", symbol);
		return false;
	}

	const unsigned int count = m_terminators.getCount();

	m_terminators.resize(count + 1);
	m_terminators[count] = symbol;

	return true;
}

bool ThAnalyzer::registerBalancedPair(unsigned int openingSymbol, unsigned int closingSymbol)
{
	if (!m_externalSymbols->translateSymbol(openingSymbol))
	{
		Error->write("Analyzer", "Opening symbol %u for balanced pair does not exist.", openingSymbol);
		return false;
	}

	if (!m_externalSymbols->translateSymbol(closingSymbol))
	{
		Error->write("Analyzer", "Closing symbol %u for balanced pair does not exist.", closingSymbol);
		return false;
	}

	const unsigned int count = m_balancedPairs.getCount();

	m_balancedPairs.resize(count + 2);
	m_balancedPairs[count + 0] = openingSymbol;
	m_balancedPairs[count + 1] = closingSymbol;

	return true;
}

bool ThAnalyzer::registerGrammar(unsigned int list, unsigned int symbol, const char* data, unsigned int level)
{
	if (!m_externalSymbols->translateSymbol(symbol))
	{
		Error->write("Analyzer", "Symbol %u for grammar rule '%s' does not exist.", symbol, data);
		return false;
	}

	ThPtr<Grammar> grammar = new Grammar(symbol);

	if (!compileGrammarNodes(grammar, data))
		return false;

	grammar->m_level = level;

	GrammarList* grammarList = findGrammarList(list);
	if (!grammarList)
	{
		grammarList = new GrammarList(list);
		m_grammarLists.attachLast(grammarList);
	}

	// calculate level count and start level
	{
		const unsigned int levels = level + 1;

		if (grammarList->m_levels < levels)
		{
			grammarList->m_levels = levels;

			if (levels > 1)
				grammarList->m_first = 1;
		}
	}

	grammarList->attachLast(grammar.detach());
	return true;
}

bool ThAnalyzer::analyze(ThTokenList& target, unsigned int root, const ThTokenList& source)
{
	GrammarList* grammarList = findGrammarList(root);
	if (!grammarList)
		return false;

	// NOTE: workaround for gcc bug!
	ThConstIterator<ThToken> token = source.iterate();

	if (!compileGrammarList(grammarList, grammarList->m_first, target, token))
	{
		m_error = false;

		return false;
	}

	return true;
}

// ThAnalyzer interface attributes ----------------------------------------------------------------

const IThSymbolTable* ThAnalyzer::getSymbolTable(void) const
{
	return m_externalSymbols;
}

// ThAnalyzer methods -----------------------------------------------------------------------------

bool ThAnalyzer::compileGrammarNodes(Grammar* grammar, const char* data)
{
	ThTokenList tokenList;

	if (!m_tokenizer->tokenize(tokenList, SYM_ROOT, data))
	{
		Error->write("Analyzer", "Could not compile grammar rule '%s'.", m_externalSymbols->translateSymbol(grammar->m_symbol));
		return false;
	}

	for (ThConstIterator<ThToken> token(tokenList);  token;  token.next())
	{
		bool required = false;
		bool optional = false;

		if (token->m_symbol == SYM_REQUIRED)
		{
			required = true;
			token.next();
		}

		if (token->m_symbol == SYM_OPTIONAL)
		{
			optional = true;
			token.next();
		}

		const unsigned int symbol = m_externalSymbols->translateSymbol(token->m_data);
		if (symbol == THSYM_INVALID)
		{
			Error->write("Analyzer", "Symbol '%s' in grammar rule '%s' does not exist.", token->m_data.getData(), m_externalSymbols->translateSymbol(grammar->m_symbol));
			return false;
		}

		grammar->m_nodes.attachLast(new GrammarNode(symbol, token->m_symbol, required, optional));
	}

	return true;
}

bool ThAnalyzer::compileGrammarList(GrammarList* grammarList, unsigned int level, ThList<ThToken>& target, ThConstIterator<ThToken>& source)
{
	const ThToken* firstToken = source;

	ThPtr<ThToken> token = new ThToken(THSYM_INVALID, source->m_line);

	for (;;)
	{
		for (ThIterator<Grammar> grammar(*grammarList);  grammar;  grammar.next())
		{
			if (grammar->m_level != level)
				continue;

			if (compileGrammar(grammar, grammarList->m_symbol, token->m_children, source))
			{
				token->m_symbol = grammar->m_symbol;

				target.attachLast(token.detach());
				return true;
			}

			if (m_error)
				return false;

			token->m_children.release();

			source = firstToken;
		}

		if (!level)
			break;

		level++;

		if (level == grammarList->m_levels)
			level = 0;
	}

	return false;
}

bool ThAnalyzer::compileGrammar(Grammar* grammar, unsigned int list, ThList<ThToken>& target, ThConstIterator<ThToken>& source)
{
	unsigned int index = 0;

	for (ThIterator<GrammarNode> grammarNode = grammar->m_nodes.iterate();  grammarNode;  grammarNode.next())
	{
		if (!source)
			return false;

		if (grammarNode->m_type == SYM_PRESCAN)
		{
			if (!examineGrammar(grammar, grammarNode->m_symbol, source))
				return false;

			continue;
		}

		if (grammarNode->m_type == SYM_RECURSIVE)
		{
			GrammarList* grammarList = findGrammarList(grammarNode->m_symbol);
			if (!grammarList)
				return false;

			unsigned int level = 1;

			if (grammar->m_level && grammarNode->m_symbol == list)
			{
				level = grammar->m_level;

				if (!index)
				{
					if (grammarList->m_levels == 1)
					{
						Error->write("Analyzer", "Grammar rule '%s' self-recursive on first node.", m_externalSymbols->translateSymbol(grammar->m_symbol));
						m_error = true;

						return false;
					}

					level++;
				}
			}

			if (level >= grammarList->m_levels)
				level = 0;

			if (!compileGrammarList(grammarList, level, target, source))
			{
				if (grammarNode->m_required)
				{
					Error->write("Analyzer", "Line (%u): Found '%s', expected '%s'.", source->m_line, m_externalSymbols->translateSymbol(source->m_symbol), m_externalSymbols->translateSymbol(grammarNode->m_symbol));
					m_error = true;
				}
				else if (grammarNode->m_optional)
				{
					target.attachLast(new ThToken(grammarNode->m_symbol, source->m_line));
					continue;
				}

				return false;
			}
		}
		else
		{
			if (grammarNode->m_symbol != source->m_symbol)
			{
				if (grammarNode->m_required)
				{
					Error->write("Analyzer", "Line (%u): Found '%s', expected '%s'.", source->m_line, m_externalSymbols->translateSymbol(source->m_symbol), m_externalSymbols->translateSymbol(grammarNode->m_symbol));
					m_error = true;
				}
				else if (grammarNode->m_optional)
				{
					if (grammarNode->m_type == SYM_ABSOLUTE)
						target.attachLast(new ThToken(grammarNode->m_symbol, source->m_line));

					continue;
				}

				return false;
			}

			if (grammarNode->m_type == SYM_ABSOLUTE)
				target.attachLast(new ThToken(*source));

			source.next();
		}

		index++;
	}

	return true;
}

bool ThAnalyzer::examineGrammar(Grammar* grammar, unsigned int symbol, const ThToken* source)
{
	int level = 0;

	while (source)
	{
		if (source->m_symbol == symbol)
			return true;

		if (!level)
		{
			for (unsigned int i = 0;  i < m_terminators.getCount();  i++)
			{
				if (source->m_symbol == m_terminators[i])
					return false;
			}
		}

		for (unsigned int i = 0;  i < m_balancedPairs.getCount();  i += 2)
		{
			if (source->m_symbol == m_balancedPairs[i + 0])
			{
				level++;
				break;
			}

			if (source->m_symbol == m_balancedPairs[i + 1])
			{
				level--;
				break;
			}
		}

		if (level < 0)
			return false;

		source = source->getNext();
	}

	return false;
}

ThAnalyzer::GrammarList* ThAnalyzer::findGrammarList(unsigned int symbol)
{
	for (ThIterator<GrammarList> grammarList(m_grammarLists);  grammarList;  grammarList.next())
	{
		if (grammarList->m_symbol == symbol)
			return grammarList;
	}

	return NULL;
}

// ThAnalyzer static data -------------------------------------------------------------------------

ThPtr<IThTokenizer> ThAnalyzer::m_tokenizer;

ThPtr<IThSymbolTable> ThAnalyzer::m_internalSymbols;

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThAnalyzer::GrammarNode constructors -----------------------------------------------------------

ThAnalyzer::GrammarNode::GrammarNode(unsigned int symbol, unsigned int type, bool required, bool optional):
	m_symbol(symbol),
	m_type(type),
	m_required(required),
	m_optional(optional)
{
}


///////////////////////////////////////////////////////////////////////////////////////////////////

// ThAnalyzer::Grammar constructors ---------------------------------------------------------------

ThAnalyzer::Grammar::Grammar(unsigned int symbol):
	m_symbol(symbol)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThAnalyzer::GrammarList constructors -----------------------------------------------------------

ThAnalyzer::GrammarList::GrammarList(unsigned int symbol):
	m_symbol(symbol),
	m_levels(0),
	m_first(0)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThSyntax constructors --------------------------------------------------------------------------

ThSyntax::ThSyntax(void)
{
	m_externalSymbols = NULL;
}

ThSyntax::~ThSyntax(void)
{
	close();
}

// ThSyntax methods -------------------------------------------------------------------------------

bool ThSyntax::open(IThSymbolTable* symbolTable, const char* script)
{
	close();

	if (!m_internalSymbols)
	{
		m_internalSymbols = IThSymbolTable::createInstance();

		m_internalSymbols->registerSymbol(SYM_ROOT, "root");
		m_internalSymbols->registerSymbol(SYM_COMMANDLIST, "command list");
		m_internalSymbols->registerSymbol(SYM_COMMAND, "command");
		m_internalSymbols->registerSymbol(SYM_DISCARD, "discard");
		m_internalSymbols->registerSymbol(SYM_NEWLINE, "newline");
		m_internalSymbols->registerSymbol(SYM_SYMBOL, "symbol");
		m_internalSymbols->registerSymbol(SYM_PATTERN, "pattern");
		m_internalSymbols->registerSymbol(SYM_TERMINATOR, "terminator");
		m_internalSymbols->registerSymbol(SYM_BALANCED_PAIR, "balanced pair");
		m_internalSymbols->registerSymbol(SYM_GRAMMAR, "grammar");
		m_internalSymbols->registerSymbol(SYM_STRING, "string");
		m_internalSymbols->registerSymbol(SYM_INTEGER, "integer");
	}

	if (!m_internalTokenizer)
	{
		m_internalTokenizer = IThTokenizer::createInstance(m_internalSymbols);

		m_internalTokenizer->registerPattern(SYM_ROOT, SYM_SYMBOL, "|t|symbol");
		m_internalTokenizer->registerPattern(SYM_ROOT, SYM_PATTERN, "|t|pattern");
		m_internalTokenizer->registerPattern(SYM_ROOT, SYM_TERMINATOR, "|t|terminator");
		m_internalTokenizer->registerPattern(SYM_ROOT, SYM_BALANCED_PAIR, "|t|balanced_pair");
		m_internalTokenizer->registerPattern(SYM_ROOT, SYM_GRAMMAR, "|t|grammar");
		m_internalTokenizer->registerPattern(SYM_ROOT, SYM_DISCARD, "|d|+[ \\t]");
		m_internalTokenizer->registerPattern(SYM_ROOT, SYM_DISCARD, "|d|#", NULL, SYM_DISCARD);
		m_internalTokenizer->registerPattern(SYM_ROOT, SYM_NEWLINE, "|t|\\n");
		m_internalTokenizer->registerPattern(SYM_ROOT, SYM_STRING, "[a-zA-Z_]+[a-zA-Z0-9_]");
		m_internalTokenizer->registerPattern(SYM_ROOT, SYM_INTEGER, "[0-9]+[0-9]");
		m_internalTokenizer->registerPattern(SYM_ROOT, SYM_STRING, "|m|\"", "", SYM_STRING);

		m_internalTokenizer->registerPattern(SYM_DISCARD, SYM_DISCARD, "|d|+![\\n]");
		m_internalTokenizer->registerPattern(SYM_DISCARD, SYM_DISCARD, "|tn|[\\n]", NULL, SYM_ROOT);

		m_internalTokenizer->registerPattern(SYM_STRING, SYM_STRING, "|m|\\\\\"", "\"");
		m_internalTokenizer->registerPattern(SYM_STRING, SYM_STRING, "|m|+![\\n\\\"]");
		m_internalTokenizer->registerPattern(SYM_STRING, SYM_DISCARD, "|d|\"", NULL, SYM_ROOT);
	}

	if (!m_internalAnalyzer)
	{
		m_internalAnalyzer = IThAnalyzer::createInstance(m_internalSymbols);

		m_internalAnalyzer->registerGrammar(SYM_ROOT, SYM_ROOT, "<command list>");

		m_internalAnalyzer->registerGrammar(SYM_COMMANDLIST, SYM_COMMANDLIST, "<command> <command list>");
		m_internalAnalyzer->registerGrammar(SYM_COMMANDLIST, SYM_COMMANDLIST, "<command>");

		m_internalAnalyzer->registerGrammar(SYM_COMMAND, SYM_SYMBOL, "[symbol] {string} [newline]");
		m_internalAnalyzer->registerGrammar(SYM_COMMAND, SYM_PATTERN, "[pattern] {string} {string} {string} +{string} +{string} [newline]");
		m_internalAnalyzer->registerGrammar(SYM_COMMAND, SYM_TERMINATOR, "[terminator] {string} [newline]");
		m_internalAnalyzer->registerGrammar(SYM_COMMAND, SYM_BALANCED_PAIR, "[balanced pair] {string} {string} [newline]");
		m_internalAnalyzer->registerGrammar(SYM_COMMAND, SYM_GRAMMAR, "[grammar] {string} {string} {string} +{integer} [newline]");
	}

	ThTokenList tokenList;

	if (!m_internalTokenizer->tokenize(tokenList, SYM_ROOT, script))
		return false;

	ThTokenList tokenTree;

	if (!m_internalAnalyzer->analyze(tokenTree, SYM_ROOT, tokenList))
		return false;

	m_externalTokenizer = IThTokenizer::createInstance(m_externalSymbols);
	m_externalAnalyzer  = IThAnalyzer::createInstance(m_externalSymbols);

	if (!evaluateCommand(tokenTree.getFirst()))
		return false;

	return true;
}

void ThSyntax::close(void)
{
	m_externalSymbols = NULL;

	m_externalTokenizer.release();
	m_externalAnalyzer.release();
}

// ThSyntax attributes ----------------------------------------------------------------------------

IThSymbolTable* ThSyntax::getSymbolTable(void)
{
	return m_externalSymbols;
}

IThTokenizer* ThSyntax::getTokenizer(void)
{
	return m_externalTokenizer;
}

IThAnalyzer* ThSyntax::getAnalyzer(void)
{
	return m_externalAnalyzer;
}

// ThSyntax methods -------------------------------------------------------------------------------

bool ThSyntax::evaluateCommand(ThToken* root)
{
	switch (root->m_symbol)
	{
		case SYM_COMMANDLIST:
		{
			for (ThIterator<ThToken> token(root->m_children);  token;  token.next())
			{
				if (!evaluateCommand(token))
					return false;
			}
			
			break;
		}
		
		case SYM_SYMBOL:
		{
			ThToken* nameToken = findToken(root, 1);
			
			if (m_externalSymbols->translateSymbol(nameToken->m_data) == THSYM_INVALID)
			{
				Error->write("Syntax", "Line (%u): Symbol '%s' does not exist in symbol table.", nameToken->m_line, nameToken->m_data.getData());
				return false;
			}

			m_foundSymbols.attachFirst(new ThStringItem(nameToken->m_data));
			break;
		}
		
		case SYM_PATTERN:
		{
			ThToken* patternToken = findToken(root, 3);

			const unsigned int listSymbol = translateSymbolToken(findToken(root, 1));
			if (!listSymbol)
				return false;

			const unsigned int nameSymbol = translateSymbolToken(findToken(root, 2));
			if (!nameSymbol)
				return false;
			
			ThToken* substToken = findToken(root, 4);
			
			const char* substitute;
			
			if (substToken)
				substitute = substToken->m_data;
			else
				substitute = NULL;
			
			ThToken* branchToken = findToken(root, 5);

			unsigned int branchSymbol = translateSymbolToken(branchToken);
			if (branchToken && !branchSymbol)
				return false;
			
			if (!m_externalTokenizer->registerPattern(listSymbol, nameSymbol, patternToken->m_data, substitute, branchSymbol))
				return false;

			break;
		}
		
		case SYM_TERMINATOR:
		{
			unsigned int terminatorSymbol = translateSymbolToken(findToken(root, 1));
			if (!terminatorSymbol)
				return false;

			if (!m_externalAnalyzer->registerTerminator(terminatorSymbol))
				return false;

			break;
		}
		
		case SYM_BALANCED_PAIR:
		{
			unsigned int openingSymbol = translateSymbolToken(findToken(root, 1));
			if (!openingSymbol)
				return false;

			unsigned int closingSymbol = translateSymbolToken(findToken(root, 2));
			if (!closingSymbol)
				return false;

			if (!m_externalAnalyzer->registerBalancedPair(openingSymbol, closingSymbol))
				return false;

			break;
		}
		
		case SYM_GRAMMAR:
		{
			const unsigned int listSymbol = translateSymbolToken(findToken(root, 1));
			if (!listSymbol)
				return false;
			
			const unsigned int nameSymbol = translateSymbolToken(findToken(root, 2));
			if (!nameSymbol)
				return false;
				
			ThToken* grammarToken = findToken(root, 3);
			
			const char* grammar = grammarToken->m_data;
			
			ThToken* levelToken = findToken(root, 4);
			
			unsigned int level;
			
			if (levelToken)
				level = levelToken->m_data.convertToInt();
			else
				level = 0;
			
			if (!m_externalAnalyzer->registerGrammar(listSymbol, nameSymbol, grammar, level))
				return false;
			
			break;
		}
	}
	
	return true;
}

ThToken* ThSyntax::findToken(ThToken* token, unsigned int depth)
{
	while (depth--)
	{
		token = token->m_children.getFirst();
		if (!token)
			return NULL;
	}
	
	return token;
}

unsigned int ThSyntax::translateSymbolToken(ThToken* symbolToken)
{
	if (!symbolToken)
		return THSYM_INVALID;
	
	if (!m_foundSymbols.findNoCase(symbolToken->m_data))
	{
		Error->write("Syntax", "Line (%u): Symbol '%s' has not been declared prior to use.", symbolToken->m_line, symbolToken->m_data.getData());
		return false;
	}
	
	return m_externalSymbols->translateSymbol(symbolToken->m_data);
}

// ThSyntax static data ---------------------------------------------------------------------------

ThPtr<IThSymbolTable> ThSyntax::m_internalSymbols;
ThPtr<IThTokenizer> ThSyntax::m_internalTokenizer;
ThPtr<IThAnalyzer> ThSyntax::m_internalAnalyzer;

///////////////////////////////////////////////////////////////////////////////////////////////////
