///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>
#include <shared/ThMath.h>
#include <shared/ThString.h>
#include <shared/ThError.h>
#include <shared/ThStream.h>
#include <shared/ThStorage.h>
#include <shared/ThServer.h>
#include <shared/ThContext.h>
#include <shared/ThDisplay.h>

#include <theresa/ThDisplay.h>

#include <libpng/png.h>

///////////////////////////////////////////////////////////////////////////////////////////////////

static void pngError(png_structp pngFile, png_const_charp error)
{
	Error->display("Display", "libpng reported an error.%s%s", THERESA_NEWLINE, error);
}

static void pngWarning(png_structp pngFile, png_const_charp warning)
{
	Error->write("Display", "libpng reported a warning.%s%s", THERESA_NEWLINE, warning);
}

static void pngReadStream(png_structp pngFile, png_bytep data, png_size_t length)
{
	IThStream* stream = reinterpret_cast<IThStream*>(png_get_io_ptr(pngFile));

	stream->readItems(data, length);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThSprite2 constructors -------------------------------------------------------------------------

ThSprite2::ThSprite2(void)
{
	m_position.set(0.f, 0.f);
	m_size.set(1.f, 1.f);
	m_area.set(0.f, 0.f, 1.f, 1.f);
	m_color.set(1.f, 1.f, 1.f, 1.f);
	m_rotation = 0.f;
}

// ThSprite2 methods ------------------------------------------------------------------------------

void ThSprite2::render(void)
{
	// TODO: implement rotation!

	glColor4fv(m_color);

	const ThVector2 offset(m_size.x * 0.5f, m_size.y * 0.5f);

	glBegin(GL_QUADS);

	glTexCoord2f(m_area.m_pos.x, m_area.m_pos.y);
	glVertex2f(m_position.x - offset.x, m_position.y - offset.y);

	glTexCoord2f(m_area.m_pos.x + m_area.m_size.x, m_area.m_pos.y);
	glVertex2f(m_position.x + offset.x, m_position.y - offset.y);

	glTexCoord2f(m_area.m_pos.x + m_area.m_size.x, m_area.m_pos.y + m_area.m_size.y);
	glVertex2f(m_position.x + offset.x, m_position.y + offset.y);

	glTexCoord2f(m_area.m_pos.x, m_area.m_pos.y + m_area.m_size.y);
	glVertex2f(m_position.x - offset.x, m_position.y + offset.y);

	glEnd();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThSprite3 constructors -------------------------------------------------------------------------

ThSprite3::ThSprite3(void)
{
	m_position.set(0.f, 0.f, -1.f);
	m_size.set(1.f, 1.f);
	m_area.set(0.f, 0.f, 1.f, 1.f);
	m_color.set(1.f, 1.f, 1.f, 1.f);
	m_rotation = 0.f;
}

// ThSprite3 methods ------------------------------------------------------------------------------

void ThSprite3::render(void)
{
	// TODO: implement rotation!

	glColor4fv(m_color);

	const ThVector2 offset(m_size.x * 0.5f, m_size.y * 0.5f);

	glBegin(GL_QUADS);

	glTexCoord2f(m_area.m_pos.x, m_area.m_pos.y);
	glVertex2f(m_position.x - offset.x, m_position.y - offset.y);

	glTexCoord2f(m_area.m_pos.x + m_area.m_size.x, m_area.m_pos.y);
	glVertex2f(m_position.x + offset.x, m_position.y - offset.y);

	glTexCoord2f(m_area.m_pos.x + m_area.m_size.x, m_area.m_pos.y + m_area.m_size.y);
	glVertex2f(m_position.x + offset.x, m_position.y + offset.y);

	glTexCoord2f(m_area.m_pos.x, m_area.m_pos.y + m_area.m_size.y);
	glVertex2f(m_position.x - offset.x, m_position.y + offset.y);

	glEnd();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThDisplayObject constructors -------------------------------------------------------------------

ThDisplayObject::ThDisplayObject(void)
{
	m_visible = true;
}

ThDisplayObject::~ThDisplayObject(void)
{
}

// ThDisplayObject methods ------------------------------------------------------------------------

void ThDisplayObject::show(void)
{
	m_visible = true;
}

void ThDisplayObject::hide(void)
{
	m_visible = false;
}

// ThDisplayObject attributes ---------------------------------------------------------------------

bool ThDisplayObject::isVisible(void) const
{
	return m_visible;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThSurface constructors ------------------------------------------------------------------------

IThSurface::~IThSurface(void)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThTexture constructors ------------------------------------------------------------------------

IThTexture::~IThTexture(void)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThDisplayList constructors --------------------------------------------------------------------

IThDisplayList::~IThDisplayList(void)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThCanvas constructors -------------------------------------------------------------------------

IThCanvas::~IThCanvas(void)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThDisplay constructors ------------------------------------------------------------------------

IThDisplay::~IThDisplay(void)
{
}

// IThDisplay static methods ----------------------------------------------------------------------

IThDisplay* IThDisplay::createInstance(const ThContextMode* mode)
{
	ThPtr<ThDisplay> display = new ThDisplay();

	if (!display->open(mode))
		return NULL;

	return display.detach();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThSurface constructors -------------------------------------------------------------------------

ThSurface::ThSurface(unsigned int width, unsigned int height, unsigned int type, unsigned int format, const char* name):
	m_width(width),
	m_height(height),
	m_type(type),
	m_format(format),
	m_name(name),
	m_hash(0)
{
	if (m_name)
		m_hash = m_name.hashNoCase();

	const unsigned int size = width * height * getPixelSize(type, format);

	m_data = IThStaticBlock::createInstance(size);

	if (memset(m_data->lock(), 0, size))
		m_data->unlock();
}

ThSurface::~ThSurface(void)
{
}

// ThSurface interface methods --------------------------------------------------------------------

void* ThSurface::lock(void)
{
	return m_data->lock();
}

void ThSurface::unlock(void)
{
	m_data->unlock();
}

// ThSurface attributes ---------------------------------------------------------------------------

unsigned int ThSurface::getHash(void) const
{
	return m_hash;
}

// ThSurface interface attributes -----------------------------------------------------------------

unsigned int ThSurface::getWidth(void) const
{
	return m_width;
}

unsigned int ThSurface::getHeight(void) const
{
	return m_height;
}

unsigned int ThSurface::getType(void) const
{
	return m_type;
}

unsigned int ThSurface::getFormat(void) const
{
	return m_format;
}

const char* ThSurface::getName(void) const
{
	return m_name;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThTexture constructors -------------------------------------------------------------------------

ThTexture::ThTexture(unsigned int texture, unsigned int width, unsigned int height, unsigned int format, const char* name):
	m_texture(texture),
	m_width(width),
	m_height(height),
	m_format(format),
	m_name(name),
	m_hash(0)
{
	if (m_name)
		m_hash = m_name.hashNoCase();
}

ThTexture::~ThTexture(void)
{
	glDeleteTextures(1, (GLuint*) &m_texture);
}

// ThTexture interface methods --------------------------------------------------------------------

void ThTexture::apply(void)
{
	glBindTexture(GL_TEXTURE_2D, m_texture);
}

// ThTexture attributes ---------------------------------------------------------------------------

unsigned int ThTexture::getHash(void) const
{
	return m_hash;
}

// ThTexture interface attributes -----------------------------------------------------------------

unsigned int ThTexture::getWidth(void) const
{
	return m_width;
}

unsigned int ThTexture::getHeight(void) const
{
	return m_height;
}

unsigned int ThTexture::getFormat(void) const
{
	return m_format;
}

const char* ThTexture::getName(void) const
{
	return m_name;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThDisplayList constructors ---------------------------------------------------------------------

ThDisplayList::ThDisplayList(unsigned int list):
	m_list(list)
{
}

ThDisplayList::~ThDisplayList(void)
{
	glDeleteLists(m_list, 1);
}

// ThDisplayList methods --------------------------------------------------------------------------

void ThDisplayList::begin(void)
{
	glNewList(m_list, GL_COMPILE);
}

void ThDisplayList::end(void)
{
	glEndList();
}

void ThDisplayList::render(void)
{
	glCallList(m_list);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThTextureCanvas constructors -------------------------------------------------------------------

ThTextureCanvas::ThTextureCanvas(IThTexture* texture):
	m_texture(texture)
{
}

ThTextureCanvas::~ThTextureCanvas(void)
{
}

// ThTextureCanvas methods ------------------------------------------------------------------------

void ThTextureCanvas::apply(void)
{
	glEnable(GL_SCISSOR_TEST);
	glScissor(0, 0, m_texture->getWidth(), m_texture->getHeight());

	glViewport(0, 0, m_texture->getWidth(), m_texture->getHeight());
}

void ThTextureCanvas::update(void)
{
	m_texture->apply();

	glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0, m_texture->getWidth(), m_texture->getHeight());
}

// ThTextureCanvas attributes ---------------------------------------------------------------------

unsigned int ThTextureCanvas::getWidth(void) const
{
	return m_texture->getWidth();
}

unsigned int ThTextureCanvas::getHeight(void) const
{
	return m_texture->getHeight();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThVirtualCanvas constructors -------------------------------------------------------------------

ThVirtualCanvas::ThVirtualCanvas(unsigned int width, unsigned int height):
	m_width(width),
	m_height(height)
{
}

ThVirtualCanvas::~ThVirtualCanvas(void)
{
}

// ThVirtualCanvas methods ------------------------------------------------------------------------

void ThVirtualCanvas::apply(void)
{
	glEnable(GL_SCISSOR_TEST);
	glScissor(0, 0, m_width, m_height);

	glViewport(0, 0, m_width, m_height);
}

void ThVirtualCanvas::update(void)
{
}

// ThVirtualCanvas attributes ---------------------------------------------------------------------

unsigned int ThVirtualCanvas::getWidth(void) const
{
	return m_width;
}

unsigned int ThVirtualCanvas::getHeight(void) const
{
	return m_height;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThContextCanvas constructors -------------------------------------------------------------------

ThContextCanvas::ThContextCanvas(void)
{
	m_context = NULL;
}

ThContextCanvas::~ThContextCanvas(void)
{
	close();
}

// ThContextCanvas methods ------------------------------------------------------------------------

bool ThContextCanvas::open(IThContext* context)
{
	close();

	m_context = context;

	return true;
}

void ThContextCanvas::close(void)
{
	m_context = NULL;
}

// ThContextCanvas interface methods --------------------------------------------------------------

void ThContextCanvas::apply(void)
{
	glDisable(GL_SCISSOR_TEST);

	glViewport(0, 0, m_context->getWidth(), m_context->getHeight());
}

void ThContextCanvas::update(void)
{
	m_context->update();
}

// ThContextCanvas interface attributes -----------------------------------------------------------

unsigned int ThContextCanvas::getWidth(void) const
{
	return m_context->getWidth();
}

unsigned int ThContextCanvas::getHeight(void) const
{
	return m_context->getHeight();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThDisplay constructors -------------------------------------------------------------------------

ThDisplay::ThDisplay(void):
	ThServerObject(THID_DISPLAY)
{
	m_layers.attachFirst(new Layer(THLAYER_SYSTEM));
	m_layers.attachFirst(new Layer(THLAYER_TOPMOST));
	m_layers.attachFirst(new Layer(THLAYER_FOREGROUND));
	m_layers.attachFirst(new Layer(THLAYER_NORMAL));
	m_layers.attachFirst(new Layer(THLAYER_BACKGROUND));
	m_layers.attachFirst(new Layer(THLAYER_CLEAR));
}

ThDisplay::~ThDisplay(void)
{
	close();
}

// ThDisplay methods ------------------------------------------------------------------------------

bool ThDisplay::open(const ThContextMode* mode)
{
	close();

	ThList<ThResolution> resolutions;

	if (!enumerateResolutions(resolutions))
	{
		Error->display("Display", "Unable to enumerate any OpenGL-capable resolutions.");
		return false;
	}

	ThContextMode internalMode;

	// map specified mode to device
	{
		if (mode)
			internalMode = *mode;
		else
		{
			if (!requestContextMode(internalMode))
				return false;
		}

		if (!internalMode.m_resolution.m_windowed)
		{
			const ThResolution* resolution = findClosestResolution(resolutions, internalMode.m_resolution.m_width, internalMode.m_resolution.m_height, internalMode.m_resolution.m_depth);
			if (!resolution)
			{
				resolution = findClosestResolution(resolutions, internalMode.m_resolution.m_width, internalMode.m_resolution.m_height);
				if (!resolution)
				{
					Error->display("Display", "Unable to map requested resolution (%u by %u, %u bits color) to any available OpenGL-capable resolution.", mode->m_resolution.m_width, mode->m_resolution.m_height, mode->m_resolution.m_depth);
					return false;
				}
			}

			internalMode.m_resolution = *resolution;
		}
	}

	// create context
	{
		m_context = IThContext::createInstance(internalMode, getID());
		if (!m_context)
			return false;
	}

	// create defalt canvas
	{
		m_canvas = new ThContextCanvas();
		m_canvases.attachFirst(m_canvas);

		if (!m_canvas->open(m_context))
			return false;

		m_canvas->apply();
	}
	
	// create default texture
	{
		ThPtr<IThSurface> surface = createSurface(64, 64, GL_UNSIGNED_BYTE, GL_RGB);
		if (!surface)
			return false;
			
		ThPtr<IThTexture> texture = createTexture(surface, GL_RGB8, 0, "default");
		if (!texture)
			return false;
	}

	return true;
}

void ThDisplay::close(void)
{
	for (ThIterator<Layer> layer(m_layers);  layer; )
	{
		layer->m_objects.release();
		
		if (layer->m_id > THLAYER_SYSTEM)
			layer.release();
		else
			layer.next();
	}

	m_textures.release();
	
	m_surfaces.release();

	if (m_displayLists.getCount())
	{
		Error->display("Display", "Display list resource leak detected.");
		m_displayLists.release();
	}

	m_canvas.release();

	if (m_canvases.getCount())
	{
		Error->display("Display", "Texture canvas resource leak detected.");
		m_canvases.release();
	}

	m_context.release();
}

// ThDisplay interface object methods -------------------------------------------------------------

IThSurface* ThDisplay::createSurface(const char* fileName)
{
	ThPtr<IThStream> file = Storage->openFile(fileName);
	if (!file)
		return NULL;
		
	return createSurface(file, fileName);
}

IThSurface* ThDisplay::createSurface(unsigned int width, unsigned int height, unsigned int type, unsigned int format, const char* name)
{
	if (name)
	{
		if (IThSurface* surface = findSurface(name))
		{
			if (surface->getWidth() != width || surface->getHeight() != height || surface->getType() != type || surface->getFormat() != format)
			{
				Error->display("Display", "Surface specification differs in cache request.");
				sendMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE);
				
				return NULL;
			}
			
			return surface;
		}
	}
	
	return new ThSurface(width, height, type, format, name);
}

IThSurface* ThDisplay::createSurface(IThStream* stream, const char* name)
{
	// check if file is valid
	{
		unsigned char header[8];

		if (!stream->readItems(header, 8))
			return false;

		if (png_sig_cmp(header, 0, 8))
			return NULL;
	}

	png_structp pngFile;
	png_infop pngInfo;
	png_infop pngEndInfo;

	// set up for image reading
	{
		pngFile = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, pngError, pngWarning);
		if (!pngFile)
			return NULL;

		png_set_read_fn(pngFile, stream, pngReadStream);

		pngInfo = png_create_info_struct(pngFile);
		if (!pngInfo)
		{
			png_destroy_read_struct(&pngFile, NULL, NULL);
			return NULL;
		}

		pngEndInfo = png_create_info_struct(pngFile);
		if (!pngEndInfo)
		{
			png_destroy_read_struct(&pngFile, &pngInfo, NULL);
			return NULL;
		}

		png_set_sig_bytes(pngFile, 8);
	}

	unsigned int width;
	unsigned int height;
	unsigned int type;
	unsigned int channels;
	unsigned int format;

	// read image information
	{
		png_read_png(pngFile, pngInfo, PNG_TRANSFORM_IDENTITY, NULL);

		width  = png_get_image_width(pngFile, pngInfo);
		height = png_get_image_height(pngFile, pngInfo);

		switch (png_get_bit_depth(pngFile, pngInfo))
		{
			case 8:
				type = GL_UNSIGNED_BYTE;
				break;
			case 16:
				type = GL_UNSIGNED_SHORT;
				break;
			default:
				return NULL;
		}

		switch (png_get_color_type(pngFile, pngInfo))
		{
			case PNG_COLOR_TYPE_GRAY:
				format = GL_LUMINANCE;
				break;
			case PNG_COLOR_TYPE_GRAY_ALPHA:
				format = GL_LUMINANCE_ALPHA;
				break;
			case PNG_COLOR_TYPE_RGB:
				format = GL_RGB;
				break;
			case PNG_COLOR_TYPE_RGB_ALPHA:
				format = GL_RGBA;
				break;
			default:
				return NULL;
		}

		channels = getPixelComponentCount(type, format);
	}

	ThPtr<ThSurface> surface;

	// read image data
	{
		surface = new ThSurface(width, height, type, format, name);

		const unsigned int size = png_get_rowbytes(pngFile, pngInfo);

		png_bytepp rows = png_get_rows(pngFile, pngInfo);

		unsigned char* data = (unsigned char*) surface->lock();
		if (!data)
			return NULL;

		for (unsigned int i = 0;  i < height;  i++)
			memcpy(data + i * size, rows[i], size);

		surface->unlock();
	}

	// cleanup library structures
	png_destroy_read_struct(&pngFile, &pngInfo, &pngEndInfo);
	
	m_surfaces.attachFirst(surface);

	return surface.detach();
}

IThSurface* ThDisplay::findSurface(const char* name)
{
	const unsigned int hash = ThString::hashNoCase(name);
	
	return findSurface(hash);
}

IThSurface* ThDisplay::findSurface(unsigned int hash)
{
	for (ThIterator<ThSurface> surface(m_surfaces);  surface;  surface.next())
	{
		if (surface->getHash() == hash)
			return surface;
	}
	
	return NULL;
}

IThTexture* ThDisplay::createTexture(const char* fileName, unsigned int format, unsigned int flags)
{
	ThPtr<IThStream> file = Storage->openFile(fileName);
	if (!file)
		return NULL;
		
	return createTexture(file, format, flags, fileName);
}

IThTexture* ThDisplay::createTexture(IThStream* stream, unsigned int format, unsigned int flags, const char* name)
{
	ThPtr<IThSurface> surface = createSurface(stream);
	if (!surface)
		return NULL;

	return createTexture(surface, format, flags, name);
}

IThTexture* ThDisplay::createTexture(IThSurface* surface, unsigned int format, unsigned int flags, const char* name)
{
	unsigned int width  = surface->getWidth();
	unsigned int height = surface->getHeight();

	// calculate optimal texture size

	unsigned int maxSize;

	glGetIntegerv(GL_MAX_TEXTURE_SIZE, (GLint*) &maxSize);

	unsigned int targetWidth  = getClosestPower(width, maxSize);
	unsigned int targetHeight = getClosestPower(height, maxSize);

	if (name)
	{
		if (IThTexture* texture = findTexture(name))
		{
			if (texture->getWidth() != targetWidth || texture->getHeight() != targetHeight || texture->getFormat() != format)
			{
				Error->display("Display", "Texture specification differs in cache request.");
				sendMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE);
				
				return NULL;
			}
			
			return texture;
		}
	}
	
	void* data = surface->lock();
	if (!data)
		return NULL;

	ThByteBlock buffer;

	// rescale and convert image
	{
		if (targetWidth != width || targetHeight != height || surface->getType() != GL_UNSIGNED_BYTE)
		{
			buffer.allocate(targetWidth * targetHeight * getPixelSize(surface->getType(), surface->getFormat()));

			gluScaleImage(surface->getFormat(), width, height, surface->getType(), data, targetWidth, targetHeight, GL_UNSIGNED_BYTE, buffer);

			data = buffer;

			width  = targetWidth;
			height = targetHeight;
		}
	}

	ThPtr<ThTexture> textureObject;

	// create texture object
	{
		unsigned int texture;

		glGenTextures(1, (GLuint*) &texture);

		textureObject = new ThTexture(texture, width, height, format, name);
	}

	textureObject->apply();

	if (flags & THFLAG(THTEX_MIPMAP))
		gluBuild2DMipmaps(GL_TEXTURE_2D, format, width, height, surface->getFormat(), GL_UNSIGNED_BYTE, data);
	else
	{
		glTexImage2D(GL_TEXTURE_2D, 0, format, width, height, 0, surface->getFormat(), GL_UNSIGNED_BYTE, data);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	}

	surface->unlock();

	if (const unsigned int error = glGetError())
	{
		Error->display("Display", "OpenGL reported an error during texture data loading.%s%s", THERESA_NEWLINE, gluErrorString(error));
		return NULL;
	}

	m_textures.attachFirst(textureObject);

	return textureObject.detach();
}

IThTexture* ThDisplay::findTexture(const char* name)
{
	const unsigned int hash = ThString::hashNoCase(name);
	
	return findTexture(hash);
}

IThTexture* ThDisplay::findTexture(unsigned int hash)
{
	for (ThIterator<ThTexture> texture(m_textures);  texture;  texture.next())
	{
		if (texture->getHash() == hash)
			return texture;
	}
	
	return NULL;
}

IThDisplayList* ThDisplay::createDisplayList(void)
{
	const unsigned int list = glGenLists(1);

	ThPtr<ThDisplayList> displayList = new ThDisplayList(list);
	m_displayLists.attachFirst(displayList);

	return displayList.detach();
}

IThTexture* ThDisplay::createCanvasTexture(unsigned int width, unsigned int height, const char* name)
{
	width  = getClosestPower(width, m_canvas->getWidth());
	height = getClosestPower(height, m_canvas->getHeight());

	ThPtr<IThSurface> surface = createSurface(width, height, GL_UNSIGNED_BYTE, GL_RGB);
	if (!surface)
		return NULL;

	ThPtr<IThTexture> texture = createTexture(surface, GL_RGB, 0, name);
	if (!texture)
		return NULL;

	return texture.detach();
}

IThCanvas* ThDisplay::createTextureCanvas(IThTexture* texture)
{
	ThPtr<ThTextureCanvas> canvas = new ThTextureCanvas(texture);
	
	m_canvases.attachFirst(canvas);
	return canvas.detach();
}

IThCanvas* ThDisplay::createVirtualCanvas(unsigned int width, unsigned int height)
{
	ThPtr<ThVirtualCanvas> canvas = new ThVirtualCanvas(width, height);

	m_canvases.attachFirst(canvas);
	return canvas.detach();
}

// ThDisplay interface object methods -------------------------------------------------------------

bool ThDisplay::registerLayer(unsigned int layerID)
{
	if (findLayer(layerID))
		return false;
	
	m_layers.attachLast(new Layer(layerID));
	return true;
}

bool ThDisplay::registerObject(ThDisplayObject* object, unsigned int layerID)
{
	Layer* layer = findLayer(layerID);
	if (!layer)
		return false;

	layer->m_objects.attachFirst(object);
	return true;
}

void ThDisplay::renderObjects(IThCanvas* target, unsigned int layerID)
{
	if (!target)
		target = m_canvas;

	if (layerID == THLAYER_INVALID)
	{
		for (ThIterator<Layer> layer(m_layers);  layer;  layer.next())
		{
			renderLayer(layer, target);
			
			if (layer->m_id == THLAYER_SYSTEM)
				break;
		}
	}
	else
	{
		Layer* layer = findLayer(layerID);
		if (!layer)
			return;
		
		renderLayer(layer, target);
	}
}

void ThDisplay::releaseObjects(void)
{
	for (ThIterator<Layer> layer(m_layers);  layer;  layer.next())
		layer->m_objects.release();
}

// ThDisplay interface attributes -----------------------------------------------------------------

IThContext* ThDisplay::getContext(void)
{
	return m_context;
}

IThTexture* ThDisplay::getDefaultTexture(void)
{
	return findTexture("default");
}

IThCanvas* ThDisplay::getDefaultCanvas(void)
{
	return m_canvas;
}

// ThDisplay methods ------------------------------------------------------------------------------

ThDisplay::Layer* ThDisplay::findLayer(unsigned int id)
{
	if (id == THLAYER_INVALID)
		return NULL;

	for (ThIterator<Layer> layer(m_layers);  layer;  layer.next())
	{
		if (layer->m_id == id)
			return layer;
	}

	return NULL;
}

void ThDisplay::renderLayer(Layer* layer, IThCanvas* target)
{
	for (ThIterator<ThDisplayObject> object(layer->m_objects);  object; )
	{
		if (object->isVisible())
		{
			if (object->render(target))
				object.next();
			else
				object.release();
		}
		else
			object.next();
	}
}

// ThDisplay callbacks ----------------------------------------------------------------------------

bool ThDisplay::receive(const IThMessage* message)
{
	switch (message->getMessage())
	{
		case THMSG_CONTEXT_RESIZE:
		{
			sendMessage(THMSG_DISPLAY_RESIZE, THID_ANNOUNCE);
			break;
		}
		
		case THMSG_LOCAL_UPDATE:
		{
			const float deltaTime = *reinterpret_cast<const float*>(message->getData());
			
			for (ThIterator<Layer> layer(m_layers);  layer;  layer.next())
			{
				for (ThIterator<ThDisplayObject> object(layer->m_objects);  object; )
				{
					if (object->update(deltaTime))
						object.next();
					else
						object.release();
				}
			}
			
			break;
		}
	}
	
	return ThServerObject::receive(message);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThDisplay::Layer constructors ------------------------------------------------------------------

ThDisplay::Layer::Layer(unsigned int id):
	m_id(id)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////
