///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef SHARED_THSTREAM_H
#define SHARED_THSTREAM_H
///////////////////////////////////////////////////////////////////////////////////////////////////

/*! \class IThStream
 *	The stream interface. Represents a generic seekable data stream.
 */
/*! \fn unsigned int IThStream::read(void* data, unsigned int size)
 *	Reads data into the specified buffer.
 *	\param data [out] The buffer to write the data to.
 *	\param size [in] The number of bytes to read.
 *	\return The number of bytes actually read.
 */
/*! \fn unsigned int IThStream::write(void* data, unsigned int size)
 *	Writes data from the specified buffer.
 *	\param data [out] The buffer to write.
 *	\param size [in] The number of bytes to write.
 *	\return The number of bytes actually written.
 */
/*! \fn bool IThStream::readItem(T& item)
 *	Reads one item of the specified type.
 *	\param item The object to store the data in.
 *	\return \c true if sucessful, or \c false if an error ocurred.
 *	\remarks This method uses blind block-reading to fill the object with data.
 */
/*! \fn bool IThStream::writeItem(T& item)
 *	Writes one item of the specified type.
 *	\param item The object to write.
 *	\return \c true if sucessful, or \c false if an error ocurred.
 */
/*! \fn bool IThStream::readItems(T* items, unsigned int count)
 *	Reads the specified number of items of the specified type.
 *	\param items The objects to store the data in.
 *	\param count The number of objects to read.
 *	\return \c true if sucessful, or \c false if an error ocurred.
 *	\remarks This method uses blind block-reading to fill the objects with data.
 */
/*! \fn bool IThStream::writeItems(T* items, unsigned int count)
 *	Writes the specified number of items of the specified type.
 *	\param items The objects to write.
 *	\param count The number of objects to write.
 *	\return \c true if sucessful, or \c false if an error ocurred.
 */
/*! \fn bool IThStream::isEOF(void) const
 *	\return \c true if the end of the file is reached, or \c false if more data follows.
 */
/*! \fn unsigned int IThStream::getSize(void) const
 *	\return The size, in bytes, of the entire data block.
 */
/*! \fn unsigned int IThStream::getPosition(void) const
 *	\return The current position as an offset, in bytes, counting from the start of the file.
 */
/*! \fn bool IThStream::setPosition(unsigned int position)
 *	Sets the file pointer to the specified position.
 *	\param position [in] The position to set the file pointer to.
 *	\return \c true if sucessful, or \c false if an error ocurred.
 */
class IThStream
{
public:
// constructors
	virtual														~IThStream(void);
// methods
	virtual unsigned int							read(void* data, unsigned int size) = 0;
	virtual unsigned int							write(const void* data, unsigned int size) = 0;
// template methods
	template <typename T>
	inline bool												readItem(T& item)
		{ return read(&item, sizeof(T)) == sizeof(T); }
	template <typename T>
	inline bool												writeItem(T& item)
		{ return write(&item, sizeof(T)) == sizeof(T); }
	template <typename T>
	inline bool												readItems(T* items, unsigned int count)
		{ return read(items, sizeof(T) * count) == sizeof(T) * count; }
	template <typename T>
	inline bool												writeItems(T* items, unsigned int count)
		{ return write(items, sizeof(T) * count) == sizeof(T) * count; }
// attributes
	virtual bool											isEOF(void) const = 0;
	virtual bool											isReadable(void) const = 0;
	virtual bool											isWritable(void) const = 0;
	virtual unsigned int							getSize(void) const = 0;
	virtual unsigned int							getPosition(void) const = 0;
	virtual bool											setPosition(unsigned int position) = 0;
// static methods
	static IThStream*									createInstance(unsigned int size = 0);
	static IThStream*									createInstance(IThStream* inner);
};

///////////////////////////////////////////////////////////////////////////////////////////////////
#endif /* SHARED_THSTREAM_H */
///////////////////////////////////////////////////////////////////////////////////////////////////
