///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef SHARED_THCONTEXT_H
#define SHARED_THCONTEXT_H
///////////////////////////////////////////////////////////////////////////////////////////////////

#if THERESA_WGL
	#include <windows.h>
	#include <GL/gl.h>
	#include <GL/glu.h>
	#if THERESA_CYGWIN
		#include <GL/glext.h>
	#endif
#elif THERESA_AGL
	#include <OpenGL/gl.h>
	#include <OpenGL/glu.h>
#elif THERESA_GLX
	#include <GL/gl.h>
	#include <GL/glu.h>
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////

/*! Calculates the power-of-two closest to the specified value, which is lesser than the specified maximum.
 *	\param value [in]The source value with which to match.
 *	\param maximum [in]The maximum allowed return value.
 *	\return The best value found.
 */
unsigned int getClosestPower(unsigned int value, unsigned int maximum);

/*! Calculates the size, in bytes, of the pixels described by the specified OpenGL type/format pair.
 *	\param type [in]The OpenGL pixel format's type.
 *	\param format [in]The OpenGL pixel format.
 *	\return The size, in bytes, of the pixels described by the format/type pair described.
 */
unsigned int getPixelSize(unsigned int type, unsigned int format);

/*! Calculates the number of components in the pixels described by the specified OpenGL type/format pair.
 *	\param type [in]The OpenGL pixel format's type.
 *	\param format [in]The OpenGL pixel format.
 *	\return The number of components in the pixels described by the format/type pair described.
 */
unsigned int getPixelComponentCount(unsigned int type, unsigned int format);

/*! Builds a list of available full-screen resolutions.
 *	\param resolutions [out]The list which is to receive the list of resolutions.
 *	\return \c true if the enumeration succeeded, or \c false if an error occurred.
 */
bool enumerateResolutions(ThList<ThResolution>& resolutions);

/*! Finds an exact match for the specified resolution in the specified list.
 *	\param resolutions [in]The list of resolutions to search.
 *	\param width [in]The requested width, in pixels, of the resolution.
 *	\param height [in]The requested height, in pixels, of the resolution.
 *	\param depth [in]The requested bit depth of the resolution, or zero for the default depth.
 *	\return A pointer to the first item in the list matching the requested properties, or \c NULL if no match could be found.
 */
const ThResolution* findExactResolution(const ThList<ThResolution>& resolutions, unsigned int width, unsigned int height, unsigned int depth = 0);

/*! Finds the closest match for the specified resolution in the specified list.
 *	\param resolutions [in]The list of resolutions to search.
 *	\param width [in]The requested width, in pixels, of the resolution.
 *	\param height [in]The requested height, in pixels, of the resolution.
 *	\param depth [in]The requested bit depth of the resolution, or zero for the default depth.
 *	\return A pointer to the item in the list closest matching the requested properties, or \c NULL if no match could be found.
 */
const ThResolution* findClosestResolution(const ThList<ThResolution>& resolutions, unsigned int width, unsigned int height, unsigned int depth = 0);

/*! Requests the user to select, via appropriate means, the desired set of device settings.
 *	\param mode [out]The object which is to receive the selected settings.
 *	\return \c true if the user made a selection, or \c false if the user aborted the selection.
 */
bool requestContextMode(ThContextMode& mode);

///////////////////////////////////////////////////////////////////////////////////////////////////

/*! \class ThResolution
 *	Screen resolution information class.
 */
class ThResolution : public ThListItem<ThResolution>
{
public:
// constructors
																		ThResolution(void);
																		ThResolution(unsigned int width, unsigned int height, unsigned int depth, bool windowed);
// methods
	void															reset(void);
// attributes
	void															set(unsigned int width, unsigned int height, unsigned int depth, bool windowed);
// data
	unsigned int											m_width;
	unsigned int											m_height;
	unsigned int											m_depth;
	bool															m_windowed;
};

//-------------------------------------------------------------------------------------------------

/*! \class ThContextMode
 *	OpenGL context settings information class.
 */
class ThContextMode
{
public:
// constructors
																		ThContextMode(void);
																		ThContextMode(const ThResolution& resolution, unsigned int depthBits, unsigned int stencilBits, unsigned int textureBits);
// methods
	void															reset(void);
// attributes
	void															set(const ThResolution& resolution, unsigned int depthBits, unsigned int stencilBits, unsigned int textureBits);
// data
	ThResolution											m_resolution;
	unsigned int											m_depthBits;
	unsigned int											m_stencilBits;
	unsigned int											m_textureBits;
};

///////////////////////////////////////////////////////////////////////////////////////////////////

/*! \class IThContext
 *	OpenGL context class. Describes a platform-neutral OpenGL context object.
 */
class IThContext
{
public:
// constructors
	virtual														~IThContext(void);
// methods
	virtual void											apply(void) = 0;
	virtual void											update(void) = 0;
// extension methods
	virtual bool											findExtension(const char* name) const = 0;
	virtual ThEntryPoint							findEntryPoint(const char* name) const = 0;
// attributes
	virtual const char*								getTitle(void) const = 0;
	virtual void											setTitle(const char* title) = 0;
	virtual unsigned int							getWidth(void) const = 0;
	virtual unsigned int							getHeight(void) const = 0;
	virtual const ThContextMode&			getMode(void) const = 0;
// static methods
	static IThContext*								createInstance(const ThContextMode& mode, unsigned int listenerID = THID_INVALID);
};

///////////////////////////////////////////////////////////////////////////////////////////////////
#endif /* SHARED_THCONTEXT_H */
///////////////////////////////////////////////////////////////////////////////////////////////////
