   // -------------------------------------------------------------
// Diffuse and specular shader using Phong specular
// 
// Copyright (c) 2003 Wolfgang F. Engel (wolf@direct3d.net)
// All rights reserved.
// -------------------------------------------------------------

// -------------------------------------------------------------
// variables that are provided by the application
// -------------------------------------------------------------
float4x4 matWorldViewProj;	
float4x4 matWorld;	
float4 vecLightDir;
float4 vecEye;
float fTime;



texture ColorMap;
sampler ColorMapSampler = sampler_state
{
	Texture = <ColorMap>;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;//Linear;   
	AddressU  = Clamp;
	AddressV  = Clamp;
};

texture BumpMap;
sampler BumpMapSampler = sampler_state
{
	Texture = <BumpMap>;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;//Linear;   
	AddressU  = Clamp;
	AddressV  = Clamp;
};

// -------------------------------------------------------------
// Output channels
// -------------------------------------------------------------
struct VS_OUTPUT
{
	float4 Pos  : POSITION;
	float2 Tex0 : TEXCOORD0;
	float2 Tex1 : TEXCOORD1;
	float3 Light : TEXCOORD2;
	float3 View : TEXCOORD3;
};

// -------------------------------------------------------------
// vertex shader function (input channels)
// -------------------------------------------------------------
VS_OUTPUT VS(float4 Pos : POSITION, float2 Tex : TEXCOORD, float3 Normal : NORMAL, float3 Tangent : TANGENT  )
{
	VS_OUTPUT Out = (VS_OUTPUT)0;      

	float4 tempPos = Pos;

	// Do bulge effect (displace vectors)
	float3 dispDir = float3( Normal.x, Normal.y, Normal.z );
	float dispValue = 0.08 * sin(tempPos.y + fTime * 3);

	// Displace vertex by some value
	tempPos.xyz += dispDir.xyz * dispValue;

	Out.Pos = mul(tempPos, matWorldViewProj);	// transform Position

	// compute the 3x3 tranform matrix 
	// to transform from world space to tangent space
	float3x3 worldToTangentSpace;
	worldToTangentSpace[0] = mul(Tangent, matWorld);		// T
	worldToTangentSpace[1] = mul(cross(Tangent, Normal), matWorld);	// B
	worldToTangentSpace[2] = mul(Normal, matWorld);			// N

	Out.Tex0 = Tex.xy;
	Out.Tex1 = Tex.xy;

	Out.Light.xyz = mul(worldToTangentSpace, vecLightDir);		// L
	float3 PosWorld = normalize(mul(tempPos, matWorld));
	float3 Viewer = vecEye - PosWorld;				// V
	Out.View = normalize(mul(worldToTangentSpace, Viewer));

	return Out;
}

// -------------------------------------------------------------
// Pixel Shader (input channels):output channel
// -------------------------------------------------------------
float4 PS(float2 Tex0: TEXCOORD0, float2 Tex1: TEXCOORD1, float3 Light : TEXCOORD2, float3 View : TEXCOORD3) : COLOR
{
	float4 color = tex2D(ColorMapSampler, Tex0);	// Fetch color map
	float3 bumpNormal = 2.0 * (tex2D(BumpMapSampler, Tex1) - 0.5); // Normal map


	// This vectors are normalized in the vertex shader
	float3 LightDir = Light;	// L
	float3 ViewDir = View;		// V

	float4 diff = saturate(dot(bumpNormal, LightDir));	// Diffuse comp.

//	float shadow = saturate(4.0 * diff);			// Compute self-shadowing term

	float3 Reflect = reflect( bumpNormal, LightDir ); //((2 * diff * bumpNormal) - LightDir);  // R

	// Gloss map in color.w restricts spec reflection
	float4 spec = min(pow(saturate(dot(Reflect, ViewDir)), 64), 0 );//color.w);

//	return 0.2 * color + shadow * (color * diff + spec);

	// ambient + diffuse
	return (0.1 * color) + ((color * diff + spec));
}





// -------------------------------------------------------------
// 
// -------------------------------------------------------------
technique TShader
{
	pass P0
	{
		ZEnable			= TRUE;
		ZWriteEnable		= TRUE;
		Lighting		= FALSE;
		FogEnable		= FALSE;
		CullMode		= CCW;
		//DitherEnable		= TRUE;

		// enable alpha blending
		AlphaBlendEnable	= FALSE;
		//SrcBlend		= SRCALPHA;
		//DestBlend		= INVSRCALPHA;

		// set up texture stage states to use the diffuse color
		ColorOp[0]		= MODULATE;
		ColorArg1[0]		= TEXTURE;
		ColorArg2[0]		= DIFFUSE;
		AlphaOp[0]		= SelectArg1;
		AlphaArg1[0]		= TEXTURE;

		ColorOp[1]		= MODULATE;
		ColorArg1[1]		= TEXTURE;
		ColorArg1[2]		= CURRENT;



		Sampler[0] = (ColorMapSampler);		
		Sampler[1] = (BumpMapSampler);		

		// compile shaders
		VertexShader = compile vs_1_1 VS();
		PixelShader  = compile ps_1_3 PS();
	}
}
