 //==============================================================================
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//==============================================================================

//==============================================================================
// File: cMixer.cpp
// Project: Shooting Star
// Author: Jarmo Hekkanen <jarski@2ndpoint.fi>
// Copyrights (c) 2003 2ndPoint ry (www.2ndpoint.fi)
//------------------------------------------------------------------------------
// Revision history
//==============================================================================

//==============================================================================
// Includes
#include "cMixer.hpp"

#include <stdexcept>
#include <SDL.h>
#include "Debug.hpp"
//------------------------------------------------------------------------------
// Namespaces
using namespace ShootingStar;
//==============================================================================


//! Constructor
cMixer::cMixer (void):
mInitialized (false),
mpMusic (NULL)
{
	// Open audio
	if ( Mix_OpenAudio (MIX_DEFAULT_FREQUENCY, MIX_DEFAULT_FORMAT, 2, 512) )
	{
		dbgError () << "Unable to open audio: " << Mix_GetError () << endl;
	}
	else
		mInitialized = true;
};

//! Destructor
cMixer::~cMixer (void)
{
	FreeAllSounds ();
	FreeMusic ();
	Mix_CloseAudio ();
	mInitialized = false;
};

//! Load sound
Uint32 
cMixer::LoadSound (string filename)
{
	if ( !mInitialized || !mSoundsEnabled )
		return 0;

	if ( mSoundIDs[filename] != 0 )
		return mSoundIDs[filename];
	
	string path = mDataDir + filename;
	
	// Load sound
	dbgInfo () << "Loading sound from file " << path << '\n';
	Mix_Chunk *pTemp = Mix_LoadWAV (path.c_str ());
	if ( pTemp == NULL )
	{
		dbgError () << "Unable to load sound: " << Mix_GetError () << endl;
		return 0;
	}		
	
	Uint32 newID = mSounds.size () + 1;
	mSounds.push_back (pTemp);
	mSoundIDs[filename] = newID;
	return newID;
}

//! Free all sounds
void 
cMixer::FreeAllSounds (void)
{
	if ( mSounds.empty () )
		return;
	
	dbgInfo () << "Freeing all sounds\n";
	
	for ( unsigned int i = 0; i < mSounds.size (); i++ )
		Mix_FreeChunk (mSounds[i]);
	mSounds.clear ();
	mSoundIDs.clear ();
}

//! Play sound
int 
cMixer::PlaySound (Uint32 sound, int loop)
{
	if ( !mInitialized )
		return -1;

	if ( sound == 0 )
		return -1;
	
	dbg::assertion (DBG_ASSERTION (sound - 1 < mSounds.size ()));
	return Mix_PlayChannel (-1, mSounds[sound - 1], loop);
}

//! Stop sound
void 
cMixer::StopSound ( int channel, int delay )
{
	if ( !mInitialized || !mSoundsEnabled )
		return;

	dbg::assertion (DBG_ASSERTION (channel >= 0));

	if ( !Mix_Playing (channel) )
	{
		dbgWarning () << "Trying to stop silent channel\n";
		return;
	}
	if ( delay != 0 && false )	// TEMP DISABLED, SDL_mixer is buggy
		Mix_FadeOutChannel( channel, delay);
	else
		Mix_HaltChannel (channel);
}

//! Stop all sounds
void 
cMixer::StopAllSounds (void)
{
	if ( !mInitialized || !mSoundsEnabled )
		return;
	
	Mix_HaltChannel (-1);
}

//! Free music
void 
cMixer::FreeMusic (void)
{
	if ( mpMusic != NULL )
	{
		Mix_HaltMusic ();
		Mix_FreeMusic (mpMusic);
		mpMusic = NULL;
	}
}

//! Play music
void 
cMixer::PlayMusic (string filename)
{
	if ( !mInitialized || !mMusicEnabled )
		return;
	
	FreeMusic ();

	// Load music from file
	string path = mDataDir + filename;
	mpMusic = Mix_LoadMUS (path.c_str ());
	if ( mpMusic == NULL )
	{
		dbgError () << "Unable to load music from file " << path
			<< ':' << Mix_GetError () << endl;
		return;
	}
	
	// Play music, loop forever
	Mix_PlayMusic (mpMusic, -1);
}

void 
cMixer::FadeMusic (void)
{
	if ( !mInitialized )
		return;
	
	if ( mpMusic != NULL )
	{
		if ( Mix_FadeOutMusic (1500) )
			SDL_Delay (1600);
		Mix_FreeMusic (mpMusic);
		mpMusic = NULL;
	}
}

void 
cMixer::SetMusicVolume (int volume)
{
	if ( !mInitialized || !mMusicEnabled )
		return;

	Mix_VolumeMusic (volume);	
}

int 
cMixer::GetMusicVolume (void)
{
	if ( !mInitialized || !mMusicEnabled )
		return -1;

	return Mix_VolumeMusic (-1);
}

void 
cMixer::SetSoundsVolume (int volume)
{
	if ( !mInitialized || !mSoundsEnabled )
		return;

	Mix_Volume (-1, volume);	
}

int 
cMixer::GetSoundsVolume (void)
{
	if ( !mInitialized || !mSoundsEnabled )
		return -1;

	return Mix_Volume (-1, -1);
}


//==============================================================================
// EOF
//==============================================================================
