// Camera.cpp: implementation of the CCamera class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Camera.h"
#include <gl/glu.h>
#include "system.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CCamera::CCamera()
{
	name[0] = '\0';
	position = CVector(0, 0, -10);
	target = CVector(0, 0, 0);
	up = CVector(0, 1, 0);
	fov = 45.0f;
	nearplane = 0.1f;
	farplane = 10000.0f;
	active = false;
	vp = CSystem::getDefaultViewport();
}

CCamera::~CCamera()
{

}

void CCamera::setName(char *n)
{
	int l = strlen(n);
	if (l + 1 > sizeof(name)) l = sizeof(name) - 1;
	memcpy(name, n, l);
	name[l] = '\0';
}

void CCamera::set(CVector &pos, CVector &target, CVector &up)
{
	position = pos;
	CCamera::target = target;
	CCamera::up = up;
	refresh();
}

void CCamera::set(CVector &pos, CVector &target)
{
	set(pos, target, up);
}

void CCamera::set(CVector &pos)
{
	set(pos, target, up);
}

void CCamera::setTarget(CVector &target)
{
	set(position, target);
}

void CCamera::setPosition(CVector &pos)
{
	set(pos);
}

void CCamera::setUp(CVector &up)
{
	set(position, target, up);
}

void CCamera::forward(float advance)
{
	CVector v = (target - position).normalize() * advance;
	position += v;
	target += v;
	refresh();
}

void CCamera::strafe(float rightinc)
{
	CVector v = ((target - position) % up).normalize() * rightinc;
	position += v;
	target += v;
	refresh();
}

void CCamera::fly(float upinc)
{
	CVector v = up.normalize() * upinc;
	position += v;
	target += v;
	refresh();
}
#include "quaternion.h"
void CCamera::refresh()
{
	if (vp == NULL) vp = CSystem::getDefaultViewport();
	if (active) {
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		float ratio = 1.0f;
		if (vp != NULL) ratio = (GLfloat)vp->getWidth() / (GLfloat)vp->getHeight();
		gluPerspective(fov, ratio, nearplane, farplane);
		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
		gluLookAt(position.x, position.y, position.z, target.x, target.y, target.z, up.x, up.y, up.z);
	}
}

void CCamera::rotate(CVector &v, float a)
{
	if (v != up) up = CVector(up + position).rotate(position, v, a) - position;
	if (v != target - position) target.rotate(position, v, a);
	refresh();
}

void CCamera::turn(float rightinc)
{
	target.rotate(position, up, -rightinc);
	refresh();
}

void CCamera::pitch(float upinc)
{
	CVector v = (target - position) % up;
	up = CVector(up + position).rotate(position, v, -upinc) - position;
	target.rotate(position, v, -upinc);
	refresh();
}

void CCamera::setViewport(CViewport *v)
{
	vp = v;
}

void CCamera::move(CVector &v)
{
	target += v - position;
	position = v;
}

void CCamera::rotate(CMatrix4 &m)
{
	up = CVector(0, 1, 0);
	up = CVector4(up) * m;
	target = CVector(0, 0, -1);
	target = CVector4(target) * m;
	target += position;
}

void CCamera::restore()
{
	position = old_position; up = old_up; target = old_target;
}

void CCamera::backup()
{
	old_position = position; old_up = up; old_target = target;
}

